# coding=utf-8

import json
import logging
import time
from sandbox import sdk2
from sandbox.common.telegram import TelegramBot
from sandbox.sandboxsdk import environments

import requests


class IncorrectlySyncedFeedId(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        ss_database_user_vault_key = sdk2.parameters.String('SS databse user vault key', required=True)
        ss_database_password_vault_key = sdk2.parameters.String('SS databse password vault key',
                                                                required=True)
        telegram_notification = sdk2.parameters.Bool("Send report to Telegram", default=False)
        with telegram_notification.value[True]:
            telegram_chat_id = sdk2.parameters.Integer("Telegram chat ID", required=True)
            telegram_bot_token_vault = sdk2.parameters.Vault("Vault secret contains Telegram bot token", required=True)

        solomon_api_url = sdk2.parameters.String('Solomon api url', default='http://solomon.yandex.net')
        project_id = sdk2.parameters.String('Project id')
        service_name = sdk2.parameters.String('Service name')
        cluster_name = sdk2.parameters.String('Cluster name')
        oauth_token_vault_key = sdk2.parameters.String('Oauth token vault key',
                                                       default='market_ss_solomon_oauth_token')
        sensor_label = sdk2.parameters.String('Sensor label')

        yql_token_name = sdk2.parameters.String(
            "YQL token name",
            default='YQL_TOKEN',
            required=True
        )

    class Requirements(sdk2.Requirements):
        disk_space = 1024 * 5
        environments = (environments.PipEnvironment('psycopg2-binary'),
                        environments.PipEnvironment('yandex-yt'),
                        environments.PipEnvironment('yql'))

    def ytFeedId(self):
        import yql.api.v1.client

        yql_token = sdk2.Vault.data(self.Parameters.yql_token_name)

        yql_query = "SELECT warehouse_id, supplier_id FROM hahn.`home/market/production/mstat/dictionaries/stock_sku/1h/latest` GROUP BY warehouse_id, supplier_id"
        with yql.api.v1.client.YqlClient(token=yql_token) as yql_client:
            logging.info(yql_query)
            query = yql_client.query(yql_query)
            query.run()

            if not query.get_results().is_success:
                raise common.errors.TaskFailure(
                    "YQL запрос не выполнен: " + '; '.join(str(error) for error in query.get_results().errors))

            result = []
            for table in query.get_results():
                table.fetch_full_data()

                for row in table.rows:
                    result.append(str(row[0]) + '_' + str(row[1]))

            return result

    def ssFeedId(self):
        conn = None
        cur = None
        try:
            logging.info('Fetching credentials')
            user = sdk2.Vault.data(self.Parameters.ss_database_user_vault_key)
            password = sdk2.Vault.data(self.Parameters.ss_database_password_vault_key)
            logging.info('Connecting to the SS database...')
            import psycopg2
            conn = psycopg2.connect(
                host="vla-ztlnd24dqcjd0m75.db.yandex.net",
                port="6432",
                database="market_stockdb",
                user=user,
                password=password)
            logging.info('Connected')

            cur = conn.cursor()
            logging.info('Getting unfreezed orders...')
            cur.execute("select warehouse_id, vendor_id from feed_id  ")

            raw_result = cur.fetchall()
            result = []
            for row in raw_result:
                result.append(str(row[0]) + '_' + str(row[1]))

            return result
        except Exception as error:
            logging.info(error)
        finally:
            if cur is not None:
                cur.close()
            if conn is not None:
                conn.close()
            logging.info('SS database connection closed.')

    def on_execute(self):
        ssFeedIds = self.ssFeedId()
        ytFeedId = self.ytFeedId()
        disjoint = list(set(ytFeedId) - set(ssFeedIds))
        logging.info("Fetched " + str(len(ssFeedIds)) + " records from SS")
        logging.info("Fetched " + str(len(ytFeedId)) + " records from YT")
        message = ''
        value = len(disjoint)
        if len(disjoint) > 0:
            message = message + 'Найдено несколько feed_id, которых нет в SS\n\n'
            message = message + "Не найдено в SS :" + str(len(disjoint)) + "\n"
            message = message + '\n Не найдены следующие пары warehouse_id и vendor_id: ' + \
                      ' warehouseId_vendorId ['
            for i in range(min(500, len(disjoint))):
                message = message + disjoint[i] + "; "

            message = message + '] не найдены в SS'
            logging.info("Next warehouse_id and vendor_id pairs were not found in SS: " + ','.join( disjoint))
        else:
            message = message + 'Не найдено расхождений в feed_id между SS и YT'

        logging.info(message)
        if self.Parameters.telegram_notification:
            try:
                bot = TelegramBot(bot_token)
                bot.send_message(self.Parameters.telegram_chat_id, message)
            except Exception as e:
                logging.warn("Telegram notification failed: {}".format(e.message))

        oauth_token = sdk2.Vault.data(self.Parameters.oauth_token_vault_key)
        pusher = Pusher(solomon_api_url=self.Parameters.solomon_api_url,
                        oauth_token=oauth_token,
                        project_id=self.Parameters.project_id,
                        service_name=self.Parameters.service_name,
                        cluster_name=self.Parameters.cluster_name
                        )

        pusher.push_single_sensor(sensor_label=self.Parameters.sensor_label, value=value)


class Pusher(object):
    def __init__(self,
                 solomon_api_url,
                 oauth_token,
                 project_id,
                 service_name,
                 cluster_name):
        self.solomon_api_url = solomon_api_url
        self.oauth_token = oauth_token
        self.project_id = project_id
        self.service_name = service_name
        self.cluster_name = cluster_name

    def push_single_sensor(self, sensor_label, value):
        logging.info('Pushing sensor "{}" with value {}'.format(sensor_label, value))
        logging.info('Project id "{}", service name "{}", cluster_name "{}"'.format(self.project_id, self.service_name,
                                                                                    self.cluster_name))
        body = self.__make_data(sensor_label, value)
        response = requests.post(
            self.solomon_api_url + '/api/v2/push?project={project}&service={service}&cluster={cluster}'.format(
                project=self.project_id, service=self.service_name, cluster=self.cluster_name
            ), headers=self.__make_headers(), data=body)

        if response.status_code != 200:
            logging.info('Solomon responded with {} {} {}'.format(response.status_code, response.reason,
                                                                  response.text))

    def __make_headers(self):
        return {
            'Authorization': 'OAuth ' + self.oauth_token,
            'Content-Type': 'application/json'
        }

    @staticmethod
    def __make_data(sensor_label, value):
        return json.dumps({
            "sensors": [{
                "labels": {"sensor": sensor_label, "period": "one_hour"},
                "ts": int(time.time()),
                "value": value
            }]
        })
