# coding=utf-8

import requests
import json
import logging

from datetime import datetime, timedelta

from sandbox import sdk2
from sandbox.common.telegram import TelegramBot
from sandbox.sandboxsdk import environments
import sandbox.common.types.misc as ctm


class MarschrouteOversalesChecker(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        ss_database_user_for_oversales_vault_key = sdk2.parameters.String('SS databse user vault key', required=True)
        ss_database_password_for_oversales_vault_key = sdk2.parameters.String('SS databse password vault key',
                                                                              required=True)
        tracker_database_user_for_oversales_vault_key = sdk2.parameters.String('Tracker databse user vault key',
                                                                               required=True)
        tracker_database_password_for_oversales_vault_key = sdk2.parameters.String('Tracker databse password vault key',
                                                                                   required=True)
        ffwf_database_user_for_oversales_vault_key = sdk2.parameters.String('FFWF databse user vault key',
                                                                            required=True)
        ffwf_database_password_for_oversales_vault_key = sdk2.parameters.String('FFWF databse password vault key',
                                                                                required=True)
        marschroute_api_key_for_oversales_vault_key = sdk2.parameters.String('Marschroute API key', required=True)
        telegram_bot_token_vault_key = sdk2.parameters.String('Telegram bot token key', required=True)
        telegram_chat_id = sdk2.parameters.String('Telegram chat id', required=True)
        unknown_issues_telegram_chat_id = sdk2.parameters.String('Unknown issues Telegram chat id', required=False)
        known_issues = sdk2.parameters.String('String in format (deliveryServiceId, ticket)'
                                              '[;(deliveryServiceId, ticker)]', default='')

    class Requirements(sdk2.Requirements):
        disk_space = 1024 * 5
        environments = (environments.PipEnvironment('psycopg2-binary'),)
        dns = ctm.DnsType.DNS64

    def ssFreezedOrders(self):
        conn = None
        cur = None
        try:
            user = sdk2.Vault.data(self.Parameters.ss_database_user_for_oversales_vault_key)
            password = sdk2.Vault.data(self.Parameters.ss_database_password_for_oversales_vault_key)
            logging.info('Connecting to the SS database...')
            import psycopg2
            conn = psycopg2.connect(
                host="vla-ztlnd24dqcjd0m75.db.yandex.net",
                port="6432",
                database="market_stockdb",
                user=user,
                password=password)
            logging.info('Connected')

            cur = conn.cursor()
            logging.info('Getting unfreezed orders...')
            cur.execute("select distinct sf.reason_id from stock_freeze sf "
                        "join stock s on sf.stock_id = s.id "
                        "join sku s2 on s.sku_id = s2.id "
                        "left join unfreeze_job uj on sf.id = uj.stock_freeze_id "
                        "where sf.deleted = false and sf.reason_type = 'ORDER' and s2.warehouse_id = 145 and uj.id is null")

            raw_result = cur.fetchall()
            result = []
            for row in raw_result:
                result.append(row[0])
            return result
        except Exception as error:
            logging.info(error)
        finally:
            if cur is not None:
                cur.close()
            if conn is not None:
                conn.close()
            logging.info('SS database connection closed.')

    def marschrouteReserved(self):
        conn = None
        cur = None
        try:
            apiKey = sdk2.Vault.data(self.Parameters.marschroute_api_key_for_oversales_vault_key)
            response = requests.get('https://api.marschroute.ru/' + apiKey + '/reserved')
            content = response.content
            mars_reserved_now = json.loads(content)
            last_orders = []
            for obj in mars_reserved_now['data']:
                last_orders.extend(obj['orders'])

            mars_orders_string = "'" + "','".join(set(last_orders)) + "'"

            user = sdk2.Vault.data(self.Parameters.tracker_database_user_for_oversales_vault_key)
            password = sdk2.Vault.data(self.Parameters.tracker_database_password_for_oversales_vault_key)
            logging.info('Connecting to the Tracker database...')
            import psycopg2
            conn = psycopg2.connect(
                host="sas-9niyk3t9k1m4cp6i.db.yandex.net",
                port="6432",
                database="market_delivery_tracker",
                user=user,
                password=password)
            logging.info('Connected')

            cur = conn.cursor()

            cur.execute("select distinct order_id, track_code from delivery_track "
                        "where track_code in (" + mars_orders_string + ") and delivery_service_id = 145 and entity_type = 0")
            raw_result = cur.fetchall()
            result = []
            existing_track_codes = []
            for row in raw_result:
                result.append(row[0])
                existing_track_codes.append(row[1])
            not_existing_track_codes = []
            existing_track_codes_set = set(existing_track_codes)
            for order in last_orders:
                if not existing_track_codes_set.__contains__(order):
                    not_existing_track_codes.append(order)
            return result, not_existing_track_codes
        except Exception as error:
            logging.info(error)
        finally:
            if cur is not None:
                cur.close()
            if conn is not None:
                conn.close()
            logging.info('Tracker database connection closed.')

    def splitFreezedInSsButNotReservedInMarschroute(self, freezedInSsButNotReservedInMarschroute):
        conn = None
        cur = None
        try:
            user = sdk2.Vault.data(self.Parameters.ss_database_user_for_oversales_vault_key)
            password = sdk2.Vault.data(self.Parameters.ss_database_password_for_oversales_vault_key)
            logging.info('Connecting to the SS database...')
            import psycopg2
            conn = psycopg2.connect(
                host="vla-ztlnd24dqcjd0m75.db.yandex.net",
                port="6432",
                database="market_stockdb",
                user=user,
                password=password)
            logging.info('Connected')

            orders_string = "'" + "','".join(set(freezedInSsButNotReservedInMarschroute)) + "'"
            cur = conn.cursor()
            logging.info('Getting new freezed')
            cur.execute("select distinct reason_id from stock_freeze where created > now() - interval '1 day' "
                        "and reason_id in (" + orders_string + ") and reason_type = 'ORDER'")

            raw_result = cur.fetchall()
            new_freezed = []
            for row in raw_result:
                new_freezed.append(row[0])

            cur.close()

            cur = conn.cursor()
            logging.info('Getting medium freezed')
            cur.execute(
                "select distinct reason_id from stock_freeze where created < now() - interval '1 day' and created > now() - interval '30 days' "
                "and reason_id in (" + orders_string + ") and reason_type = 'ORDER'")

            raw_result = cur.fetchall()
            medium_freezed = []
            for row in raw_result:
                medium_freezed.append(row[0])

            cur.close()

            cur = conn.cursor()
            logging.info('Getting old freezed')
            cur.execute("select distinct reason_id from stock_freeze where created < now() - interval '30 days' "
                        "and reason_id in (" + orders_string + ") and reason_type = 'ORDER'")

            raw_result = cur.fetchall()
            old_freezed = []
            for row in raw_result:
                old_freezed.append(row[0])
            return new_freezed, medium_freezed, old_freezed
        except Exception as error:
            logging.info(error)
        finally:
            if cur is not None:
                cur.close()
            if conn is not None:
                conn.close()
            logging.info('SS database connection closed.')

    def getTrackCodes(self, orders):
        conn = None
        cur = None
        try:
            orders_string = "'" + "', '".join(set(orders)) + "'"
            user = sdk2.Vault.data(self.Parameters.tracker_database_user_for_oversales_vault_key)
            password = sdk2.Vault.data(self.Parameters.tracker_database_password_for_oversales_vault_key)
            logging.info('Connecting to the Tracker database...')
            import psycopg2
            conn = psycopg2.connect(
                host="sas-9niyk3t9k1m4cp6i.db.yandex.net",
                port="6432",
                database="market_delivery_tracker",
                user=user,
                password=password)
            logging.info('Connected')

            cur = conn.cursor()

            cur.execute("select order_id, delivery_service_id from delivery_track "
                        "where order_id in (" + orders_string + ") and entity_type = 0")
            raw_result = cur.fetchall()
            result = {}

            for row in raw_result:
                order_id = row[0]
                delivery_service_id = row[1]
                if result.__contains__(order_id):
                    current = result[order_id]
                else:
                    current = {}
                current_tracks = []
                current[delivery_service_id] = current_tracks
                result[order_id] = current

            cur.close()

            cur = conn.cursor()

            cur.execute("select dt.order_id, dt.delivery_service_id, c.raw_status from delivery_track dt "
                        "join checkpoint c on dt.id = c.track_id "
                        "where dt.order_id in (" + orders_string + ") order by c.checkpoint_ts asc")
            raw_result = cur.fetchall()
            for row in raw_result:
                order_id = row[0]
                delivery_service_id = row[1]
                track = row[2]
                current = result[order_id]
                current_tracks = current[delivery_service_id]
                current_tracks.append(str(track))
                current[delivery_service_id] = current_tracks
                result[order_id] = current
            return result
        except Exception as error:
            logging.info(error)
        finally:
            if cur is not None:
                cur.close()
            if conn is not None:
                conn.close()
            logging.info('Tracker database connection closed.')

    def checkUnknownReserves(self, unknownReserves):
        conn = None
        cur = None
        try:
            unknown_reserves_string = "'" + "', '".join(set(unknownReserves)) + "'"
            user = sdk2.Vault.data(self.Parameters.ffwf_database_user_for_oversales_vault_key)
            password = sdk2.Vault.data(self.Parameters.ffwf_database_password_for_oversales_vault_key)
            logging.info('Connecting to the FFWF database...')
            import psycopg2
            conn = psycopg2.connect(
                host="market-ff-workflow01f.db.yandex.net",
                port="6432",
                database="market_ff_workflow",
                user=user,
                password=password)
            logging.info('Connected')

            cur = conn.cursor()

            cur.execute("select coalesce(parent_request_id, id), service_request_id from shop_request "
                        "where type in (1, 17) and service_id = 145 "
                        "and service_request_id in (" + unknown_reserves_string + ")")
            raw_result = cur.fetchall()

            all_found_service_ids = []
            all_found_ffwf_ids = []
            for row in raw_result:
                ffwf_id = row[0]
                service_request_id = row[1]
                all_found_service_ids.append(service_request_id)
                all_found_ffwf_ids.append(str(ffwf_id))

            cur.close()
            conn.close()

            all_found_ffwf_ids_set = set(all_found_ffwf_ids)

            all_found_ffwf_ids_string = "'" + "', '".join(all_found_ffwf_ids_set) + "'"

            user = sdk2.Vault.data(self.Parameters.ss_database_user_for_oversales_vault_key)
            password = sdk2.Vault.data(self.Parameters.ss_database_password_for_oversales_vault_key)
            logging.info('Connecting to the SS database...')
            import psycopg2
            conn = psycopg2.connect(
                host="vla-ztlnd24dqcjd0m75.db.yandex.net",
                port="6432",
                database="market_stockdb",
                user=user,
                password=password)
            logging.info('Connected')

            cur = conn.cursor()
            logging.info('Getting unfreezed outbounds...')
            cur.execute("select distinct sf.reason_id from stock_freeze sf "
                        "join unfreeze_job uj on sf.id = uj.stock_freeze_id "
                        "where sf.reason_type = 'OUTBOUND' "
                        "and sf.reason_id in ( " + all_found_ffwf_ids_string + ")")

            raw_result = cur.fetchall()
            finished_outbounds = []
            for row in raw_result:
                finished_outbounds.append(str(row[0]))

            cur.close()

            cur = conn.cursor()
            logging.info('Getting unfreezed outbounds...')
            cur.execute("select distinct sf.reason_id from stock_freeze sf "
                        "where sf.reason_type = 'OUTBOUND' "
                        "and sf.reason_id in ( " + all_found_ffwf_ids_string + ")")

            raw_result = cur.fetchall()
            exists_in_ss = []
            for row in raw_result:
                exists_in_ss.append(str(row[0]))

            exists_in_ss_set = set(exists_in_ss)

            for ffwf_id in all_found_ffwf_ids_set:
                if not exists_in_ss_set.__contains__(ffwf_id):
                    finished_outbounds.append(ffwf_id)

            all_found_set = set(all_found_service_ids)
            absolutely_unknown = []
            apiKey = sdk2.Vault.data(self.Parameters.marschroute_api_key_for_oversales_vault_key)
            for unknown in unknownReserves:
                if not all_found_set.__contains__(unknown):
                    response = requests.get('https://api.marschroute.ru/' + apiKey + '/order/' + unknown)
                    content = response.content
                    order_data = json.loads(content)
                    date_string = order_data['data']['order']['date_create']
                    parsed_date = datetime.strptime(date_string, '%d.%m.%Y %H:%M:%S')
                    now = datetime.now()
                    if parsed_date + timedelta(hours=4) < now:
                        absolutely_unknown.append(str(unknown))
            return finished_outbounds, absolutely_unknown
        except Exception as error:
            logging.info(error)
        finally:
            if cur is not None:
                cur.close()
            if conn is not None:
                conn.close()
            logging.info('Tracker database connection closed.')

    def getServiceToTicket(self, serviceToTicketString):
        if not serviceToTicketString:
            return {}
        result = {}
        parts = serviceToTicketString.split(';')
        for part in parts:
            trimmed_part = part[1:-1].strip()
            key_and_value = trimmed_part.split(',')
            trimmed_key = key_and_value[0].strip()
            trimmed_value = key_and_value[1].strip()
            result[trimmed_key] = trimmed_value
        return result

    def on_execute(self):
        ssFreezedOrders = set(self.ssFreezedOrders())
        logging.info("Fetched " + str(len(ssFreezedOrders)) + " records from SS")
        marschrouteReservedKnownAndUnknown = self.marschrouteReserved()
        marschrouteReserved = set(marschrouteReservedKnownAndUnknown[0])
        marschrouteReservedUnknown = set(marschrouteReservedKnownAndUnknown[1])
        logging.info("Fetched " + str(len(marschrouteReserved)) + " from Marschroute")
        freezedOnSsButNotReservedInMarschroute = []
        reservedInMarschrouteButNotFreezedInSs = []

        for ssFreezedOrder in ssFreezedOrders:
            if not marschrouteReserved.__contains__(ssFreezedOrder):
                freezedOnSsButNotReservedInMarschroute.append(ssFreezedOrder)

        for marschrouteReserve in marschrouteReserved:
            if not ssFreezedOrders.__contains__(marschrouteReserve):
                reservedInMarschrouteButNotFreezedInSs.append(marschrouteReserve)

        splitted = self.splitFreezedInSsButNotReservedInMarschroute(freezedOnSsButNotReservedInMarschroute)
        new_freezed = splitted[0]
        medium_freezed = splitted[1]
        old_freezed = splitted[2]

        logging.info(
            "Freezed in SS no more than 1 day ago but not reserved in Marschroute count = " + str(len((new_freezed))))
        logging.info("Freezed in SS no more than 1 day ago but not reserved in Marschroute: " + "'" + "', '".join(
            set(new_freezed)) + "'")

        logging.info("Freezed in SS from 1 day to 30 days ago but not reserved in Marschroute count = " + str(
            len((medium_freezed))))
        logging.info("Freezed in SS from 1 day to 30 days ago but not reserved in Marschroute: " + "'" + "', '".join(
            set(medium_freezed)) + "'")

        logging.info(
            "Freezed in SS more than 30 days ago but not reserved in Marschroute count = " + str(len((old_freezed))))
        logging.info("Freezed in SS more than 30 days ago but not reserved in Marschroute: " + "'" + "', '".join(
            set(old_freezed)) + "'")

        logging.info("Reserved in Marschroute but not freezed in SS count = " + str(
            len((reservedInMarschrouteButNotFreezedInSs))))
        logging.info("Reserved in Marschroute but not freezed in SS: " + "'" + "', '".join(
            set(reservedInMarschrouteButNotFreezedInSs)) + "'")

        messages = []

        if len(reservedInMarschrouteButNotFreezedInSs) > 0:
            serviceToTicket = self.getServiceToTicket(self.Parameters.known_issues)
            trackCodesByOrderAndService = self.getTrackCodes(reservedInMarschrouteButNotFreezedInSs)
            ordersCountByPartners = {}
            message = 'Найдено ' + str(len(reservedInMarschrouteButNotFreezedInSs)) + \
                      ' заказов, для которых нет фриза в SS, но есть резервы на Маршруте.\n\n'
            for orderId, trackCodesByService in trackCodesByOrderAndService.iteritems():
                order_message = ''
                order_message = order_message + 'Чекпоинты для заказа ' + orderId + ':\n'
                i = 1
                knownServiceIds = []
                for serviceId, trackCodes in trackCodesByService.iteritems():
                    if serviceToTicket.__contains__(str(serviceId)):
                        knownServiceIds.append(serviceId)
                    if ordersCountByPartners.__contains__(serviceId):
                        currentCount = ordersCountByPartners[serviceId]
                    else:
                        currentCount = 0
                    currentCount += 1
                    if serviceId != 145:
                        ordersCountByPartners[serviceId] = currentCount
                    order_message = order_message + str(i) + ') Партнер ' + str(serviceId) + ': ' + ', '.join(trackCodes) + '\n'
                    i += 1
                if len(knownServiceIds) > 0:
                    knownMessage = 'Возможно, проблема известная, есть тикеты для служб ('
                    j = 0
                    for knownServiceId in knownServiceIds:
                        if j > 0:
                            knownMessage = knownMessage + ', '
                        knownMessage = knownMessage + str(knownServiceId) + ' - ' + \
                                       serviceToTicket[str(knownServiceId)]
                        j += 1
                    order_message = order_message + knownMessage + ')\n'
                order_message = order_message + '\n'
                if len(message) + len(order_message) > 3800:
                    messages.append(message)
                    message = order_message
                else:
                    message += order_message
            partner_messages = 'Количество заказов для партнеров:\n'
            for partnerId, count in ordersCountByPartners.iteritems():
                partner_messages = partner_messages + str(partnerId) + ' - ' + str(count) + ' заказов'
                if serviceToTicket.__contains__(str(partnerId)):
                    knownTicket = serviceToTicket[str(partnerId)]
                    partner_messages = partner_messages + '; возможно, проблема известная, есть тикет для службы - ' + knownTicket
                partner_messages = partner_messages + '\n'
            if len(message) + len(partner_messages) > 3800:
                messages.append(message)
                message = partner_messages
            else:
                message += partner_messages
        else:
            message = 'Не найдено заказов, у которых нет фриза в SS, но есть резерв на Маршруте\n'
        unknown_telegram_chat_message = ''
        if len(marschrouteReservedUnknown) > 0:
            finishedOutboundsAndUnknown = self.checkUnknownReserves(marschrouteReservedUnknown)
            finishedOutbounds = set(finishedOutboundsAndUnknown[0])
            absolutelyUnknown = set(finishedOutboundsAndUnknown[1])
            outbounds_message = ''
            if len(finishedOutbounds) > 0:
                outbounds_message = outbounds_message + '\nТакже есть финализированные в FFWF изъятия, на которые висит резерв в ' \
                                    'Маршруте: ' + ", ".join(finishedOutbounds) + '\n'
            else:
                outbounds_message = outbounds_message + '\nПроблемных изъятий не обнаружено\n'
            if len(message) + len(outbounds_message) > 3800:
                messages.append(message)
                message = outbounds_message
            else:
                message += outbounds_message
            unknown_message = ''
            if len(absolutelyUnknown) > 0:
                unknown_message = unknown_message + '\nТакже есть резервы под что-то неизвестное ' \
                                    '(не найдено ни в Трекере, ни в FFWF): ' + ", ".join(absolutelyUnknown) + '\n'
                unknown_telegram_chat_message = unknown_telegram_chat_message + \
                                                '\nЕсть резервы под что-то неизвестное ' \
                                                '(не найдено ни в Трекере, ни в FFWF): ' + \
                                                ", ".join(absolutelyUnknown) + '\n'
            else:
                unknown_message = unknown_message + '\nЧего-то неизвестного не обнаружено\n'
                unknown_telegram_chat_message = unknown_telegram_chat_message + '\nНеизвестных резервов не обнаружено\n'
            if len(message) + len(unknown_message) > 3800:
                messages.append(message)
                message = unknown_message
            else:
                message += unknown_message
        else:
            nothing_message = '\nПроблемных изъятий и чего-то неизвестного не обнаружено\n'
            if len(message) + len(nothing_message) > 3800:
                messages.append(message)
                message = nothing_message
            else:
                message += nothing_message
            unknown_telegram_chat_message = unknown_telegram_chat_message + '\nНеизвестных резервов не обнаружено\n'
        messages.append(message)
        for mess in messages:
            bot = TelegramBot(bot_token=sdk2.Vault.data(self.Parameters.telegram_bot_token_vault_key))
            bot.send_message(self.Parameters.telegram_chat_id, mess)
        if self.Parameters.unknown_issues_telegram_chat_id is not None:
            bot = TelegramBot(bot_token=sdk2.Vault.data(self.Parameters.telegram_bot_token_vault_key))
            bot.send_message(self.Parameters.unknown_issues_telegram_chat_id, unknown_telegram_chat_message)
