# coding=utf-8

import logging
from sandbox import sdk2
from sandbox.common.telegram import TelegramBot
from sandbox.sandboxsdk import environments
from datetime import datetime, timedelta
import time


class NoMbiMappingTasksUnparking(sdk2.Task):
    UNPARKING_DELAY = 60  # in seconds

    DATE_IN_FUTURE = datetime(3000, 1, 1)
    DATE_IN_PAST = datetime.now() - timedelta(days=1)

    MAX_ATTEMPTS_NUMBER = 14
    MIN_ATTEMPTS_NUMBER = 1

    TASK_TYPE_PATTERN = "'CHANGE%'"
    TASK_FAIL_REASON_PATTERN = "'%Feed id mapping not found%'"

    STARTRACK_URL = 'https://st.yandex-team.ru/'

    class Parameters(sdk2.Task.Parameters):
        warehouses_for_monitoring = sdk2.parameters.String('Warehouses for monitoring: '
                                                           'String in format warehouseId[;warehouseId]', required=True)
        ss_database_user_vault_key = sdk2.parameters.String('SS database user vault key', required=True)
        ss_database_password_vault_key = sdk2.parameters.String('SS database password vault key', required=True)
        ss_database_hosts = sdk2.parameters.String('SS database hosts: '
                                                   'String in format host[;host]', required=True)
        ss_database_name = sdk2.parameters.String('SS database name', required=True)
        st_user_agent = sdk2.parameters.String('Startrack user agent', required=True)
        st_url = sdk2.parameters.String('Startrack url', required=True)
        st_token_vault_key = sdk2.parameters.String('Startrack token vault key', required=True)
        st_ticket = sdk2.parameters.String('Startrack ticket', required=True)
        st_summonees = sdk2.parameters.String('Summonees: '
                                              'String in format staff_login[,staff_login]', required=True)
        telegram_bot_token_vault_key = sdk2.parameters.String('Telegram bot token key', required=True)
        telegram_chat_id = sdk2.parameters.String('Telegram chat id', required=True)

    class Requirements(sdk2.Requirements):
        disk_space = 1024 * 5
        environments = (environments.PipEnvironment('psycopg2-binary'),
                        environments.PipEnvironment('yandex_tracker_client', version="1.3",
                                                    custom_parameters=["--upgrade-strategy only-if-needed"]),
                        environments.PipEnvironment('startrek_client', version="2.3.0",
                                                    custom_parameters=["--upgrade-strategy only-if-needed"]))

    def on_execute(self):
        import psycopg2
        from psycopg2.extras import RealDictCursor
        conn = None
        cursor = None
        try:
            user = sdk2.Vault.data(self.Parameters.ss_database_user_vault_key)
            password = sdk2.Vault.data(self.Parameters.ss_database_password_vault_key)
            hosts = self.Parameters.ss_database_hosts.split(";")
            logging.info('Connecting to SS database...')

            for host in hosts:
                conn = psycopg2.connect(host=host, port="6432",
                                        database=self.Parameters.ss_database_name,
                                        user=user, password=password, cursor_factory=RealDictCursor)
                cursor = conn.cursor()
                if self.is_master(cursor) is False:
                    logging.info('Connected to SS db on master')
                    sql = self.get_parked_tasks_sql()
                    cursor.execute(sql)
                    raw_result = cursor.fetchall()
                    self.process_results(raw_result, cursor, conn)
                    return
        except Exception as error:
            logging.error(error)
        finally:
            if cursor is not None:
                cursor.close()
            if conn is not None:
                conn.close()
            logging.info('SS database connection closed.')

    def process_results(self, raw_result, cursor, conn):
        ids_to_unpark = []
        fields_as_id_for_parking = []
        for row in raw_result:
            ids_to_unpark.append((row['id'],))
            fields_as_id_for_parking.append({'type': str(row['type']), 'uuid': row['uuid']})
        if len(ids_to_unpark) <= 0:
            logging.info('No parked tasks found.')
            return

        self.unpark_tasks(cursor, conn, ids_to_unpark)
        time.sleep(self.UNPARKING_DELAY)
        self.process_failed_tasks(fields_as_id_for_parking, cursor)

    def process_failed_tasks(self, fields_as_id_for_parking, cursor):
        rows = []
        for entry in fields_as_id_for_parking:
            sql = self.get_unparked_tasks_sql(entry['uuid'], entry['type'])
            cursor.execute(sql)
            rows.append(cursor.fetchall())
        if len(rows) <= 0:
            logging.info('All unparked tasks terminated successfully.')
            return

        problem_vendors_ids_by_warehouse = dict()
        ids_to_park = []
        for row in rows:
            payload_data = dict()
            payload = row[0]['payload']
            payload_data['id'] = row[0]['id']
            payload_data['vendorId'] = str(payload['vendorId'])
            warehouse_id = str(payload['warehouseId'])

            ids_to_park.append((payload_data['id'],))
            if not (warehouse_id in problem_vendors_ids_by_warehouse):
                problem_vendors_ids_by_warehouse[warehouse_id] = []
            info = next((item for item in problem_vendors_ids_by_warehouse[warehouse_id] if
                         item['vendorId'] == payload_data['vendorId']), None)
            if info is None:
                problem_vendors_ids_by_warehouse[str(payload['warehouseId'])].append(payload_data)

        self.update_ticket(problem_vendors_ids_by_warehouse)
        self.send_telegram_msg()

    def unpark_tasks(self, cursor, conn, ids):
        sql = self.get_updated_time_sql(self.DATE_IN_PAST, self.MIN_ATTEMPTS_NUMBER)
        cursor.executemany(sql, ids)
        conn.commit()
        logging.info('Unparked ' + str(len(ids)) + ' tasks')

    def get_parked_tasks_sql(self):
        return "SELECT id, type, uuid FROM execution_queue" + \
               " WHERE attempt_number >= " + str(self.MAX_ATTEMPTS_NUMBER) + \
               " AND execute_after >= '" + str(self.DATE_IN_FUTURE) + "'" + \
               " AND TYPE LIKE " + self.TASK_TYPE_PATTERN + \
               " AND (" + self.warehouse_ids_to_sql() + ")" + \
               " AND fail_reason LIKE " + self.TASK_FAIL_REASON_PATTERN + ";"

    def get_unparked_tasks_sql(self, uuid, task_type):
        return "SELECT id, payload FROM execution_queue" + \
               " WHERE attempt_number >= " + str(self.MAX_ATTEMPTS_NUMBER) + \
               " AND execute_after < '" + str(self.DATE_IN_FUTURE) + "'" + \
               " AND uuid = '" + uuid + "'" + \
               " AND type = '" + task_type + "'" + \
               " AND fail_reason LIKE " + self.TASK_FAIL_REASON_PATTERN + ";"

    def get_failed_tasks_sql(self):
        return "SELECT id FROM execution_queue" + \
               " WHERE attempt_number >= " + str(self.MAX_ATTEMPTS_NUMBER) + \
               " AND execute_after < '" + str(self.DATE_IN_FUTURE) + "'" + \
               " AND TYPE LIKE " + self.TASK_TYPE_PATTERN + \
               " AND (" + self.warehouse_ids_to_sql() + ")" + \
               " AND fail_reason LIKE " + self.TASK_FAIL_REASON_PATTERN + ";"

    def warehouse_ids_to_sql(self):
        sql_expression = ''
        warehouses_for_monitoring = self.Parameters.warehouses_for_monitoring.split(";")
        for warehouse_id in warehouses_for_monitoring:
            sql_expression += "uuid like '%-" + warehouse_id + "' OR "
        return sql_expression[:(len(sql_expression) - 4)]

    @staticmethod
    def get_updated_time_sql(date, attempt_number):
        return "UPDATE execution_queue" + \
               " SET execute_after = '" + str(date) + "', attempt_number = " + str(attempt_number) + \
               " WHERE id = %s"

    @staticmethod
    def is_master(cursor):
        cursor.execute("SELECT pg_is_in_recovery();")
        return cursor.fetchall()[0]['pg_is_in_recovery']

    def update_ticket(self, problem_vendors_ids_by_warehouse):
        from startrek_client import Startrek
        startrek_client = Startrek(useragent=self.Parameters.st_user_agent,
                                   base_url=self.Parameters.st_url,
                                   token=sdk2.Vault.data(self.Parameters.st_token_vault_key))
        logging.info('Updating Startrack ticket')
        issue = startrek_client.issues[self.Parameters.st_ticket]
        text = 'Необходимо включить связи для мерчей/складов:\n\n%%'
        for warehouse_id, infos in problem_vendors_ids_by_warehouse.items():
            for info in infos:
                text += 'ff-link create ' + info['vendorId'] + ' ' + warehouse_id + '\n'
        text += '%%'
        issue.comments.create(text=text, summonees=[self.Parameters.st_summonees])
        logging.info('Startrack ticket updated')

    def send_telegram_msg(self):
        logging.info('Sending telegram message')
        ticket_url = self.STARTRACK_URL + str(self.Parameters.st_ticket)
        telegram_message = 'Необходимо включить связи для мерча и склада: тикет ' + ticket_url
        bot = TelegramBot(bot_token=sdk2.Vault.data(self.Parameters.telegram_bot_token_vault_key))
        bot.send_message(self.Parameters.telegram_chat_id, telegram_message)
        logging.info('Telegram message was sent' + telegram_message)
