# coding=utf-8
import os

from sandbox.projects.market.ff_wf_api.endpointToStartrackPusher.common import generate_csv, split_and_strip, comma_list
from datetime import datetime

import logging


def create_st_client(st_user_agent, st_url, st_token):
    from startrek_client import Startrek
    return Startrek(useragent=st_user_agent, base_url=st_url, token=st_token)


def create_st_csv_attachment(client, mappings_for_csv):
    if mappings_for_csv is None:
        return []
    filename = generate_csv(mappings_for_csv) if len(mappings_for_csv) > 0 else None
    if filename:
        att = client.attachments.create(file=filename, params={'filename': os.path.basename(filename)})
        return [int(att.id)]
    return []


def format_exploding_comma_list(title, list):
    if not list:
        return ''
    elif len(list) < 6:
        return title + comma_list(list) + '\n'
    else:
        return '<{' + title + '\n' + comma_list(list) + '}>\n'


def create_or_update_ticket(st, queue, ticket_summary, st_params, entity_name, mappings_for_csv, comment):
    issues = st.issues.find('Queue: ' + queue +
                            ' Summary: #"' + ticket_summary + '"' +
                            ' "Sort By": Key ASC'
                            )
    if issues:
        main_issue = issues[0]
        close_duplicates(issues, main_issue, ticket_summary)

        issue = main_issue
        logging.info(datetime.now(), 'Found existing ticket', issue.key)

        issue.macroTag = datetime.strptime(issue.createdAt[:10], '%Y-%m-%d').strftime('%d.%m.%Y')
        issue = reopen_closed(issue, st)
        update_ticket_if_needed(st, issue, st_params, entity_name, mappings_for_csv)
        comment_if_needed(issue, comment)

        return None

    st_params['attachmentIds'] = create_st_csv_attachment(st, mappings_for_csv)
    st_params['macroTag'] = datetime.now().strftime('%d.%m.%Y')

    issue = st.issues.create(**st_params)
    logging.info(datetime.now(), 'Created new ticket:', issue.key)
    comment_if_needed(issue, comment)

    return issue


def comment_if_needed(issue, comment):
    if comment:
        new_comment = True
        for issue_comment in issue.comments:
            if issue_comment.text == comment:
                new_comment = False
        if new_comment:
            issue.comments.create(text=comment)


def update_ticket_if_needed(st, issue, st_params, entity_name, mappings_for_csv):
    if 'customerOrderNumber' in st_params:
        issue_entities = split_and_strip(issue.customerOrderNumber, ',')
        current_entities = split_and_strip(st_params['customerOrderNumber'], ',')
        missing_entities = set(issue_entities).difference(current_entities)
        new_orders = set(current_entities).difference(issue_entities)
        if missing_entities or new_orders:
            update_ticket(st, issue, st_params, mappings_for_csv,
                          format_exploding_comma_list(u'Удалены неактуальные ' + entity_name +
                                                      u'(' + str(len(missing_entities)) + u' шт): ',
                                                      missing_entities) +
                          format_exploding_comma_list(u'Добавлены новые ' + entity_name +
                                                      u'(' + str(len(new_orders)) + u' шт): ',
                                                      new_orders))
    else:
        update_ticket(st, issue, st_params, mappings_for_csv)


def reopen_closed(issue, st):
    if issue.status.key == 'closed':
        logging.info(str(datetime.now()) + ' Ticket ' + str(
            issue.key) + ' has been closed but the problem remains - reopening the ticket')
        reopen_ticket(issue, u'Тикет снова актуален')
        issue = st.issues[issue.key]
    return issue


def close_duplicates(issues, main_issue, ticket_summary):
    duplicating_issues = [i for i in issues if i.status.key != 'closed' and i != main_issue]
    if duplicating_issues:
        logging.info(str(datetime.now()) + ' Found duplicating tickets matching summary " ' +
                     str(ticket_summary) + ' ", main ticket: ' + str(main_issue.key) +
                     ', duplicating tickets:' + str([i.key for i in duplicating_issues]))
        for issue in duplicating_issues:
            transition = issue.transitions['close']
            transition.execute(comment=u'Тикет дублирует ' + main_issue.key,
                               resolution='duplicate')


def reopen_ticket(issue, comment):
    from startrek_client.exceptions import NotFound
    for tran_name in ('reopen', 'new'):
        try:
            transition = issue.transitions[tran_name]
        except NotFound:
            continue
        transition.execute(comment=comment)
        return
    raise Exception('Could not find appropriate transition to reopen ticket ' + issue.key +
                    ' Available transitions for this ticket: ' + str([t.id for t in issue.transitions]))


def update_ticket(st, issue, st_params, mappings_for_csv, comment_text=''):
    logging.info(str(datetime.now()) + ' Updating ticket ' + str(issue.key))

    for key in ('queue', 'description', 'summary'):
        st_params.pop(key)
    comment_text = \
        u'Информация в тикете была автоматически изменена.\n\n' + \
        comment_text
    issue.update(**st_params)
    issue.comments.create(
        text=comment_text,
        attachmentIds=create_st_csv_attachment(st, mappings_for_csv))


def create_or_update_tickets_from_endpoint_info(st_user_agent, st_url, st_token, queue_name, components,
                                                supplier_to_data, entity_name):
    logging.info(str(datetime.now()) + " Creating ST tickets for " + str(len(supplier_to_data)) + " rows")

    st = create_st_client(st_user_agent, st_url, st_token)

    st_queues = set()
    st_queues.add(queue_name)
    actual_entities = set()
    components_list = split_and_strip(components, ',')
    for (supplier_id, data_for_supplier) in supplier_to_data.iteritems():
        ticket_summary = data_for_supplier['summary']
        mappings_for_csv = None
        if 'mappings' in data_for_supplier:
            mappings_for_csv = data_for_supplier['mappings']

        st_params = {'queue': queue_name, 'summary': ticket_summary, 'customerOrderNumber': '',
                     'components': components_list}
        if 'description' in data_for_supplier:
            st_params['description'] = data_for_supplier['description']
        if 'tags' in data_for_supplier:
            st_params['tags'] = [tag.replace(' ', '_') for tag in data_for_supplier['tags']]
        if 'entities' in data_for_supplier is not None:
            entities = data_for_supplier['entities']
            st_params['customerOrderNumber'] = comma_list(entities)
            for entity in entities:
                actual_entities.add(entity)

        comment = None
        if 'comment' in data_for_supplier:
            comment = data_for_supplier['comment']
        try:
            create_or_update_ticket(st, queue_name, ticket_summary, st_params, entity_name, mappings_for_csv, comment)
        except Exception as e:
            logging.error("Got exception " + str(e) + " for supplier " + str(supplier_id))

    close_non_actual_tickets(st, st_queues, actual_entities, components_list)

    logging.info(str(datetime.now()) + " Finished creating tickets")


def close_non_actual_tickets(st, st_queues, actual_entities, components_list):
    components_list.sort()
    if not st_queues:
        logging.info(
            str(datetime.now()) + ' WARNING: Found no problem orders - cannot figure out ST-queue and clean i - ',
            'you can clean it by yourself - just close all open tickets :)')
        return
    if len(st_queues) > 1:
        logging.info(
            str(datetime.now()) + ' WARNING: Found more than one ST-queue in sql results - cannot clean ST-queue!')
        return
    st_queue = st_queues.pop()
    logging.info(str(datetime.now()) + ' Cleaning ST queue ' + str(st_queue))

    issues = st.issues.find('Queue: ' + st_queue + ' Status: !Closed Components: ' + comma_list(components_list))

    logging.info(str(datetime.now()) + ' Found ' + str(len(actual_entities)) + ' actual problem entities: ' + str(
        actual_entities))
    logging.info(str(datetime.now()) + ' Found ' + str(len(issues)) + ' open issues matching criteria in ' + st_queue)

    for issue in issues:
        issue_key = issue.key
        issue_components = [component.name for component in issue.components]
        logging.info(str(datetime.now()) + ' Checking ticket ' + str(issue_key))

        issue_components.sort()
        if issue_components != components_list:
            continue
        issue_entities = split_and_strip(issue.customerOrderNumber, ',')
        actual_issue_entities = [o for o in issue_entities if o in actual_entities]

        logging.info(str(datetime.now()) + ' Ticket ' + str(issue_key) + ' has ' + str(len(actual_issue_entities)) +
                     ' actual problem entities: ' + str(comma_list(actual_issue_entities)))

        if not actual_issue_entities:
            issue.update(customerOrderNumber='')
            logging.info(str(datetime.now()) + ' Closing ticket ' + str(issue_key))
            transition = issue.transitions['close']
            transition.execute(comment='Проблема более не актуальна', resolution="won'tFix")
