# -*- coding: utf-8 -*-
import os
import sys
import logging
import datetime

from sandbox import sdk2
from sandbox import sandboxsdk
import sandbox.common.types.task as ctt
import sandbox.common.types.resource as ctr
from sandbox.common.telegram import TelegramBot
from .. import MarketForecasterBuyboxApp


class RunMarketForecasterBuybox(sdk2.Task):
    """Make buybox forecast (forecast of shows in buybox) for all MSKU and suppliers and upload it to YT
    """

    class Parameters(sdk2.Task.Parameters):
        oauth_token = sdk2.parameters.Vault("Vault secret contains OAuth token; should provide access to YT, YQL, STEP", required=True)

        use_latest_version = sdk2.parameters.Bool("Use latest version of forecaster-buybox", default=True)
        with use_latest_version.value[False]:
            forecaster_app_resource = sdk2.parameters.Resource(
                "Resource with forecaster-buybox application",
                resource_type=MarketForecasterBuyboxApp,
                state=ctr.State.READY,
                required=True
            )

        yt_dest_directory = sdk2.parameters.String("YT directory where forecast table will be created", required=True)

        telegram_notification = sdk2.parameters.Bool("Send report to Telegram", default=False)
        with telegram_notification.value[True]:
            telegram_chat_id = sdk2.parameters.Integer("Telegram chat ID", required=True)
            telegram_bot_token_vault = sdk2.parameters.Vault("Vault secret contains Telegram bot token", required=True)

    class Requirements(sdk2.Task.Requirements):
        environments = (
            sandboxsdk.environments.PipEnvironment('yandex-yt'),
            sandboxsdk.environments.PipEnvironment('yandex-yt-yson-bindings-skynet'),
            sandboxsdk.environments.PipEnvironment('yql')
        )

    def _notify(self, status):
        if not self.Parameters.telegram_notification:
            return

        message = "Sandbox forecaster task #{} has finished: {}".format(
            self.id, "SUCCESS" if status == ctt.Status.SUCCESS else "FAILED"
        )
        try:
            bot = TelegramBot(bot_token=self.Parameters.telegram_bot_token_vault.data())
            bot.send_message(self.Parameters.telegram_chat_id, message)
        except Exception as e:
            logging.warn("Telegram notification failed: {}".format(e.message))

    def on_finish(self, prev_status, status):
        self._notify(status)

    def on_break(self, prev_status, status):
        self._notify(status)

    def on_execute(self):
        # get package content
        root_dir = self._get_app()

        # run forecaster-buybox
        date = datetime.datetime.today().strftime("%Y-%m-%d")
        yt_dest_path = "{}/{}".format(self.Parameters.yt_dest_directory, date)
        args = [
            sys.executable, os.path.join(root_dir, "scripts/run_forecaster_buybox.py"),
            "--root-dir", root_dir,
            "--start-date", date,
            "--yt-dest-path", yt_dest_path
        ]

        env = os.environ.copy()
        env.update({
            "YT_TOKEN": self.Parameters.oauth_token.data(),
            "YQL_TOKEN": self.Parameters.oauth_token.data()
        })

        with sdk2.helpers.ProcessLog(self, logger="run_forecaster_buybox") as pl:
            sdk2.helpers.subprocess.check_call(args, env=env, stdout=pl.stdout, stderr=pl.stderr, close_fds=True)

    def _get_app(self):
        if self.Parameters.use_latest_version:
            rsc = MarketForecasterBuyboxApp.find(state=ctr.State.READY).order(-sdk2.Resource.id).first()
            logging.info("Found {} resource #{} that should contain forecaster-buybox application"
                         .format(rsc.type, rsc.id))
        else:
            rsc = self.Parameters.forecaster_app_resource
            logging.info("Taken {} resource #{} that should contain forecaster-buybox application"
                         .format(rsc.type, rsc.id))

        # unpack package
        package_path = str(sdk2.ResourceData(rsc).path)
        sdk2.helpers.subprocess.Popen("tar -xf {}".format(package_path), shell=True).wait()
        return os.getcwd()
