# -*- coding: utf-8 -*-
import os
import sys
import logging
import datetime

from sandbox import sdk2
from sandbox import sandboxsdk
import sandbox.common.types.task as ctt
import sandbox.common.types.resource as ctr
from sandbox.projects.market.sre.RunMarketDataGetter import RunMarketDataGetter
from sandbox.common.telegram import TelegramBot

from .. import MarketDemandForecasterCatboostModel


_necessary_resources = [
    "yt_money:demand-prediction-sales-and-variance.mmap",
    "yt_money:demand-prediction-one-p-input.mmap",
    "forecaster:region_weights.tsv"
]


class MarketForecasterOfflineData(sdk2.resource.AbstractResource):
    """Resource bundle for offline Market Forecaster
    """
    any_arch = True


class MarketForecasterOfflineApp(sdk2.Resource):
    """Offline Market Forecaster application - intended to create big daily forecasts for all MSKU
    """
    executable = True


class MarketForecasterOfflineForecast(sdk2.Resource):
    """Forecast produced by offline Market Forecaster
    """
    any_arch = True


class RunMarketForecasterOffline(sdk2.Task):
    """Make forecast for all MSKU and optionally upload it to YT
    """

    class Parameters(sdk2.Task.Parameters):
        oauth_token = sdk2.parameters.Vault("Vault secret contains OAuth token; should provide access to YT, YQL, STEP", required=True)

        use_latest_version = sdk2.parameters.Bool("Use latest version of forecaster-offline", default=True)
        with use_latest_version.value[False]:
            forecaster_app_resource = sdk2.parameters.Resource(
                "Resource with forecaster-offline application",
                resource_type=MarketForecasterOfflineApp,
                state=ctr.State.READY,
                required=True
            )

        create_data_resource = sdk2.parameters.Bool("Create up-to-date resource bundle (via data-getter)", default=True)
        with create_data_resource.value[False]:
            data_resource = sdk2.parameters.Resource(
                "Resource with data bundle for forecaster-offline",
                resource_type=MarketForecasterOfflineData,
                state=ctr.State.READY,
                required=True
            )

        yt_dest_directory = sdk2.parameters.String("YT directory where forecast table will be created", required=True)
        experimental_yt_dest_directory = sdk2.parameters.String("YT directory where experimental forecast table will be created", required=False)

        telegram_notification = sdk2.parameters.Bool("Send report to Telegram", default=False)
        with telegram_notification.value[True]:
            telegram_chat_id = sdk2.parameters.Integer("Telegram chat ID", required=True)
            telegram_bot_token_vault = sdk2.parameters.Vault("Vault secret contains Telegram bot token", required=True)

    class Requirements(sdk2.Task.Requirements):
        environments = (
            sandboxsdk.environments.PipEnvironment('yandex-yt'),
            sandboxsdk.environments.PipEnvironment('yandex-yt-yson-bindings-skynet'),
            sandboxsdk.environments.PipEnvironment('yql')
        )

    def _notify(self, status):
        if not self.Parameters.telegram_notification:
            return

        message = "Sandbox forecaster task #{} has finished: {}".format(
            self.id, "SUCCESS" if status == ctt.Status.SUCCESS else "FAILED"
        )
        try:
            bot = TelegramBot(bot_token=self.Parameters.telegram_bot_token_vault.data())
            bot.send_message(self.Parameters.telegram_chat_id, message)
        except Exception as e:
            logging.warn("Telegram notification failed: {}".format(e.message))

    def on_finish(self, prev_status, status):
        self._notify(status)

    def on_break(self, prev_status, status):
        self._notify(status)

    def on_execute(self):
        if self.Parameters.create_data_resource:
            # run data-getter to produce necessary data resources
            with self.memoize_stage.preparing_data(commit_on_entrance=False):
                raise sdk2.WaitTask(
                    tasks=self._run_data_getter_task(),
                    statuses=[ctt.Status.Group.FINISH, ctt.Status.Group.BREAK],
                    wait_all=True,
                    timeout=3600  # 1 hour
                )

        # get data and application
        data_path = self._get_forecaster_data()
        app_path = self._get_forecaster_app()
        script_path = "./scripts"
        logging.info((
            "\nData path: {}"
            "\nApplication path: {}"
            "\nScripts path: {}"
        ).format(data_path, app_path, script_path))

        # run forecaster
        date = datetime.datetime.today().strftime("%Y-%m-%d")
        yt_dest_path = "{}/{}".format(self.Parameters.yt_dest_directory, date)
        args = [
            sys.executable, os.path.join(script_path, "make_total_forecast.py"),
            "--app-path", app_path,
            "--data-path", data_path
        ]

        env = os.environ.copy()
        env.update({
            "YT_TOKEN": self.Parameters.oauth_token.data(),
            "YQL_TOKEN": self.Parameters.oauth_token.data()
        })

        with sdk2.helpers.ProcessLog(self, logger="make_total_forecast") as pl:
            sdk2.helpers.subprocess.check_call(
                args + ["--yt-dest-path", yt_dest_path], env=env, stdout=pl.stdout, stderr=pl.stderr, close_fds=True
            )

        if self.Parameters.experimental_yt_dest_directory:
            ml_model_path, ml_model_date = find_ml_model()
            yt_dest_path = "{}/{}".format(self.Parameters.experimental_yt_dest_directory, date)
            args += [
                "--ml-path", ml_model_path,
                "--ml-date", ml_model_date
            ]
            with sdk2.helpers.ProcessLog(self, logger="make_total_forecast-v3") as pl:
                sdk2.helpers.subprocess.check_call(
                    args + ["--yt-dest-path", yt_dest_path], env=env, stdout=pl.stdout, stderr=pl.stderr, close_fds=True
                )

        # if we are here then no exception was thrown and forecast is done
        self._send_step_event(path=yt_dest_path)

    def _get_forecaster_data(self):
        if self.Parameters.create_data_resource:
            getter_task = sdk2.Task.find(RunMarketDataGetter, parent=self).order(-sdk2.Task.id).first()
            rsc = MarketForecasterOfflineData.find(task_id=getter_task.id).order(-sdk2.Resource.id).first()
            logging.info("Found {} resource #{} that should contain forecaster-offline data bundle".format(rsc.type, rsc.id))
        else:
            rsc = self.Parameters.data_resource
            logging.info("Taken {} resource #{} that should contain forecaster-offline data bundle".format(rsc.type, rsc.id))
        return str(sdk2.ResourceData(rsc).path)

    def _get_forecaster_app(self):
        if self.Parameters.use_latest_version:
            rsc = MarketForecasterOfflineApp.find(state=ctr.State.READY).order(-sdk2.Resource.id).first()
            logging.info("Found {} resource #{} that should contain forecaster-offline application"
                         .format(rsc.type, rsc.id))
        else:
            rsc = self.Parameters.forecaster_app_resource
            logging.info("Taken {} resource #{} that should contain forecaster-offline application"
                         .format(rsc.type, rsc.id))

        # unpack package
        package_path = str(sdk2.ResourceData(rsc).path)
        sdk2.helpers.subprocess.Popen("tar -xf {}".format(package_path), shell=True).wait()

        app_path = "./forecaster-offline"  # should be in the package

        with sdk2.helpers.ProcessLog(self, logger="forecaster-version") as pl:
            sdk2.helpers.subprocess.check_call(
                [app_path, "-V"], stdout=pl.stdout, stderr=pl.stderr, close_fds=True
            )
        return app_path

    def _run_data_getter_task(self):
        """ Launch data-getter as a child task to produce needed data
        """

        task = RunMarketDataGetter(
            parent=self,
            description="Generate data for {} (task #{})".format(self.type, self.id),
            target_service="forecaster-standalone",
            resource_type_name=str(MarketForecasterOfflineData),
            resources="\n".join(_necessary_resources),
            environment_type="stable",
            create_nanny_release=False,
            yt_token_vault=self.Parameters.oauth_token
        ).enqueue()
        logging.info("Create child {} task #{} to produce necessary data resources...".format(task.type, task.id))
        return task

    def _send_step_event(self, path):
        import requests

        logging.info("Create STEP event 'cluster_table_publish'...")
        resp = requests.post(
            url="https://step.sandbox.yandex-team.ru/api/v1/events",
            headers={"Authorization": "OAuth {}".format(self.Parameters.oauth_token.data())},
            json={"events": [{
                "name": "cluster_table_publish",
                "params": {
                    "cluster": "hahn",
                    "path": path
                }
            }]}
        )
        logging.info("STEP response: {}".format(resp.text))


def find_ml_model():
    rsc = MarketDemandForecasterCatboostModel.find(state=ctr.State.READY).order(-sdk2.Task.id).first()
    logging.info(
        "Found {} resource #{} that should contain ML model for forecaster".format(rsc.type, rsc.id))
    return str(sdk2.ResourceData(rsc).path), rsc.created.strftime("%Y-%m-%d")
