# coding=utf-8

import tempfile
import os
import os.path
import pipes
import shutil
import time
import re

import sandbox.common.types.misc as ctm
import sandbox.sdk2.path as spath
from sandbox import sdk2
from sandbox.projects.common.arcadia import sdk
from sandbox.projects.common.build.parameters import ArcadiaUrl, BASE_BUILD_GROUP_NAME, RESOURCES_GROUP_NAME
from sandbox.projects.market.resources import FRONT_BUILD_BASE, MARKET_FRONT_BUILD_TAR
from sandbox.projects.market.front.MarketFrontNodejsBuildBase import RESOURCE_TYPES
from sandbox.projects.market.front.helpers.sandbox_helpers import rich_check_call
from sandbox.projects.market.front.helpers.node import create_node_selector
from sandbox.projects.market.front.helpers.ubuntu import create_ubuntu_selector, setup_container
from sandbox.projects.market.front.helpers.MetatronEnv import MetatronEnv
from sandbox.projects.sandbox_ci import SANDBOX_CI_ARTIFACT


class FrontBuildBase(sdk2.Task):
    """
    Task to build front
    """

    class Requirements(sdk2.Task.Requirements):
        dns = ctm.DnsType.DNS64

    class Parameters(sdk2.Task.Parameters):
        checkout_arcadia_from_url = ArcadiaUrl()

        with sdk2.parameters.Group(BASE_BUILD_GROUP_NAME) as misc:
            ubuntu_version = create_ubuntu_selector()
            node_version = create_node_selector()
            npm_commands = sdk2.parameters.List(
                'NPM команды',
                sdk2.parameters.String
            )
            work_dir = sdk2.parameters.String("Path to build", required=True)
            dist_dir = sdk2.parameters.String("Папка дистрибутива сборки", required=True, default="dist")
            build_debian = sdk2.parameters.Bool("Собрать дебиан пакет", required=True, default=False)
            build_tarball = sdk2.parameters.Bool("Собрать tarball пакет", required=True, default=False)
            with build_tarball.value[True]:
                tarball_name = sdk2.parameters.String("Имя архива", required=True, default="front.tar.gz")
                tarball_ui_directory_path = sdk2.parameters.String("Путь до директории UI", required=True, default="ui/front")

            auto_versioning = sdk2.parameters.Bool("Сгенерировать версию", required=True, default=True)
            with auto_versioning.value[False]:
                version = sdk2.parameters.String("Релиз версия", required=True, default="1.0.0")
            product_env = sdk2.parameters.Bool("Продуктовая сборка", required=True, default=True)
            html_page_resources = sdk2.parameters.List(
                'Пути до HTML страниц',
                sdk2.parameters.String
            )

    class Context(sdk2.Context):
        pass

    def on_enqueue(self):
        super(FrontBuildBase, self).on_enqueue()
        setup_container(self)

    @property
    def resource_type(self):
        return FRONT_BUILD_BASE

    @property
    def version(self):
        return '1.0.{}'.format(time.time()) if self.Parameters.auto_versioning else self.Parameters.version

    @property
    def dist_dir(self):
        """
        Папка дистрибутива приложения, относительно корня репозитория
        """
        return self.Parameters.dist_dir if self.Parameters.dist_dir else None

    @property
    def make_env(self):
        make_env = os.environ.copy()
        make_env["YENV"] = "production" if self.Parameters.product_env else "development"
        make_env["NODE_ENV"] = "production" if self.Parameters.product_env else "development"

        return make_env
    
    def _get_arcadia(self):
        return sdk.mount_arc_path(self.Parameters.checkout_arcadia_from_url)

    def _install_packages(self):
        rich_check_call(
            ["npm", "install"],
            task=self, alias="npm_install", cwd=self.Context.app_src_path, env=self.make_env
        )

    def _app_run_npm_command(self, command):
        rich_check_call(
            ["npm", "run"] + command.split(" "),
            task=self, alias="npm_run_command", cwd=self.Context.app_src_path, env=self.make_env
        )

    def _build_debian(self, version):
        rich_check_call(
            ["./build.sh", version],
            task=self, alias="debian_build", cwd=os.path.join(self.Context.app_src_path, "conductor/src"), env=self.make_env
        )
    
    def _build_tarball(self, version):
        shutil.copytree(os.path.join(self.Context.app_src_path, self.dist_dir), os.path.join(self.Context.app_src_path, self.Parameters.tarball_ui_directory_path), symlinks=True)
        rich_check_call(
            ["tar", "cvf" , self.Parameters.tarball_name, re.search(r'^/?(\w+)/', self.Parameters.tarball_ui_directory_path).group(1)],
            task=self, alias="build_tarball", cwd=self.Context.app_src_path, env=self.make_env
        )
        shutil.copy(os.path.join(self.Context.app_src_path, self.Parameters.tarball_name), '.')
        sdk2.ResourceData(MARKET_FRONT_BUILD_TAR(
            self, "Tar archive", self.Parameters.tarball_name,
            version=version
        )).ready()

    def _create_html_resource(self, dir_path, description):
        resource_path = os.path.join(self.Context.app_src_path, dir_path)
        if os.path.exists(resource_path):
            shutil.copytree(resource_path, dir_path, symlinks=True)
            resource = SANDBOX_CI_ARTIFACT(
                task=self,
                description=description,
                path=dir_path,
            )
            sdk2.ResourceData(resource).ready()

    def _create_resource(self, version):
        resource_type = RESOURCE_TYPES[self.resource_type]
        shutil.copytree(os.path.join(self.Context.app_src_path, self.dist_dir), self.dist_dir, symlinks=True)
        sdk2.ResourceData(resource_type(
            self, "Resource", self.dist_dir,
            version=version
        )).ready()

    def on_execute(self):
        with self._get_arcadia() as arcadia_src_dir:
            self.Context.app_src_path = os.path.join(arcadia_src_dir, self.Parameters.work_dir)
            with MetatronEnv(self, nodejs_version=self.Parameters.node_version):

                version = self.version
                # self._install_packages()
                for command in self.Parameters.npm_commands:
                    self._app_run_npm_command(command)
                if self.Parameters.build_debian:
                    self._build_debian(version)
                    self._create_resource(version)
                if self.Parameters.build_tarball:
                    self._build_tarball(version)

    def on_finish(self, prev_status, status):
        with MetatronEnv(self, nodejs_version=self.Parameters.node_version):
            for path in self.Parameters.html_page_resources:
                self._create_html_resource(path, path)
