# -*- coding: utf-8 -*-

import logging
import os
import pipes
import tempfile
import uuid

import sandbox.common.types.misc as ctm
import sandbox.common.types.resource as ctr
import sandbox.common.types.task as ctt
import sandbox.sdk2.path as spath
from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.utils import get_task_link
from sandbox.projects.common import task_env
from sandbox.projects.market.front.MarketAutotestsGemini import MarketAutotestsGemini
from sandbox.projects.market.front.MarketAutotestsGeminiUpdate import MarketAutotestsGeminiUpdate
from sandbox.projects.market.front.MarketAutotestsHermione import MarketAutotestsHermione
from sandbox.projects.market.front.helpers.MetatronEnv import MetatronEnv
from sandbox.projects.market.front.helpers.github import clone_repo, clone_repo_for_merge_commit, \
    find_opened_release_branch
from sandbox.projects.market.front.helpers.metrics import create_metrics_params, select_metrics_parameters
from sandbox.projects.market.front.helpers.node import NODE_DEFAULT
from sandbox.projects.market.front.helpers.node import create_node_selector
from sandbox.projects.market.front.helpers.sandbox_helpers import rich_check_call, \
    report_data, format_header
from sandbox.projects.market.front.helpers.solomon import make_solomon_parameters, select_solomon_parameters
from sandbox.projects.market.front.helpers.startrack import FixVersion, ReleaseTicket
from sandbox.projects.market.front.helpers.ubuntu import create_ubuntu_selector, setup_container
from sandbox.projects.market.resources import MARKET_APP, MARKET_AUTOTEST_REPORT, MARKET_BROKEN_AUTOTEST_REPORT, \
    MARKET_SCREENSHOTS_PACK
from sandbox.projects.sandbox_ci.managers import ConfigManager, MetaTaskManager
from sandbox.projects.sandbox_ci.utils import env

KILL_TIMEOUT = 3 * 60 * 60  # 3 h
SUBTASK_TIMEOUT = 3 * 60 * 60  # 3 h
DISK_SPACE = 3 * 1024  # 3 Gb
GH_OWNER = 'MARKET'

SCREENSHOTS_DIR = 'hermione-screenshots'
SCREENSHOTS_ARCHIVE_NAME = "screenshots-pack.tar.gz"


class MarketAutotests(sdk2.Task):
    """
    Австотесты сервисов Маркета
    """

    APP_SRC_PATH = ''
    APP_REPO_PATH = ''
    root_dir = ''
    failure_links = None
    app_branch_final = None

    class Context(sdk2.Context):
        resources_to_report = []
        child_tasks_ids = []

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = KILL_TIMEOUT
        ubuntu_version = create_ubuntu_selector()
        node_version = create_node_selector()

        iteration_rerun_broken_tests = sdk2.parameters.Bool(
            'Включить итеративный перепрогон упавших hermione-тестов',
            description='Для включения этой опции в проекте, необходимо использовать hermione-broken-tests@1.2.0',
            default=False
        )

        broken_tests_resource_id = sdk2.parameters.Integer(
            'Id ресурса со списком поломанных тестов',
            required=False
        )

        testpalm_ids_resource_id = sdk2.parameters.Integer(
            'Id ресурса со списком testpalm id',
            required=False
        )

        try_to_find_last_screen_pack = sdk2.parameters.Bool(
            'Поискать готовые эталоны для скринтестов',
            description='Попытается найти последнюю таску того же типа с веткой master и возьмёт эталоны оттуда',
            default=False,
        )

        rebuild_app = sdk2.parameters.Bool(
            'Пересобрать приложение перед запуском.',
            description='Для пересоборки даже если передан ресурс с приложением в app_resource_id',
            default=True
        )
        with rebuild_app.value[False]:
            app_resource_id = sdk2.parameters.Integer(
                'Id ресурса с приложением',
                required=True
            )

        with sdk2.parameters.Group('GitHub репозиторий проекта') as github_repo_block:
            app_owner = sdk2.parameters.String(
                'GitHub owner',
                description='Логин владельца репозитория или название организации',
                default=GH_OWNER,
                required=True
            )
            app_repo = sdk2.parameters.String(
                "Репозиторий",
                default='market',
                required=True
            )
            app_branch = sdk2.parameters.String(
                "Ветка",
                default='master',
                required=True
            )
            app_src_dir = sdk2.parameters.String(
                "Кастомный путь корня приложения внутри репозитория"
            )
            use_release_branch = sdk2.parameters.Bool(
                'Использовать текущую релизную ветку',
                description='Отметьте, если нужно поискать ветку с текущим релизом, и прогнать АТ из неё '
                            '(если не будет найдена, прогонится из ветки, указанной в app_branch)',
                default=False
            )
            with use_release_branch.value[True]:
                app_release_branch_starts_with = sdk2.parameters.String(
                    'Начало названия релизной ветки для поиска её на гитхабе',
                    default='release/',
                    required=True
                )
            app_merge_commit = sdk2.parameters.String(
                "Merge-коммит",
                default='',
                required=False
            )
            app_status_check_commit = sdk2.parameters.String(
                "Коммит для отправки статус-чеков",
                default='',
                required=False
            )

        with sdk2.parameters.Group('Project build') as project_build_block:
            project_build_context = sdk2.parameters.String(
                'Профиль конфигурации в Genisys',
                default=''
            )
            external_config = sdk2.parameters.String(
                'Внешний конфиг',
                default=''
            )

            # текущий пайплайн передает это поле при запуске, перейти на dist_dir по возможности
            # todo: ???
            platform = sdk2.parameters.String('Платформа для сборки')
            dist_dir = sdk2.parameters.String('Путь дистрибутива сборки')

        with sdk2.parameters.Group('Тестовое окружение') as test_target:
            base_url = sdk2.parameters.String(
                'Тестируемый хост',
                description='Адрес машины, на который будут ходить тесты'
            )

        with sdk2.parameters.Group('Отчет') as report_section:
            report_type = sdk2.parameters.String(
                'Тип отчета',
                choices=[
                    ('allure', 'allure'),
                    ('html', 'html'),
                ],
                default='allure',
            )

            report_dir = sdk2.parameters.String(
                'Корневая дирректория отчета',
                default='html_reports',
            )

            report_send_to_st_comment = sdk2.parameters.Bool(
                'Отправлять отчёт в Startrek',
                default=True,
            )

        with sdk2.parameters.Group('Hermione cкринтесты') as screentests_section:
            screenshots_test = sdk2.parameters.Bool(
                'Ран с проверкой скриншотов',
                default=False,
            )

            screenshots_dir = sdk2.parameters.String(
                'Корневая дирректория со скриншотами',
                default=SCREENSHOTS_DIR,
                required=True,
            )

            screenshots_pack_resource_id = sdk2.parameters.Integer(
                'Id ресурса MARKET_SCREENSHOTS_PACK с эталонными скриншотами',
                required=False
            )

            update_refs = sdk2.parameters.Bool(
                'Запустить с обновлением ресурса MARKET_SCREENSHOTS_PACK',
                default=True,
            )

            with update_refs.value[True]:
                screenshots_sample_url = sdk2.parameters.String(
                    'Url для снятия эталонов',
                    required=False,
                )

        with sdk2.parameters.Group('Gemini скринтесты') as gemini_screentest_section:
            gemini_update_screen_pack = sdk2.parameters.Bool(
                'Обновить эталоны',
                description='Отметьте, '
                            'если нужно обновить ресурс эталонов скриншотов до Gemini-тестов',
                default=False
            )

            gemini_screenshots_pack_resource_id = sdk2.parameters.Integer(
                'Id ресурса MARKET_SCREENSHOTS_PACK с эталонными скриншотами',
                required=False
            )

            gemini_screen_pack_url = sdk2.parameters.String(
                'Хост стенда эталонов',
                description='При обновлении или поиске ресурса с эталонами скриншотов'
                            ' можно изменить адрес хоста, которому он соответствует'
            )


        with sdk2.parameters.Group('TestPalm') as testpalm_block:
            testpalm_run_export = sdk2.parameters.Bool(
                'Экспортировать ран',
                default=False,
                description='Создает тестовый ран в TestPalm')
            testpalm_run_version = FixVersion(
                'Релиз-версия',
                description='Пример: "4000.40.4"'
            )
            testpalm_run_issue = ReleaseTicket(
                'Релизный тикет',
                description='Пример: "MARKETVERSTKA-12345"'
            )

        with sdk2.parameters.Group('Семафор: проект') as semaphore_block:
            sem_name = sdk2.parameters.String(
                'Имя',
                default='',
                description='Фактическое название семафора будет "market-autotests_<имя>"'
            )
            sem_limit = sdk2.parameters.Integer(
                'Лимит',
                default=10,
                description='Доступный для одновременной работы лимит'
            )
            sem_weight = sdk2.parameters.Integer(
                'Вес',
                default=1,
                description='Сколько единиц из лимита тратится на каждый запуск'
            )

        with sdk2.parameters.Group('Семафор: Hermione') as h_semaphore_block:
            h_use_browser_semaphore = sdk2.parameters.Bool(
                'Использовать семафор для браузеров hermione',
                description='Использовать семафор для браузеров'
            )

        with sdk2.parameters.Group('Семафор: Gemini') as g_semaphore_block:
            g_use_browser_semaphore = sdk2.parameters.Bool(
                'Использовать семафор для браузеров gemini',
                default=False,
                description='Использовать семафор для браузеров'
            )

        metrics_params = create_metrics_params()
        # TODO: скорее всего, уже нигде не используется (@fenruga не нашел); поспрашивать, удалить
        solomon_params = make_solomon_parameters()

        with sdk2.parameters.Group('Environment') as environ_block:
            environ = sdk2.parameters.Dict('Environment variables')

    class Requirements(task_env.TinyRequirements):
        dns = ctm.DnsType.DNS64
        disk_space = DISK_SPACE

    @property
    def dist_dir(self):
        # удалить platform в MARKETFRONTECH-506
        return self.Parameters.dist_dir if self.Parameters.dist_dir else str(self.project_dir)

    @property
    def has_dist_dir(self):
        return bool(self.Parameters.dist_dir or self.Parameters.platform)

    # Genisys-config manager
    # requires `project_conf` property to be specified
    @property
    def config_manager(self):
        return ConfigManager(self)

    @property
    def project_conf(self):
        whole_config = self.config_manager.get_properties('sandbox-ci-market')

        return self.config_manager.get_project_conf(whole_config, {
            'project_name': self.Parameters.app_repo,
            'build_context': self.Parameters.project_build_context or None,
            'external_config': self.Parameters.external_config or None,
        })

    @property
    def project_name(self):
        return self.Parameters.app_repo

    @property
    def node_version(self):
        return self.Parameters.node_version or NODE_DEFAULT

    @property
    def app_resource_id(self):
        return self._app_resource_id

    @app_resource_id.setter
    def app_resource_id(self, value):
        self._app_resource_id = value

    @sdk2.header()
    def header(self):
        # todo: there must be a better way
        if self.Context.child_tasks_ids:
            tasks = list(
                sdk2.Task.find(
                    parent=self,
                    children=True,
                    id=self.Context.child_tasks_ids
                )
                .limit(2)
            )
            resources = list(
                sdk2.Resource.find(
                    resource_type=MARKET_AUTOTEST_REPORT,
                    task=tasks
                )
                .limit(2)
            )

            reports = []

            for resource in resources:
                resource = self.server.resource[resource.id].read()
                data = report_data(resource)
                reports.append(format_header(**data))

            if reports:
                return {'<h3 id="checks-reports">Report</h3>': reports}

    def _clone_repo(self):
        if self.Parameters.app_merge_commit:
            clone_repo_for_merge_commit(
                pipes.quote(self.Parameters.app_owner),
                pipes.quote(self.Parameters.app_repo),
                pipes.quote(self.Parameters.app_merge_commit),
                self.APP_REPO_PATH
            )
        else:
            clone_repo(
                pipes.quote(self.Parameters.app_owner),
                pipes.quote(self.Parameters.app_repo),
                pipes.quote(self.app_branch_final),
                self.APP_REPO_PATH
            )

    def _configure(self):
        rich_check_call(
            ["make", "configure"],
            task=self, alias="configure", cwd=self.APP_SRC_PATH
        )

    def _build(self):
        logging.info('Running project build with environment: {}'.format(os.environ))

        rich_check_call(
            ["make", "bootstrap"],
            task=self, alias="bootstrap", cwd=self.APP_SRC_PATH
        )

    def _create_app_pack(self):
        pack_app_archive_path = tempfile.mktemp(suffix=".tar.gz", prefix=self.Parameters.app_repo)

        rich_check_call(
            ["tar", "-C", os.path.join(self.APP_REPO_PATH, '..'), "-czf", pack_app_archive_path, self.Parameters.app_repo],
            task=self, alias="create_app_pack"
        )

        self._app_create_resource(pack_app_archive_path)

    def _app_create_resource(self, pack_app_archive_path):
        resource = MARKET_APP(
            self, "App tarball", "{}.tar.gz".format(self.Parameters.app_repo),
            app_repo=self.Parameters.app_repo,
            app_branch=self.app_branch_final,
            app_merge_commit=self.Parameters.app_merge_commit,
            app_status_check_commit=self.Parameters.app_status_check_commit,
        )
        app_res = sdk2.ResourceData(resource)

        app_res.path.write_bytes(spath.Path(pack_app_archive_path).read_bytes())
        app_res.ready()

        self.app_resource_id = resource.id

    def _prepare_env(self):
        # export order for correct priority
        env.export(self.project_conf.get('environ', {}))
        env.export(self.Parameters.environ)
        return

    def _prepare_app_branch(self):
        release_branch = find_opened_release_branch(
            owner=self.Parameters.app_owner,
            repo=self.Parameters.app_repo,
            branch_starts_with=self.Parameters.app_release_branch_starts_with,
        ) if self.Parameters.use_release_branch else None

        self.app_branch_final = release_branch or self.Parameters.app_branch

    def _run_subtasks(self):
        with MetatronEnv(self, nodejs_version=self.Parameters.node_version), self.memoize_stage.subtasks_ex(max_runs=1):
            self._prepare_env()

            logging.info('Current project configuration: {}'.format(self.project_conf))
            extra_params = {}
            extra_params.update(select_metrics_parameters(self.Parameters.metrics_params))
            extra_params.update(select_solomon_parameters(self.Parameters.solomon_params))

            hermione_task = self._create_subtask(
                task_type_name='hermione',
                task_type=MarketAutotestsHermione,
                app_branch=self.app_branch_final,
                testpalm_run_export=self.Parameters.testpalm_run_export,
                testpalm_run_version=self.Parameters.testpalm_run_version.encode('utf-8'),
                testpalm_run_issue=self.Parameters.testpalm_run_issue.encode('utf-8'),
                iteration_rerun_broken_tests=self.Parameters.iteration_rerun_broken_tests,
                broken_tests_resource_id=self.Parameters.broken_tests_resource_id,
                testpalm_ids_resource_id=self.Parameters.testpalm_ids_resource_id,

                use_browser_semaphore=self.Parameters.h_use_browser_semaphore,

                report_type=self.Parameters.report_type,
                report_dir=self.Parameters.report_dir,
                report_send_to_st_comment=self.Parameters.report_send_to_st_comment,

                screenshots_test=self.Parameters.screenshots_test,
                update_refs=self.Parameters.update_refs,
                screenshots_sample_url=self.Parameters.screenshots_sample_url,
                screenshots_pack_resource_id=self.Parameters.screenshots_pack_resource_id,
                screenshots_dir=self.Parameters.screenshots_dir,

                **extra_params
            )

            if hermione_task:
                self.Context.child_tasks_ids.append(hermione_task.id)

            gemini_task = self._create_subtask(
                task_type_name='gemini',
                task_type=MarketAutotestsGemini,
                app_branch=self.app_branch_final,
                screen_pack_url=self._resolve_screen_pack_url(),
                update_screen_pack=self.Parameters.gemini_update_screen_pack,
                st_issue=self.Parameters.testpalm_run_issue.encode('utf-8'),

                use_browser_semaphore=self.Parameters.g_use_browser_semaphore,
            )

            if gemini_task:
                self.Context.child_tasks_ids.append(gemini_task.id)

            raise sdk2.WaitTask(
                self.Context.child_tasks_ids,
                ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
                wait_all=True,
                timeout=SUBTASK_TIMEOUT
            )

    def _create_subtask(self, task_type, task_type_name, **params):
        if not self._get_test_param(task_type_name, 'enabled', False):
            return None

        actual_params = {
            'id': int(str(uuid.uuid4().int)[:4] + str(uuid.uuid4().int)[:4]),
            'description': self.Parameters.description,
            'environ': self.Parameters.environ,
            'node_version': self.Parameters.node_version,
            'ubuntu_version': self.Parameters.ubuntu_version,
            'app_resource_id': self.app_resource_id,
            'kill_timeout': max(self.Parameters.kill_timeout, KILL_TIMEOUT),
            'app_repo': self.Parameters.app_repo,
            'app_owner': self.Parameters.app_owner,
            'app_merge_commit': self.Parameters.app_merge_commit,
            'app_src_dir': self.Parameters.app_src_dir,
            'app_status_check_commit': self.Parameters.app_status_check_commit,
            'base_url': self.Parameters.base_url,
            'project_build_context': self.Parameters.project_build_context,
            'priority': {
                'class': self.Parameters.priority.cls,
                'subclass': self.Parameters.priority.scls,
            }
        }

        # Предподгатавливаем ресурс, чтобы сабтаска гермионы его наполнила и завершила
        if task_type_name == 'hermione' and not self.Parameters.broken_tests_resource_id:
            actual_params['broken_tests_resource'] = \
                MARKET_BROKEN_AUTOTEST_REPORT(self, 'Broken tests report', 'broken_tests_report')

        if self.Parameters.try_to_find_last_screen_pack:
            # Считаем, что если screenshots_pack_resource_id передан в параметрах таски (т.е. в params он не 0 и не None), то
            # он сильнее требования найти последние эталоны.
            # Если же параметра нет или там лежит 0 или None, то убираем его из params - т.к. params перетирают actual_params
            try:
                if not params['screenshots_pack_resource_id']:
                    params.pop('screenshots_pack_resource_id')
            except KeyError:
                pass

            latest_pack = self._find_last_screen_pack(task_type_name)
            if latest_pack and task_type_name == 'hermione':
                actual_params['screenshots_pack_resource_id'] = latest_pack.id
            if latest_pack and task_type_name == 'gemini':
                actual_params['screen_pack'] = latest_pack

        if task_type_name == 'hermione' and not self.Parameters.screenshots_pack_resource_id and self.Parameters.screenshots_test and self.Parameters.update_refs:
            actual_params['market_screenshots_pack'] = self._create_hermione_screenshots_pack_resource()

        if task_type_name == 'gemini' and self.Parameters.gemini_screenshots_pack_resource_id:
            actual_params['screen_pack'] = sdk2.Resource.find(id=self.Parameters.gemini_screenshots_pack_resource_id).first()

        actual_params.update(params)

        logging.info("Creating {type} subtask with id: {id} and params: {parameters}".format(
            type=task_type_name,
            id=actual_params['id'],
            parameters=actual_params
        ))

        subtask = task_type(self, **actual_params)
        subtask.enqueue()

        return subtask

    def _find_last_screen_pack(self, task_type_name):
        if task_type_name not in ['hermione', 'gemini']:
            return None

        task_type = None
        input_parameters = None

        logging.info("Ищем свежие эталоны скриншотов")

        if task_type_name == 'hermione':
            task_type = MarketAutotests
            input_parameters = {
                "project_build_context": self.Parameters.project_build_context,
                "screenshots_test": True,
                "update_refs": True,
                "app_branch": "master",
            }

        if task_type_name == 'gemini':
            task_type = MarketAutotestsGeminiUpdate
            input_parameters = {
                "project_build_context": self.Parameters.project_build_context,
                "app_branch": "master",
            }

        latest_update_pack_tasks = sdk2.Task.find(
            task_type=task_type,
            status=ctt.Status.SUCCESS,
            input_parameters=input_parameters,
            children=True,
        ).order(-sdk2.Task.id).limit(5)

        if latest_update_pack_tasks.count == 0:
            logging.warning("Не нашли ни одной таски обновления скриншотов")
            return None

        for task in latest_update_pack_tasks:  # type: sdk2.Task
            logging.debug("Ищем ресурс скриншотов у таски {}".format(task.id))
            pack = sdk2.Resource.find(
                resource_type=MARKET_SCREENSHOTS_PACK,
                task=task,
                state=ctr.State.READY
            ).first()  # type: sdk2.Resource

            if pack:
                logging.debug("Нашли ресурс {}".format(pack.id))
                return pack  # type: MARKET_SCREENSHOTS_PACK

        logging.info("Не нашли ни одного живого ресурса со скриншотами")
        return None

    def _create_hermione_screenshots_pack_resource(self):
        attributes = dict({
            'status': ctt.Status.SUCCESS,
            'ttl': 365,
            'tool': 'hermione',
            'type': 'screenshots-pack',
            'host': self.Parameters.base_url or 'unknown_host'
        })

        return MARKET_SCREENSHOTS_PACK(self, "Screenshots pack", 'hermione_screenshots_pack.tar.gz', **attributes)

    def _resolve_screen_pack_url(self):
        params = self.Parameters

        if params.gemini_screen_pack_url:
            return str(params.gemini_screen_pack_url)
        else:
            return self._get_test_param('gemini-update', 'screen_pack_url', str(params.base_url))

    def _get_test_param(self, family, flag, default=None):
        logging.info("family {}, flag {}".format(family, flag))
        return self.config_manager.get_deep_value(['tests', family, flag], default)

    def _fail_on_childs_fail(self):
        with self.memoize_stage.fail_on_childs_fail(max_runs=1):
            subtasks = list(
                sdk2.Task.find(
                    parent=self,
                    children=True,
                    id=self.Context.child_tasks_ids
                )
                .limit(2)
            )

            failed_tasks = MetaTaskManager.filter_failed(subtasks)

            if failed_tasks:
                failure_msg = 'Some of child tasks failed, see message below'
                failure_links = []

                for task in failed_tasks:
                    failure_links.append('<a href="{url}" target="_blank">{text}</a>'.format(
                        url=get_task_link(task.id),
                        text=task.type
                    ))

                self.failure_links = '\n'.join(failure_links)

                raise TaskFailure(failure_msg)

    def on_enqueue(self):
        setup_container(self)

        if self.Parameters.sem_name:
            self.Requirements.semaphores = ctt.Semaphores(
                acquires=[
                    ctt.Semaphores.Acquire(
                        name="market-autotests_{}".format(self.Parameters.sem_name),
                        capacity=self.Parameters.sem_limit,
                        weight=self.Parameters.sem_weight
                    )
                ],
                release=(
                    ctt.Status.Group.BREAK, ctt.Status.Group.FINISH
                )
            )

    def on_prepare(self):
        # NB: MetatronEnv clears env on exit
        with MetatronEnv(self, nodejs_version=self.Parameters.node_version), self.memoize_stage.preparation(max_runs=1):
            logging.info('prepare task with conf: {}'.format(self.project_conf))
            self.root_dir = tempfile.mkdtemp()
            self.APP_REPO_PATH = os.path.join(self.root_dir, self.Parameters.app_repo)

            if self.Parameters.app_src_dir:
                self.APP_SRC_PATH = os.path.join(self.APP_REPO_PATH, self.Parameters.app_src_dir)
            else:
                self.APP_SRC_PATH = self.APP_REPO_PATH

            self._prepare_env()
            self._prepare_app_branch()

            if self.Parameters.rebuild_app:
                self._clone_repo()
                self._configure()
                self._build()
                self._create_app_pack()
            else:
                self.app_resource_id = self.Parameters.app_resource_id

    def on_execute(self):
        self._run_subtasks()
        self._fail_on_childs_fail()

    def on_failure(self, prev_status):
        if self.failure_links:
            self.set_info(self.failure_links, do_escape=False)
