# -*- coding: utf-8 -*-

import logging
import os

from sandbox import sdk2
from sandbox.common.types import misc as ctm
from sandbox.common.types import task as ctt
from sandbox.projects.market.front.helpers.MetatronEnv import MetatronEnv
from sandbox.projects.market.front.helpers.sandbox_helpers import rich_check_call, unpack_resource, format_header, report_data, prepare_ginny_user_config
from sandbox.projects.market.front.helpers.node import create_node_selector
from sandbox.projects.market.front.helpers.ubuntu import create_ubuntu_selector, setup_container
from sandbox.projects.market.resources import MARKET_SCREENSHOTS_PACK, MARKET_AUTOTEST_REPORT
from sandbox.projects.sandbox_ci.managers import ConfigManager
from sandbox.projects.sandbox_ci.utils import env
from sandbox.sandboxsdk import errors
from sandbox.projects.common import task_env

DISK_SPACE = 3 * 1024  # 3 Gb


class MarketAutotestsGeminiUpdate(sdk2.Task):
    """
    Обновление эталонов для Gemini-тестов сервисов Маркета
    """
    TOOL = 'gemini'
    GEMINI_ROOT_PATH = 'index.html'

    class Context(sdk2.Context):
        report_resource_id = None

    class Parameters(sdk2.Task.Parameters):
        ubuntu_version = create_ubuntu_selector()
        node_version = create_node_selector()

        app_resource_id = sdk2.parameters.Integer(
            'Id ресурса с приложением',
            required=True
        )

        with sdk2.parameters.Group('GitHub репозиторий проекта') as github_repo_block:
            app_repo = sdk2.parameters.String(
                "Репозиторий",
                default='MarketNode',
                required=True
            )

            app_branch = sdk2.parameters.String(
                "Ветка",
                default='master',
                required=True
            )

            app_src_dir = sdk2.parameters.String(
                "Кастомный путь корня приложения внутри репозитория"
            )

        with sdk2.parameters.Group('Project build') as project_build_block:
            project_build_context = sdk2.parameters.String(
                'Профиль конфигурации в Genisys',
                default=''
            )

        with sdk2.parameters.Group('Тестовое окружение') as test_target:
            screen_pack_url = sdk2.parameters.String(
                'Хост стенда эталонов',
                description='При обновлении или поиске ресурса с эталонами скриншотов'
                            ' можно изменить адрес хоста, которому он соответствует'
            )

    class Requirements(task_env.BuildRequirements):
        dns = ctm.DnsType.DNS64
        disk_space = DISK_SPACE

    def on_enqueue(self):
        super(MarketAutotestsGeminiUpdate, self).on_enqueue()
        setup_container(self)

    # Genisys-config manager
    # requires `project_conf` property to be specified
    @property
    def config_manager(self):
        return ConfigManager(self)

    @property
    def project_conf(self):
        whole_config = self.config_manager.get_properties(section='sandbox-ci-market')

        return self.config_manager.get_project_conf(whole_config, {
            'project_name': self.Parameters.app_repo,
            'build_context': self.Parameters.project_build_context or None,
        })

    @property
    def screen_pack_url(self):
        if self.Parameters.screen_pack_url:
            return str(self.Parameters.screen_pack_url)
        else:
            return str(
                self._get_conf_param(['tests', 'gemini-update', 'screen_pack_url'])
                or
                self._get_conf_param(['environ', 'gemini_root_url'])
            )

    @property
    def project_name(self):
        return self.Parameters.project_github_repo

    @property
    def report_description(self):
        return self.project_name

    @property
    def github_context(self):
        return self.format_github_context(self.report_description)

    @classmethod
    def format_github_context(cls, description):
        return u'{}: {}'.format(cls.TOOL.capitalize(), description)

    @sdk2.header()
    def header(self):
        resource_id = self.Context.report_resource_id

        if resource_id:
            resource = self.server.resource[resource_id].read()
            data = report_data(resource)
            report = {'<h3 id="checks-reports">Report</h3>': [format_header(**data)]}

            return report

    def _call_gemini_update(self):
        rich_check_call(
            ["make", "gemini_update"],
            task=self, alias="gemini_update", cwd=self.app_src_path,
        )

    def _call_gemini_pack(self):
        rich_check_call(
            ["make", "gemini_pack"],
            task=self, alias="gemini_pack", cwd=self.app_src_path,
        )

    def _prepare_dirs(self):
        self.root_dir = str(self.path())
        self.app_src_path = os.path.join(self.root_dir, self.Parameters.app_repo)

        if self.Parameters.app_src_dir:
            self.app_src_path = os.path.join(self.app_src_path, self.Parameters.app_src_dir)

    def _prepare_env(self):
        env.export(self._get_conf_environ())
        env.export({
            'gemini_root_url': self.screen_pack_url
        })

    def _prepare_ginny_user_config(self):
        prepare_ginny_user_config(self, 'gemini')

    def _prepare_app_resource(self):
        _id = self.Parameters.app_resource_id
        self.app_resource = sdk2.Resource[_id]

        logging.info('Unpacking application resource {}'.format(_id))
        unpack_resource(self, self.app_resource, self.root_dir)

    def _update(self):
        status = ctt.Status.SUCCESS

        try:
            logging.info('Running {} update screenshot pack'.format(self.TOOL))
            self._call_gemini_update()
        except errors.SandboxSubprocessError:
            status = ctt.Status.FAILURE
            raise
        finally:
            self._create_report(status)

    def _create_pack(self, **kwargs):
        logging.info('Saving {} screenshot pack'.format(self.TOOL))

        self._call_gemini_pack()

        abs_path = self.path(self.app_src_path, 'screenshots-pack.tar.gz')

        if not abs_path.exists():
            logging.debug('Not adding resource {}: no such path'.format(abs_path))
            return

        rel_path = abs_path.relative_to(self.path())
        attributes = dict({
            'status': ctt.Status.SUCCESS,
            'ttl': 365,
            'tool': self.TOOL,
            'type': 'screenshots-pack',
            'host': self.screen_pack_url
        },
            **kwargs
        )

        resource = MARKET_SCREENSHOTS_PACK(self, "Screenshots pack", str(rel_path), **attributes)

        sdk2.ResourceData(resource).ready()

    def _create_report(self, status):
        logging.info('Generating Gemini report')

        params = {
            'type': '{}-report'.format(self.TOOL),
            'status': status,
            'root_path': self.GEMINI_ROOT_PATH
        }
        src_path = os.path.join(self.app_src_path, "html_reports")

        resource = MARKET_AUTOTEST_REPORT(self, "Gemini report", src_path, **params)
        sdk2.ResourceData(resource).ready()

        self.Context.report_resource_id = resource.id

        return resource

    def _get_conf_param(self, path):
        return self.config_manager.get_deep_value(path, None)

    def _get_conf_environ(self):
        return self.project_conf.get('environ', {})

    def _get_conf_browsers(self):
        return self.project_conf.get('gemini_browsers', {})

    def _get_browsers(self):
        return self._get_conf_browsers()

    def on_prepare(self):
        with MetatronEnv(self, nodejs_version=self.Parameters.node_version):
            logging.debug('Start task prepare')
            self._prepare_dirs()
            self._prepare_app_resource()

    def on_execute(self):
        # all environmental variables must be set inside MetatronEnv
        with MetatronEnv(self, nodejs_version=self.Parameters.node_version):
            logging.debug('Start task execute')
            self._prepare_env()
            self._update()
            self._create_pack()
