# -*- coding: utf-8 -*-
import json
import logging
import os

import sandbox.sdk2.path as spath
from sandbox import sdk2
from sandbox.common.types import misc as ctm
from sandbox.common.types import task as ctt
from sandbox.common.types import resource as ctr
from sandbox.common.utils import get_task_link
from sandbox.projects.market.front.helpers.MetatronEnv import MetatronEnv
from sandbox.projects.market.front.helpers.github import change_status, GitHubStatus, GitHubStatusDescription
from sandbox.projects.market.front.helpers.node import NODE_DEFAULT
from sandbox.projects.market.front.helpers.sandbox_helpers import get_resource_http_proxy_link as get_resource_link, \
    rich_check_call, rich_get_output, unpack_resource, report_data, format_header, \
    prepare_browsers_semaphores, prepare_ginny_user_config
from sandbox.projects.market.front.helpers.metrics import create_metrics_params, collect_cases_stats, collect_cases_errors, select_metrics_parameters
from sandbox.projects.market.front.helpers.node import create_node_selector
from sandbox.projects.market.front.helpers.ubuntu import create_ubuntu_selector, setup_container
from sandbox.projects.market.front.helpers.startrack import FixVersion, ReleaseTicket, build_test_statistic
from sandbox.projects.market.front.helpers.solomon import make_solomon_parameters, push_sensors_by_parameters, \
    is_solomon_parameters_defined
from sandbox.projects.market.resources import MARKET_AUTOTEST_REPORT, MARKET_BROKEN_AUTOTEST_REPORT, MARKET_AUTOTEST_STATS, MARKET_AUTOTEST_ERRORS
from sandbox.projects.sandbox_ci.managers import ConfigManager
from sandbox.projects.sandbox_ci.utils import env
from sandbox.projects.sandbox_ci.utils.context import GitRetryWrapper
from sandbox.sandboxsdk import errors
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.projects.common import task_env
from sandbox.common.utils import chain
from sandbox.common.types.task import Status

node_param = '--max-old-space-size=4096'
DISK_SPACE = 3 * 1024  # 3 Gb
SCREENSHOTS_DIR = 'hermione-screenshots'
SCREENSHOTS_ARCHIVE_NAME = "screenshots-pack.tar.gz"


class MarketAutotestsHermione(sdk2.Task):
    """
    Hermione-тесты сервисов Маркета
    """
    TOOL = 'hermione'
    TEST_RUN_FILE = 'test_run.json'
    REPORT_ROOT_PATH = 'index.html'

    class Context(sdk2.Context):
        report_resource_id = None
        stats_resource_id = None
        errors_resource_id = None

    class Parameters(sdk2.Task.Parameters):
        ubuntu_version = create_ubuntu_selector()
        node_version = create_node_selector()
        # todo: is there better way to pass Resource?
        app_resource_id = sdk2.parameters.Integer(
            'Id ресурса с приложением',
            required=True
        )
        # Ресурс, из которого берутся тесты для прогона
        broken_tests_resource_id = sdk2.parameters.Integer(
            'Id ресурса со списком поломанных тестов',
            required=False
        )

        iteration_rerun_broken_tests = sdk2.parameters.Bool(
            'Включить итеративный перепрогон упавших hermione-тестов',
            description='Для включения этой опции, в проекте необходимо использовать hermione-broken-tests@1.2.0',
            default=False
        )

        # Ресурс, в который пишем названия упавших тестов
        broken_tests_resource = sdk2.parameters.ParentResource(
            'Ресурс для списка поломанных тестов',
            required=False
        )

        # Ресурс, который содержит testpalm id для фильтрации тестов
        testpalm_ids_resource_id = sdk2.parameters.Integer(
            'Id ресурса со списком testpalm id',
            required=False
        )

        # todo: this section is only informational use in CI, so, maybe just for description?
        with sdk2.parameters.Group('GitHub репозиторий проекта') as github_repo_block:
            # todo: some kinda special naming for sb_ci
            # project_github_owner
            app_owner = sdk2.parameters.String(
                'GitHub owner',
                description='Логин владельца репозитория или название организации',
                default='market',
                required=True
            )
            app_repo = sdk2.parameters.String(
                "Репозиторий",
                default='marketfront',
                required=True
            )

            app_branch = sdk2.parameters.String(
                "Ветка",
                default='master',
                required=True
            )

            app_merge_commit = sdk2.parameters.String(
                "Merge-коммит",
            )

            app_src_dir = sdk2.parameters.String(
                "Кастомный путь корня приложения внутри репозитория"
            )

            app_status_check_commit = sdk2.parameters.String(
                "Коммит для отправки статус-чеков",
            )

        # todo: desc?
        with sdk2.parameters.Group('Project build') as project_build_block:
            project_build_context = sdk2.parameters.String(
                'Профиль конфигурации в Genisys',
                description='https://wiki.yandex-team.ru/Market/Verstka/SandboxCookBook/#profilikonfiguracijjzapuska',
                default=''
            )
            external_config = sdk2.parameters.String(
                'Внешний конфиг',
                default=''
            )

        with sdk2.parameters.Group('Тестовое окружение') as test_target:
            base_url = sdk2.parameters.String(
                'Тестируемый хост',
                description='Адрес машины, на который будут ходить тесты'
            )

        with sdk2.parameters.Group('Отчет') as report_section:
            report_type = sdk2.parameters.String(
                'Тип отчета',
                choices=[
                    ('allure', 'allure'),
                    ('html', 'html'),
                ],
                default='allure',
            )

            report_dir = sdk2.parameters.String(
                'Корневая дирректория отчета',
                default='html_reports',
            )

            report_send_to_st_comment = sdk2.parameters.Bool(
                'Отправлять отчёт комментарием в ST?',
                default=True,
            )

        with sdk2.parameters.Group('Hermione cкринтесты') as screentests_section:
            screenshots_test = sdk2.parameters.Bool(
                'Ран с проверкой скриншотов',
                default=False,
            )

            # Ресурс, в который пишем полученные "эталонные" скриншоты
            market_screenshots_pack = sdk2.parameters.ParentResource(
                'Ресурс MARKET_SCREENSHOTS_PACK с эталонными скриншотами',
                required=False
            )

            screenshots_dir = sdk2.parameters.String(
                'Корневая дирректория со скриншотами',
                default=SCREENSHOTS_DIR,
                required=True,
            )

            screenshots_pack_resource_id = sdk2.parameters.Integer(
                'Id ресурса MARKET_SCREENSHOTS_PACK с эталонными скриншотами',
                required=False
            )

            update_refs = sdk2.parameters.Bool(
                'Запустить с обновлением ресурса MARKET_SCREENSHOTS_PACK',
                default=True,
            )

            with update_refs.value[True]:
                screenshots_sample_url = sdk2.parameters.String(
                    'Url для снятия эталонов',
                    required=False,
                )

        with sdk2.parameters.Group('TestPalm') as testpalm_block:
            testpalm_run_export = sdk2.parameters.Bool(
                'Экспортировать ран',
                default=False,
                description='Создает тестовый ран в TestPalm')

            testpalm_run_version = FixVersion(
                'Релиз-версия',
                description='Пример: "4000.40.4"'
            )

            testpalm_run_issue = ReleaseTicket(
                'Релизный тикет',
                description='Пример: "MARKETVERSTKA-12345"'
            )

        with sdk2.parameters.Group('Семафор: Hermione') as semaphore_block:
            use_browser_semaphore = sdk2.parameters.Bool(
                'Использовать семафор для браузеров hermione',
                default=False,
                description='Использовать семафор для браузеров'
            )

        metrics_params = create_metrics_params()
        # TODO: скорее всего, уже нигде не используется (@fenruga не нашел); поспрашивать, удалить
        solomon_params = make_solomon_parameters()

        with sdk2.parameters.Group('Environment') as environ_block:
            environ = sdk2.parameters.Dict('Environment variables')

    class Requirements(task_env.BuildRequirements):
        dns = ctm.DnsType.DNS64
        disk_space = DISK_SPACE
        environments = [
            PipEnvironment('yandex_tracker_client', version="1.3", custom_parameters=["--upgrade-strategy only-if-needed"]),
            PipEnvironment('startrek_client', version="2.3.0", custom_parameters=["--upgrade-strategy only-if-needed"])
        ]

    @property
    def app_commit(self):
        return self._get_commit_sha()

    # Genisys-config manager
    # requires `project_conf` property to be specified
    @property
    def config_manager(self):
        return ConfigManager(self)

    @property
    def project_conf(self):
        whole_config = self.config_manager.get_properties('sandbox-ci-market')

        return self.config_manager.get_project_conf(whole_config, {
            'project_name': self.Parameters.app_repo,
            'build_context': self.Parameters.project_build_context or None,
            'external_config': self.Parameters.external_config or None,
        })

    @property
    def project_name(self):
        return self.Parameters.app_repo

    @property
    def report_description(self):
        return self.project_name

    @property
    def github_context(self):
        return self.format_github_context(self.Parameters.project_build_context)

    @property
    def node_version(self):
        return self.Parameters.node_version or NODE_DEFAULT

    @property
    def testpalm_run_path(self):
        return os.path.join(self.app_src_path, self.TEST_RUN_FILE)

    @classmethod
    def format_github_context(cls, description):
        return u'[{}] {}'.format(cls.TOOL.capitalize(), description)

    @sdk2.header()
    def header(self):
        resource_id = self.Context.report_resource_id

        if resource_id:
            self.parent.Context.resources_to_report.append(resource_id)

            resource = self.server.resource[resource_id].read()
            data = report_data(resource)
            report = {'<h3 id="checks-reports">Report</h3>': [format_header(**data)]}

            return report

    def _get_conf_environ(self):
        return self.project_conf.get('environ', {})

    def _get_conf_browsers(self):
        return self.project_conf.get('hermione_browsers', {})

    def _get_browsers(self):
        return self._get_conf_browsers()

    def _get_sessions_per_browser(self):
        return self.Parameters.sessions_per_browser or self._get_conf_browsers().get('sessions_per_browser', 10)

    def _prepare_dirs(self):
        self.root_dir = str(self.path())
        self.app_src_path = os.path.join(self.root_dir, self.Parameters.app_repo)

        if self.Parameters.app_src_dir:
            self.app_src_path = os.path.join(self.app_src_path, self.Parameters.app_src_dir)

    def _prepare_env(self):
        project_dir = str(self.app_src_path)

        if 'NODE_PATH' in os.environ:
            os.environ['NODE_PATH'] = '{}:{}'.format(project_dir, os.environ['NODE_PATH'])
        else:
            os.environ['NODE_PATH'] = project_dir

        os.environ['CURRENT_GIT_BRANCH'] = self.Parameters.app_branch
        os.environ['CURRENT_GIT_REPOSITORY'] = self.Parameters.app_repo

        # todo: выпилить?
        os.environ['EXT_ENV'] = 'sanboxCiHermione'

        # export order for correct priority
        env.export(self._get_conf_environ())

        if self.Parameters.base_url:
            os.environ['hermione_base_url'] = self.Parameters.base_url

        if self.Parameters.screenshots_sample_url and self.Parameters.update_refs:
            os.environ['hermione_base_url'] = self.Parameters.screenshots_sample_url

        title = 'Релиз: {}, пак: {}'.format(
            self.Parameters.testpalm_run_version or 'не указан',
            self.Parameters.project_build_context,
        )

        os.environ['hermione-allure_executor_title'] = title
        os.environ['hermione-allure_executor_link'] = get_task_link(self.id)

        if self.Parameters.testpalm_run_export:
            os.environ['hermione-allure_testpalm_run_export'] = 'true'
            os.environ['hermione-allure_testpalm_run_export_to_file'] = self.testpalm_run_path
            os.environ['hermione-allure_testpalm_run_version'] = self.Parameters.testpalm_run_version
            os.environ['hermione-allure_testpalm_run_issue'] = self.Parameters.testpalm_run_issue
            os.environ['hermione-allure_testpalm_run_author'] = self.author
            os.environ['hermione-allure_testpalm_token'] = os.environ['TESTPALM_OAUTH_API_TOKEN'] # для совместимости со старыми скриптами

        # Заставляем плагин фильтровать тесты, если на входе есть ресурс с их именами
        if self.Parameters.broken_tests_resource_id:
            os.environ['hermione-broken-tests_mode'] = 'filter'

            # В hermione-broken-tests@1.2.0 добавилась возможность генерить отчет по отфильтрованным тестам
            if self.Parameters.iteration_rerun_broken_tests:
                os.environ['hermione-broken-tests_mode'] = 'recollect'

        os.environ['SANDBOX_TASK_ID'] = str(self.id)

        os.environ['market_kadavr_meta_sandbox_task_id'] = str(self.id)
        os.environ['market_kadavr_meta_git_branch'] = self.Parameters.app_branch
        os.environ['market_kadavr_meta_git_repository'] = self.Parameters.app_repo

        # export order for correct priority
        env.export(self.Parameters.environ)

    def _prepare_app_resource(self):
        id = self.Parameters.app_resource_id
        app_resource = sdk2.Resource[id]

        logging.info('Unpacking application resource {}'.format(id))
        unpack_resource(self, app_resource, self.root_dir)

    def _prepare_broken_tests_resource(self):
        broken_tests_resource_id = self.Parameters.broken_tests_resource_id
        iteration_rerun_broken_tests = self.Parameters.iteration_rerun_broken_tests

        if broken_tests_resource_id:
            report_resource = None

            if iteration_rerun_broken_tests:
                report_resource = MARKET_BROKEN_AUTOTEST_REPORT.find(
                    state=ctr.State.READY,
                    attrs=dict(
                        reference_resource_id=broken_tests_resource_id
                    )
                ).order(-sdk2.Resource.id).first()

            if not report_resource:
                report_resource = MARKET_BROKEN_AUTOTEST_REPORT[broken_tests_resource_id]

            resource_data = sdk2.ResourceData(report_resource)
            report_dir = os.path.join(self.app_src_path, 'broken_tests_report')

            logging.debug('Copy resource, path: {} to {}'.format(resource_data.path, report_dir))

            rich_check_call(
                ['cp', '-R', '{}'.format(resource_data.path), report_dir],
                task=self, alias='copy broken tests resource', cwd=self.root_dir,
            )

            # нужны права для записи, так как в итеративном режиме перезапуска тестов,
            # мы обновляем файл со списком упавших тестов
            os.chmod(os.path.join(report_dir, 'broken-tests.txt'), 0o755)

    def _prepare_screen_pack_resource(self):
        """ Копируем скриншоты из ресурса MARKET_SCREENSHOTS_PACK
        """
        id = self.Parameters.screenshots_pack_resource_id

        if id:
            screen_pack_resource = sdk2.Resource['MARKET_SCREENSHOTS_PACK'].find(id=id).first()

            resource_data = sdk2.ResourceData(screen_pack_resource)
            screens_dir = os.path.dirname(os.path.join(self.app_src_path, self.Parameters.screenshots_dir))

            logging.info('Unpacking resource, path: {} to {}'.format(resource_data.path, screens_dir))
            unpack_resource(self, screen_pack_resource, screens_dir)

    def _prepare_testpalm_ids_resource(self):
        """Если на входе есть ресурс с файлом для фильтрации тестов по testpalm id,
        то подкладываем его в приложение, чтобы плагин-фильтратор смог его использовать.
        В случае если на входе есть еще и ресурс с поломаными тестами, то фильтрацию
        по testlpalm id не производим
        """
        id = self.Parameters.testpalm_ids_resource_id
        broken_tests_id = self.Parameters.broken_tests_resource_id

        if id and not broken_tests_id:
            report_resource = sdk2.Resource['MARKET_TESTPALM_IDS'].find(id=id).first()

            resource_data = sdk2.ResourceData(report_resource)
            resource_data_txt = resource_data.path.read_bytes()

            if resource_data_txt:
                os.environ['AUTOTESTS_HERMIONE_CASE_FILTER'] = '{{"id": "{}"}}'.format(resource_data_txt)

    def _prepare_browser_semaphore(self):
        prepare_browsers_semaphores(self)

    def _prepare_ginny_user_config(self):
        prepare_ginny_user_config(self, 'hermione')

    def _test(self):
        status = ctt.Status.SUCCESS

        try:
            logging.info('Running {} tests with environ: {}'.format(self.TOOL, os.environ))
            self._call_test()
        except errors.SandboxSubprocessError:
            status = ctt.Status.FAILURE
            raise
        finally:
            self._make_reports(status)
            self._upload_metrics()

    def _screens_test(self):
        status = ctt.Status.SUCCESS

        try:
            logging.info('Running {} screen tests with environ: {}'.format(self.TOOL, os.environ))

            if self.Parameters.update_refs:
                self._call_screens_update()
                self._create_screenshots_resource()
                return

            self._call_test()
        except errors.SandboxSubprocessError:
            status = ctt.Status.FAILURE
            raise
        finally:
            self._make_reports(status)

    def _create_at_report(self, **params):
        self.at_report_path = os.path.join(self.app_src_path, self.Parameters.report_dir)

        resource = MARKET_AUTOTEST_REPORT(self, "{} report".format(self.Parameters.report_type), self.at_report_path, **params)
        sdk2.ResourceData(resource).ready()

        self.Context.report_resource_id = resource.id

        return resource

    def _create_metrics_resource(self, stats, **params):
        stats_path = os.path.join(str(self.app_src_path), "stats.json")
        logging.info("Writing run's stats: path: {}\n", stats_path)

        with open(stats_path, 'w') as f:
            json.dump(stats, f, indent=4)

        resource = MARKET_AUTOTEST_STATS(self, "Cases stats", stats_path, **params)
        sdk2.ResourceData(resource).ready()

        self.Context.stats_resource_id = resource.id

        return resource

    def _create_errors_resource(self, errors, **params):
        errors_path = os.path.join(str(self.app_src_path), "errors.json")
        logging.info("Writing run's errors: path: {}\n", errors_path)

        with open(errors_path, 'w') as f:
            json.dump(errors, f, indent=4)

        resource = MARKET_AUTOTEST_ERRORS(self, "Cases errors", errors_path, **params)
        sdk2.ResourceData(resource).ready()

        self.Context.errors_resource_id = resource.id

        return resource

    def _create_broken_tests_report(self):
        """На этапе выполнения тестов может создаваться файл с именами упавших тестов, по которым можно будет
        фильтровать. Если такой файл есть, то создаём из него ресурс. В любом случае создаём ресурс с пустым файлом.
        Если включена поция iteration_rerun_broken_tests, то создаем отчет с атрибутом reference_resource_id,
        по которому будем искать самый свежый отчет с упавшими тестами
        """

        resource = None
        broken_tests_resource = self.Parameters.broken_tests_resource
        broken_tests_resource_id = self.Parameters.broken_tests_resource_id
        iteration_rerun_broken_tests = self.Parameters.iteration_rerun_broken_tests

        if broken_tests_resource_id and iteration_rerun_broken_tests:
            resource = MARKET_BROKEN_AUTOTEST_REPORT(
                self, 'Broken tests report', 'broken_tests_report',
                reference_resource_id=broken_tests_resource_id
            )
        else:
            resource = broken_tests_resource

        resource_data = sdk2.ResourceData(resource)
        resource_data.path.mkdir(0o755, parents=True, exist_ok=True)

        src_path = os.path.join(self.app_src_path, 'broken_tests_report')
        src_file = os.path.join(src_path, 'broken-tests.txt')

        # Если в проекте влючен плагин
        data = spath.Path(src_file).read_bytes() if resource and os.path.exists(src_file) else ''

        resource_data.path.joinpath('broken-tests.txt').write_bytes(data)
        resource_data.ready()

    def _create_screenshots_resource(self):
        logging.info('Saving {} screenshot pack'.format(self.TOOL))

        resource = self.Parameters.market_screenshots_pack
        resource_data = sdk2.ResourceData(resource)

        abs_path = os.path.join(self.app_src_path, self.Parameters.screenshots_dir)

        if not os.path.exists(abs_path):
            logging.debug('Not adding resource {}: no such path'.format(abs_path))
            return

        archive_path = self._create_screens_pack(abs_path)

        data = spath.Path(archive_path).read_bytes()
        resource_data.path.write_bytes(data)
        resource_data.ready()

    def _create_screens_pack(self, dist_path):
        screens_pack_dir = os.path.dirname(dist_path)
        screens_pack_name = os.path.basename(dist_path)

        logging.info("Screenshots directory: {}".format(dist_path))

        rich_check_call(
            ["tar", "-C", screens_pack_dir, "-czf", SCREENSHOTS_ARCHIVE_NAME, screens_pack_name],
            task=self, alias="create_screens_pack", cwd=self.app_src_path
        )

        return os.path.join(self.app_src_path, SCREENSHOTS_ARCHIVE_NAME)

    def _make_reports(self, status):
        logging.info('Generating {} report'.format(self.Parameters.report_type))

        if self.Parameters.report_type == 'allure':
            self._call_allure()

        self.at_report_resource = self._create_at_report(
            config=self.Parameters.project_build_context,
            project=self.Parameters.app_repo,
            report_description=self.Parameters.app_repo,
            type='{}-{}-report'.format(self.TOOL, self.Parameters.report_type),
            task_id=self.id,
            tool=self.TOOL,
            status=status,
            root_path=self.REPORT_ROOT_PATH
        )

        report_url = '{}/index.html'.format(get_resource_link(self.at_report_resource)) if self.at_report_resource else ''

        if not self.Parameters.broken_tests_resource_id or self.Parameters.iteration_rerun_broken_tests:
            self._create_broken_tests_report()

        self._import_testpalm_run(report_url)
        self._send_startrek_report(report_url)
        self._send_solomon_sensors()

    def _import_testpalm_run(self, report_url):
        if self.Parameters.testpalm_run_export:
            run_patch = '{{"launcherInfo":{{"reportLink":"{}"}}}}'.format(report_url)
            os.environ['GINNY_TESTPALM_RUN_PATCH'] = run_patch

            self._call_testpalm()

    def _send_solomon_sensors(self):
        if is_solomon_parameters_defined(self.Parameters.solomon_params):
            with open('{}/html_reports/history/duration-trend.json'.format(self.app_src_path), 'r') as file:
                duration = json.load(file)[0]['data']['duration']
                push_sensors_by_parameters(self.Parameters.solomon_params, sensors=[{
                    'labels': {
                        'sensor': 'autotests_hermione_duration_ms',
                        'pack': self.Parameters.project_build_context,
                    },
                    'value': duration,
                }])

    def _send_startrek_report(self, report_url):
        if not self.Parameters.report_send_to_st_comment:
            logging.info('Will not send report to Startrek')
            return

        logging.info('Export statistic to startrek')
        stat_reporter_path = '{}/stat-reporter.json'.format(self.app_src_path)

        if not self.Parameters.testpalm_run_issue:
            logging.debug('Release ticket not specified')
            return

        if not os.path.isfile(stat_reporter_path):
            logging.debug('Report file {} not found'.format(self.app_src_path))
            return

        oauth_token = sdk2.Vault.data('robot-metatron-st-token')

        from startrek_client import Startrek

        st = Startrek(useragent='robot-metatron', token=oauth_token)
        issue = st.issues[self.Parameters.testpalm_run_issue]
        task_url = get_task_link(self.id)

        with open(stat_reporter_path, 'r') as f:
            report_data = json.loads(f.read())

            if not report_data:
                logging.info('Empty report data in {}'.format(stat_reporter_path))
                return

            self._change_gh_status(
                GitHubStatus.SUCCESS if report_data[0]['status'] == 'passed' else GitHubStatus.ERROR,
                report_url
            )

            html_report = build_test_statistic(
                tool=self.TOOL,
                pack=self.Parameters.project_build_context,
                data=report_data,
                report_url=report_url or '{} [Отсутствует] '.format(task_url),
                task_url=task_url
            )

            issue.comments.create(text=html_report)

            logging.debug('Statistic has been sent to {}'.format(self.Parameters.testpalm_run_issue))

    def _call_test(self):
        rich_check_call(
            ["make", "autotest"],
            task=self, alias="test", cwd=self.app_src_path,
        )

    def _call_screens_update(self):
        rich_check_call(
            ["make", "autotest_update"],
            task=self, alias="screens_update", cwd=self.app_src_path,
        )

    def _call_allure(self):
        rich_check_call(
            ["make", "allure"],
            task=self, alias="allure", cwd=self.app_src_path,
        )

    def _call_testpalm(self):
        project = os.environ['hermione-allure_testpalm_project']
        token = os.environ['hermione-allure_testpalm_token']

        # todo: may be better to script inside app repo?
        rich_check_call(
            [
                "node_modules/.bin/ginny",
                "testpalm",
                "--project={}".format(project),
                "--token={}".format(token),
                "--run={}".format(self.testpalm_run_path),
                "import_run"
            ],
            task=self,
            alias="testpalm",
            cwd=self.app_src_path,
            check_return_code=False
        )

    def _change_gh_status(self, status, report_url=None):
        return change_status(
            owner=self.Parameters.app_owner,
            repo=self.Parameters.app_repo,
            context=self.github_context,
            sha=self.Parameters.app_status_check_commit if self.Parameters.app_status_check_commit else self.app_commit,
            url=report_url or get_task_link(self.id),
            state=status,
            description=GitHubStatusDescription[status]
        )

    def _get_commit_sha(self):
        return rich_get_output(
            ["git", "rev-parse", "HEAD"],
            task=self, alias="get_commit", cwd=self.app_src_path
        ).strip()

    def _upload_metrics(self):
        if not self.Parameters.upload_metrics:
            logging.debug("Metrics was not uploaded: upload_metrics is false")
            return

        if not self.at_report_resource:
            logging.debug("Metrics was not uploaded: resource with at report wasn't found")
            return

        repo = "{app_owner}/{app_repo}"\
            .format(app_owner=self.Parameters.app_owner, app_repo=self.Parameters.app_repo)\
            .lower()

        is_retry = bool(self.Parameters.broken_tests_resource_id)
        cases_stats = collect_cases_stats(self.at_report_path,
                                          self.Parameters.testpalm_run_issue,
                                          repo,
                                          self.Parameters.app_branch,
                                          self.Parameters.project_build_context,
                                          self.id,
                                          is_retry)

        logging.debug('Shared common info: {}'.format(cases_stats))
        stats_resource = self._create_metrics_resource(cases_stats)

        cases_errors = collect_cases_errors(self.at_report_path,
                                            self.Parameters.testpalm_run_issue,
                                            repo,
                                            self.Parameters.app_branch,
                                            self.Parameters.project_build_context,
                                            self.id)
        logging.debug('Shared error info: {}'.format(cases_errors))
        errors_resource = self._create_errors_resource(cases_errors)

        if not self.Parameters.send_metrics:
            logging.debug("Metrics and errors was not sent: send_metrics is false")
            return

        params = {
            'stats_resource': stats_resource,
            'errors_resource': errors_resource,
        }
        params.update(select_metrics_parameters(self.Parameters.metrics_params))
        self.exec_send_metrics_task(params)


    def exec_send_metrics_task(self, params):
        task = self.server.task({
            "type": "MARKET_AUTOTESTS_HERMIONE_SEND_METRICS",
            "description": "dummy",
            "owner": self.owner,
            "notifications": self.server.task[self.id].read()["notifications"],
            "priority": {
                "class": self.Parameters.priority.cls,
                "subclass": self.Parameters.priority.scls,
            },
            "children": True,
            "custom_fields": [{"name": k, "value": v} for k, v in params.items()],
        })
        task_id = task["id"]
        self.server.batch.tasks.start.update([task_id])

    def on_enqueue(self):
        self._prepare_browser_semaphore()
        setup_container(self)

    def on_failure(self, prev_status):
        self._change_gh_status(GitHubStatus.FAILURE)

    def on_break(self, prev_status, status):
        self._change_gh_status(GitHubStatus.FAILURE)

    def on_prepare(self):
        with MetatronEnv(self, nodejs_version=self.Parameters.node_version), GitRetryWrapper():
            logging.debug('Start task prepare')
            self._prepare_dirs()
            self._prepare_app_resource()
            self._prepare_broken_tests_resource()

            # NB: depends on app resource already unpacked
            self._change_gh_status(GitHubStatus.PENDING)

    def on_execute(self):
        # NB: MetatronEnv clears env on exit
        with MetatronEnv(self, nodejs_version=self.Parameters.node_version), GitRetryWrapper():
            logging.debug('Start task execute')
            self._prepare_ginny_user_config()
            self._prepare_env()
            self._prepare_testpalm_ids_resource()

            if self.Parameters.screenshots_test:
                self._prepare_screen_pack_resource()
                self._screens_test()
            else:
                self._test()
