# -*- coding: utf-8 -*-

import pipes
import os
import logging
from distutils.dir_util import copy_tree

from sandbox import sdk2
from sandbox.projects.market.front.helpers.github import clone_repo
from sandbox.projects.market.front.helpers.sandbox_helpers import rich_check_call
from sandbox.projects.market.resources import MARKET_FRONT_CMS_SCREENSHOTS
from sandbox.projects.sandbox_ci.utils import env

ACTUAL_SRC_DIR = "actual_src"
SCREENSHOOTER_LOGFILE = "screenshooter.log"
SCREENSHOT_DIR = "screenshots"
CONTAINER_RESOURCE_ID = 1300990332


class GenericParameters(sdk2.Task.Parameters):
    with sdk2.parameters.Group("Приложение") as app:
        github_owner = sdk2.parameters.String(
            "Организация на GitHub",
            default_value="market",
            required=True
        )

        github_repo = sdk2.parameters.String(
            "Имя репозитория",
            default_value="cms-screenshooter",
            required=True
        )

        head_branch = sdk2.parameters.String(
            "Актуальная ветка",
            default_value="master",
            required=True
        )

    with sdk2.parameters.Group("Настройки приложения") as app_settings:
        telegram_secret = sdk2.parameters.YavSecret(
            "Yav secret с токенами телеграма",
            description="Ожидается, что в секрете будут ключи: telegram_bot_token и cms_internal_user_id",
            default="sec-01dyhqjkm6295c2mxk2mbp63vs"  # cms-screenshooter-secrets
        )

        cms_page_types = sdk2.parameters.String(
            "Типы страниц в CMS",
            description="Параметры для ручки `getResourceList` в CMS. Параметры должны быть разделены "
                        "точкой с запятой, без пробелов",
            default_value="franchise;licensor;brand;brand_article;promo;promo_desktop;morda_context;product;"
                          "navnode_touch;catalog",
            required=True
        )

        chat_id = sdk2.parameters.String(
            "ID чата в телеграм для отправки уведомлений",
            required=True
        )

        manual_time = sdk2.parameters.Bool(
            "Задать период для проверки вручную?",
            description="По-умолчанию - предыдущий полный час от момента создания.\nТ.е. если таска была создана "
                        "в 17:53:02, то проверятся будет промежуток от 16:00:00 до 17:00:00",
            default=False,
            required=False
        )

        with manual_time.value[True]:
            check_interval_start = sdk2.parameters.String(
                "Начало периода",
                description="Нижняя граница времени публикации. Формат:\n%b %d, %Y %I:%M:%S %p\n"
                            "Пример: Jan 13, 2020 6:00:00 PM",
                required=True
            )
            check_interval_end = sdk2.parameters.String(
                "Окончание периода",
                description="Верхняя граница времени публикации. Формат:\n%b %d, %Y %I:%M:%S %p\n"
                            "Пример: Jan 13, 2020 7:00:00 PM",
                required=True
            )

    with sdk2.parameters.Group("Environment") as environ_block:
        environ = sdk2.parameters.Dict("Environment variables")


class MarketCmsScreenShooter(sdk2.Task):
    """
    Task to take screenshots of recently published CMS pages
    """

    class Requirements(sdk2.Task.Requirements):
        container_resource = CONTAINER_RESOURCE_ID

    class Parameters(GenericParameters):
        pass

    def _prepare_repo(self, branch, app_src_path):
        clone_repo(
            pipes.quote(self.Parameters.github_owner),
            pipes.quote(self.Parameters.github_repo),
            pipes.quote(branch),
            app_src_path
        )

    def _install_req(self, app_src_path):
        rich_check_call(
            ["pip3", "install", "--index-url", "https://pypi.yandex-team.ru/simple/", "-r", "requirements.txt"],
            task=self,
            alias="install req",
            cwd=app_src_path
        )

    def _call_screenshooter_script(self, app_src_path):
        rich_check_call(
            ["python3", "__init__.py"],
            task=self,
            alias="ScreenShooter",
            cwd=app_src_path
        )

    def _set_environment(self):
        custom_env = {"LC_TIME": "en_US.UTF-8"}

        if self.Parameters.manual_time:
            custom_env["CHECK_INTERVAL_END"] = self.Parameters.check_interval_end
            custom_env["CHECK_INTERVAL_START"] = self.Parameters.check_interval_start
        else:
            from datetime import timedelta
            import pytz
            local_timezone = pytz.timezone("Europe/Moscow")

            datetime_end = self.created.astimezone(tz=local_timezone)
            datetime_start = datetime_end - timedelta(hours=1)

            custom_env["CHECK_INTERVAL_END"] = datetime_end.strftime("%b %d, %Y %I:00:00 %p")
            custom_env["CHECK_INTERVAL_START"] = datetime_start.strftime("%b %d, %Y %I:00:00 %p")

        logging.info('Date. Checking published pages: from {} to {}'.format(
            custom_env["CHECK_INTERVAL_START"], custom_env["CHECK_INTERVAL_END"]
        ))

        secret = self.Parameters.telegram_secret
        custom_env["TELEGRAM_BOT_TOKEN"] = secret.data()["telegram_bot_token"]
        custom_env["TELEGRAM_CHANNEL_ID"] = self.Parameters.chat_id
        custom_env["CMS_INTERNAL_USER_ID"] = secret.data()["cms_internal_user_id"]

        custom_env["SCREENSHOT_DIR"] = SCREENSHOT_DIR
        custom_env["SC_LOGFILE"] = SCREENSHOOTER_LOGFILE
        custom_env["PAGE_TYPES"] = self.Parameters.cms_page_types

        env.export(custom_env)
        env.export(self.Parameters.environ)

    def _move_logs_and_pictures(self, app_src_path):
        """
        Saves app logs and screenshots in log resource
        """
        old_logfile = os.path.join(app_src_path, os.environ.get("SC_LOGFILE"))
        new_logfile = os.path.join(str(self.log_path()), os.environ.get("SC_LOGFILE"))
        if os.path.exists(old_logfile):
            os.rename(old_logfile, new_logfile)

        screenshot_dir = os.path.join(app_src_path, os.environ.get("SCREENSHOT_DIR"))
        if os.path.exists(screenshot_dir):
            resource = MARKET_FRONT_CMS_SCREENSHOTS(
                self, "Screenshots captured by CmsScreenShooter", screenshot_dir
            )
            resource_data = sdk2.ResourceData(resource)
            resource_data.ready()

    def on_execute(self):
        super(MarketCmsScreenShooter, self).on_execute()

        self._set_environment()

        root_dir = str(self.path())
        actual_version_dir = os.path.join(root_dir, ACTUAL_SRC_DIR)

        self._prepare_repo(self.Parameters.head_branch, actual_version_dir)

        self._install_req(actual_version_dir)

        self._call_screenshooter_script(actual_version_dir)

        self._move_logs_and_pictures(actual_version_dir)
