# -*- coding: utf-8 -*-

import json
import logging
import os
import pipes
import re
import tempfile

import requests

from sandbox import sdk2
from sandbox.common.types import misc as ctm
from sandbox.common.types import task as ctt
from sandbox.common.utils import get_task_link
from sandbox.projects.common import task_env
from sandbox.projects.market.front.helpers.MetatronEnv import MetatronEnv
from sandbox.projects.market.front.helpers.github import clone_repo
from sandbox.projects.market.front.helpers.staff import get_telegram_login_by_staff_login
from sandbox.projects.market.front.helpers.tsum import send_long_message_list
from sandbox.projects.market.front.helpers.ubuntu import create_ubuntu_selector, setup_container
from sandbox.projects.sandbox_ci.utils.request import send_request
from sandbox.sandboxsdk.environments import PipEnvironment
from .custom_config import CUSTOM_CONFIG

DISK_SPACE = 1 * 1024  # 1 Gb
ABT_API_URL = "https://ab.yandex-team.ru/api"
API_LIMIT_PER_PAGE = 50
DESKTOP_FLAGS_PATH = os.path.join("market", "platform.desktop", "configs", "flags_allowed.d")
TOUCH_FLAGS_PATH = os.path.join("market", "platform.touch", "configs", "flags_allowed.d")
EXCLUDED_PLATFORMS = [
    "AFFILIATE"
]
ISSUE_KEY_REGEX = "[a-zA-Z]+-\\d+"
ERROR_QA = "Завершились с ошибкой:"
SINGLE_EXP_TEMPLATE = """{flag}
{issue_link}
"""
MSG_SPLITTER = "\n----\n"


class MarketFrontActiveExpCallQA(sdk2.Task):
    """
    Запрашивает из ABT активные экспы, выбирает из них фронтовые и призывает QA в чат для проверки экспов
    """
    root_dir = None
    APP_REPO_PATH = None

    class Parameters(sdk2.Task.Parameters):
        # common parameters
        kill_timeout = 15 * 60  # 15 min

        with sdk2.parameters.Group("Конфиги") as configs:
            notification_chat_id = sdk2.parameters.String(
                "ID чата в TG",
                default="-1001410453054",  # Чат "Оповещения об активных экспериментах"
                description="Чат для отправки уведомлений об необходимости проверки. Оставить пустым если уведомление не требуется."
            )
            current_release_version = sdk2.parameters.String(
                "Текущая фикс-версия",
                required=True,
                description="Фикс-версия текущего релиза. По ней будет искаться релизный тикет и проверяться наличие связи с релизом"
            )
            previous_release_version = sdk2.parameters.String(
                "Предыдущая фикс-версия",
                description="Фикс-версия предыдущего релиза. Не используется"
            )

            ubuntu_version = create_ubuntu_selector()

        with sdk2.parameters.Group("GitHub репозиторий проекта") as github_repo_block:
            app_owner = sdk2.parameters.String(
                "GitHub owner",
                description="Логин владельца репозитория или название организации",
                default="MARKET",
                required=True
            )
            app_repo = sdk2.parameters.String(
                "Репозиторий",
                default="marketfront",
                required=True
            )
            app_branch = sdk2.parameters.String(
                "Ветка",
                default="master",
                required=True
            )

    class Requirements(task_env.TinyRequirements):
        dns = ctm.DnsType.DNS64
        disk_space = DISK_SPACE
        environments = [
            PipEnvironment("yandex_tracker_client", version="1.3", custom_parameters=["--upgrade-strategy only-if-needed"]),
            PipEnvironment("startrek_client", version="2.3.0", custom_parameters=["--upgrade-strategy only-if-needed"])
        ]

    def on_enqueue(self):
        super(MarketFrontActiveExpCallQA, self).on_enqueue()
        setup_container(self)

    def on_prepare(self):
        self.root_dir = tempfile.mkdtemp()
        self.APP_REPO_PATH = os.path.join(str(self.root_dir), str(self.Parameters.app_repo))

    def on_execute(self):
        if not self._call_is_needed():
            logging.info("Таска запускается в релизе {} повторно, призыв не нужен".format(self.Parameters.current_release_version))
            return
        active_testid_list = self._get_active_experiments_list()
        front_expflag_dict = self._filter_only_front_expflags(active_testid_list)
        self._clone_repo()
        active_flags = self._match_expflags_with_config(front_expflag_dict)
        flags_grouped_by_qa = _group_flags_by_qa(active_flags)
        active_flags = self._update_already_checked_flags(active_flags)
        self._notify_qa(active_flags, flags_grouped_by_qa)

    def _call_is_needed(self):
        if len(str(self.Parameters.current_release_version)) == 0:
            logging.info("Передали пустую версию текущего релиза")
            return True

        previous_task_for_this_release = sdk2.Task.find(
            task_type=MarketFrontActiveExpCallQA,
            status=ctt.Status.SUCCESS,
            input_parameters={
                "current_release_version": str(self.Parameters.current_release_version)
            }
        ).limit(1)

        # Если таска в этом релизе ещё не запускалась успешно - нужно её запустить.
        return previous_task_for_this_release.count == 0


    def _get_active_experiments_list(self):
        logging.info("Пытаемся получить список экспермиентов из ABT")
        with MetatronEnv(self):
            token = os.environ["ABT_API_TOKEN"]
            url = ABT_API_URL + "/testid"
            params = {
                "form": "full",
                "format": "json",
                "limit": API_LIMIT_PER_PAGE,
                "queue_id": 7,  # MARKET
                "skip": 0,
                "task__state": "RUNNING",
                "grep": "flags",  # берём только testid-ы в которых есть слово flags (фронтовые)
            }
            headers = {
                "Authorization": "OAuth {}".format(token),
            }
            return get_paged_abt_api_results(url, params, headers)

    def _filter_only_front_expflags(self, active_testid_list):
        """
        :param active_testid_list:
        :return: {"flag_1": {"platforms": ("MARKET", "DESKTOP")}, {"flag_2": {"platforms": ("TOUCH")}}, ...}
        """
        logging.info("Собираем активные фронтовые флаги в кучку.")
        active_exp_flags = {}
        for testid in active_testid_list:
            logging.debug("Читаем testid {}".format(testid["testid"]))
            try:
                params = json.loads(testid["params"])[0]
                context = params["CONTEXT"]
                for platform, config in context.items():
                    if platform in EXCLUDED_PLATFORMS:
                        logging.info("Пропускаем платформу {} для testid {}".format(platform, testid["testid"]))
                        continue

                    # Обложились обработкой граничных случаев
                    if "flags" not in config:
                        logging.info("Не найден flags в конфиге {} для testid {}".format(config, testid["testid"]))
                        continue

                    flags = config["flags"]
                    if not isinstance(flags, list):
                        logging.error("Flags в конфиге {} для testid {} не является списком".format(config, testid["testid"]))
                        continue

                    for flag_with_value in flags:  # Например "dsk_new-search=test-default" или "dsk_foo_foo"
                        flag = flag_with_value.split("=")[0]
                        if flag not in active_exp_flags:
                            active_exp_flags[flag] = {"platforms": set()}

                        # И когда всё хорошо - положили новый ключик в мапу
                        active_exp_flags[flag]["platforms"].add(platform)
            except ValueError as err:
                logging.error(err)
                logging.info("Пытались распарсить testid: {}".format(testid))
            except KeyError as err:
                logging.error(err)
                logging.info("Пытались распарсить testid: {}".format(testid))

        logging.info("Получили список активных флагов.")
        logging.debug("Активные флаги: {}".format(active_exp_flags))
        return active_exp_flags

    def _clone_repo(self):
        with MetatronEnv(self):
            clone_repo(
                pipes.quote(self.Parameters.app_owner),
                pipes.quote(self.Parameters.app_repo),
                pipes.quote(self.Parameters.app_branch),
                self.APP_REPO_PATH
            )

    def _match_expflags_with_config(self, front_expflag_dict):
        logging.info("Загружаем конфиги указанные в репозитории")
        with MetatronEnv(self):
            from startrek_client import Startrek, exceptions as st_exceptions
            oauth_token = os.environ["ST_OAUTH_TOKEN"]
            st = Startrek(useragent="robot-metatron", token=oauth_token)

            full_data_expflags = {}
            for active_flag in front_expflag_dict:
                logging.info("Обогащаем флаг {}".format(active_flag))
                full_data_expflags[active_flag] = front_expflag_dict[active_flag]
                flag_config = None
                dsk_path = os.path.join(self.APP_REPO_PATH, DESKTOP_FLAGS_PATH, "{}.json".format(active_flag))
                touch_path = os.path.join(self.APP_REPO_PATH, TOUCH_FLAGS_PATH, "{}.json".format(active_flag))

                # Получаем конфиг из репозитория
                try:
                    if os.path.exists(dsk_path):
                        with open(dsk_path) as file:
                            flag_config = json.load(file)
                    if os.path.exists(touch_path):
                        with open(touch_path) as file:
                            flag_config = json.load(file)
                except ValueError as err:
                    logging.error(err)
                    full_data_expflags[active_flag]["error"] = "Не удалось распарсить json-конфиг в репозитории."
                    continue

                if flag_config is None:
                    logging.debug("Не найдены файлы:\ndsk: {}\ntouch: {}".format(dsk_path, touch_path))
                    full_data_expflags[active_flag]["error"] = "Отсутствует конфиг для флага в репозитории."
                    continue

                description = None
                issue_key = None
                qa = None
                # Оверрайдим конфиг кастомными значениями. Это нужно делать до парсинга, т.к. в репозитории фронта
                # могут быть указаны тикеты, к которым у робота нет доступа.
                if active_flag in CUSTOM_CONFIG:
                    override = CUSTOM_CONFIG[active_flag]
                    if "description" in override:
                        description = override["description"]
                    if "issue_key" in override:
                        issue_key = override["issue_key"]
                    if "qa" in override:
                        qa = override["qa"]

                try:
                    if not description:
                        description = flag_config["description"]

                    if not issue_key:
                        issue_key_match = re.search(ISSUE_KEY_REGEX, description)

                        if not issue_key_match:
                            full_data_expflags[active_flag]["error"] = "Не удалось найти номер тикета в описаниии флага"
                            continue

                        if "issue_key" not in full_data_expflags[active_flag]:
                            issue_key = issue_key_match.group(0)

                    description = re.sub(ISSUE_KEY_REGEX + ":? ?", "", description)

                    if not qa:
                        qa = find_qa_by_issue_key(st, oauth_token, issue_key)

                    full_data_expflags[active_flag]["description"] = description
                    full_data_expflags[active_flag]["qa"] = qa
                    full_data_expflags[active_flag]["issue_key"] = issue_key
                except KeyError as err:
                    logging.error(err)
                    full_data_expflags[active_flag]["error"] = "Не удалось получить какое-то из значений конфига из репозитория"
                except st_exceptions.Forbidden as err:
                    logging.error(err)
                    full_data_expflags[active_flag]["error"] = "Нет доступа к тикету https://st.yandex-team.ru/{}".format(issue_key)

        logging.info("Закончили обогащать экспфлаги")
        logging.debug(full_data_expflags)
        return full_data_expflags

    def _update_already_checked_flags(self, front_expflag_dict):
        if len(str(self.Parameters.current_release_version)) == 0:
            logging.info("Передали пустую версию текущего релиза")
            return front_expflag_dict

        with MetatronEnv(self):
            from startrek_client import Startrek
            oauth_token = os.environ["ST_OAUTH_TOKEN"]
            st = Startrek(useragent="robot-metatron", token=oauth_token)
            try:
                release_issue = st.issues.find("Queue: MARKETFRONT and \"Fix Version\": \"{}\" and Type: Release".format(self.Parameters.current_release_version))[0]
                release_issue_linked_keys = set(link.object.key for link in release_issue.links.get_all())
                logging.info("Список связанных с релизом тикетов:\n{}".format(release_issue_linked_keys))
            except IndexError:
                logging.error("Не удалось найти релиз с фикс-версией {}".format(self.Parameters.current_release_version))
                return front_expflag_dict

            for active_flag_name in front_expflag_dict:
                if "issue_key" not in front_expflag_dict[active_flag_name]:
                    logging.debug("Нет issue_key в флаге {}".format(front_expflag_dict[active_flag_name]))
                    continue

                # замьючиваем призыв если у релизного тикета уже есть связь
                if front_expflag_dict[active_flag_name]["issue_key"] in release_issue_linked_keys:
                    front_expflag_dict[active_flag_name]["mute"] = True
        logging.debug("После мьюта проверенных экспов:\n{}".format(front_expflag_dict))
        return front_expflag_dict

    def _notify_qa(self, active_expflags, flags_grouped_by_qa):
        if len(str(self.Parameters.notification_chat_id)) == 0:
            logging.info("Уведомление в чат не требуется. Не отсылаем сообщение.")
            return
        chat_id = str(self.Parameters.notification_chat_id)

        logging.info("Оповещаем QA об экспериментах через ЦУМ-бота")
        with MetatronEnv(self):
            # Используем любой валидный OAuth токен для авторизации
            token = os.environ["ST_OAUTH_TOKEN"]
            messages = []

            for qa, qa_flags in flags_grouped_by_qa.items():
                single_exp_messages = []
                for qa_flag_name in qa_flags:
                    if "mute" in active_expflags[qa_flag_name] and active_expflags[qa_flag_name]["mute"]:
                        # Уведомление не требуется
                        continue

                    if "error" in active_expflags[qa_flag_name]:
                        single_exp_messages.append("{}: {}\n".format(qa_flag_name, active_expflags[qa_flag_name]["error"]))
                        continue

                    try:
                        single_exp_messages.append(SINGLE_EXP_TEMPLATE.format(
                            flag=qa_flag_name,
                            description=active_expflags[qa_flag_name]["description"],
                            issue_link="https://st.yandex-team.ru/{}".format(active_expflags[qa_flag_name]["issue_key"]),
                            platforms=", ".join(active_expflags[qa_flag_name]["platforms"]),
                        ))
                    except KeyError as err:
                        logging.error(err)

                if len(single_exp_messages) > 0:
                    messages.append("{}\n\n{}".format(qa, "".join(single_exp_messages)))

            if len(active_expflags) == 0 or len(messages) == 0:
                messages.append("Не удалось получить список активных флагов. См. логи джобы.\n{}".format(get_task_link(self.id)))

            messages.sort()
            messages.insert(0, "Релиз {}\n\n".format(self.Parameters.current_release_version))
            send_long_message_list(token, chat_id, messages, MSG_SPLITTER, concat_messages=False)



def find_qa_by_issue_key(st, oauth_token, issue_key):
    issue = st.issues[issue_key]
    qa = issue["qaEngineer"]
    if qa is None:
        return "Не указан QA"

    # Токен Startrek даёт так же доступ к стаффу
    qa_tg = get_telegram_login_by_staff_login(oauth_token, qa.login)
    if qa_tg is None:
        qa_tg = "https://staff.yandex-team.ru/{}".format(qa.login)
    else:
        qa_tg = "@{}".format(qa_tg)

    return qa_tg


def get_paged_abt_api_results(url, params, headers):
    logging.info("Запрашиваем первую страницу из ABT.")
    results = []
    should_request_next_page = True

    if "limit" not in params:
        params["limit"] = API_LIMIT_PER_PAGE

    try:
        this_query_limit = int(params["limit"])
        while should_request_next_page:
            url_with_get_params = "{}?{}".format(url, params_to_str(params))
            should_request_next_page = False

            logging.info("Запрашиваем данные из ABT по урлу {}".format(url_with_get_params))
            abt_response = send_request(
                method="get",
                url=url_with_get_params,
                headers=headers,
                max_retries=2
            )  # type: requests.models.Response

            if abt_response.status_code != 200:
                logging.error("Ответ ABT: {}, {}".format(abt_response.status_code, abt_response.text))
                break

            abt_list = abt_response.json()
            results += abt_list

            if len(abt_list) == this_query_limit:
                logging.info("Получили {} записей из ABT. Запрашиваем следующую страницу.".format(this_query_limit))
                should_request_next_page = True
                if "skip" in params:
                    params["skip"] = int(params["skip"]) + this_query_limit
                else:
                    params["skip"] = this_query_limit

    except (ValueError, KeyError) as err:
        logging.error(err)
    finally:
        logging.debug("Получили список:\n{}".format(str(results)))
        return results


def _group_flags_by_qa(exp_flags):
    logging.info("Группируем флаги по QA")
    qa_to_flags_dict = {}
    for flag in exp_flags:
        # Если парсинг флага завершился с ошибкой и нет QA - кладём в отдельный блок
        if "error" in exp_flags[flag]:
            if ERROR_QA not in qa_to_flags_dict:
                qa_to_flags_dict[ERROR_QA] = []
            qa_to_flags_dict[ERROR_QA].append(flag)
            continue

        # Map qa to flag
        try:
            qa = exp_flags[flag]["qa"]
            if qa not in qa_to_flags_dict:
                qa_to_flags_dict[qa] = []

            qa_to_flags_dict[qa].append(flag)
        except KeyError as err:
            logging.error(err)

    logging.info("Сгруппировали флаги по QA")
    logging.debug(qa_to_flags_dict)
    return qa_to_flags_dict


def params_to_str(
    params
):
    param_list = ["{}={}".format(key, value) for key, value in params.items()]
    return "&".join(param_list)
