# coding=utf-8

import os
import tempfile
import pipes
import json
import logging

from sandbox import sdk2
import sandbox.common.types.misc as ctm
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.common.utils import get_task_link
from sandbox.common.errors import TaskError

from sandbox.projects.market.front.helpers.sandbox_helpers import rich_check_call, \
    get_resource_http_proxy_link
from sandbox.projects.market.front.helpers.node import create_node_selector
from sandbox.projects.market.front.helpers.ubuntu import create_ubuntu_selector, setup_container
from sandbox.projects.market.front.helpers.MetatronEnv import MetatronEnv
from sandbox.projects.market.front.helpers.startrack import hidden_id
from sandbox.projects.market.front.helpers.github import GitHubStatus, change_status, \
    clone_repo, GitHubStatusDescription
from sandbox.projects.market.front.helpers.ci import extract_ticket_id


APP_REPO_OWNER = "market"
ACTUAL_SRC_DIR = "actual_src"
CHECK_SRC_DIR = "check_src"

GITHUB_CONTEXT = "API Compatible"

COMPATIBLE_TYPE_CHECKER_CONFIG_PATH = "configs/compatible-type-checker.json"
REPORT_FILE_NAME = "compatible-type-checker.json"

FILE_STATUS = {
    "NOT_CHANGED": "Не изменился",
    "CHANGED": "Изменился",
    "ADDED": "Добавился",
    "REMOVED": "Удалился",
    "INVALID": "Некорректный",
}

class GenericParameters(sdk2.Task.Parameters):
    with sdk2.parameters.Group("Приложение") as app:
        github_owner = sdk2.parameters.String(
            "Организация на GitHub",
            default_value=APP_REPO_OWNER,
            required=True
        )

        github_repo = sdk2.parameters.String(
            "Имя репозитория",
            required=True
        )

        app_src_dir = sdk2.parameters.String(
            "Кастомный путь корня приложения внутри репозитория"
        )

        head_branch = sdk2.parameters.String(
            "Актуальная ветка",
            default_value="master",
            required=True
        )

        check_branch = sdk2.parameters.String(
            "Проверяемая ветка",
            required=True
        )

        pr_title = sdk2.parameters.String(
            "Заголовок PR"
        )

        ubuntu_version = create_ubuntu_selector()
        node_version = create_node_selector()

    with sdk2.parameters.Group("GitHub") as Github:
        send_hook = sdk2.parameters.Bool(
            "Посылать отчет",
            default_value=False,
            required=True
        )

        with send_hook.value[True]:
            github_sha = sdk2.parameters.String(
                "Хеш коммита",
                default_value="",
                required=True
            )

            github_context = sdk2.parameters.String(
                "Контекст",
                default_value=GITHUB_CONTEXT,
                required=True
            )

    with sdk2.parameters.Group("Отладка") as Dev:
        debug = sdk2.parameters.Bool(
            "Я программист",
            default=False,
            required=True
        )

        with debug.value[True]:
            checker_version = sdk2.parameters.String(
                "Compatible Type Checker version",
                default_value="latest",
                required=True,
            )

            project_conf = sdk2.parameters.String(
                "Compatible Type Checker config path",
                default_value=COMPATIBLE_TYPE_CHECKER_CONFIG_PATH,
                required=True
            )

            strict = sdk2.parameters.Bool(
                "Strict mode",
                default_value=True,
            )

            check_version_config = sdk2.parameters.Bool(
                "Use check version config",
                default_value=False,
            )

class MarketFrontApiCheck(sdk2.Task):
    """
    Task to check files for API compatible
    """

    class Parameters(GenericParameters):
        pass

    class Requirements(sdk2.Task.Requirements):
        dns = ctm.DnsType.DNS64
        environments = [
            PipEnvironment('yandex_tracker_client', version="1.3", custom_parameters=["--upgrade-strategy only-if-needed"]),
            PipEnvironment('startrek_client', version="2.3.0", custom_parameters=["--upgrade-strategy only-if-needed"])
        ]

    @property
    def _task_url(self):
        return get_task_link(self.id)

    @property
    def _report_url(self):
        return "{}/{}".format(get_resource_http_proxy_link(self.log_resource), REPORT_FILE_NAME)

    @property
    def _github_context(self):
        return "[Sandbox CI] {}".format(self.Parameters.github_context or GITHUB_CONTEXT)

    @staticmethod
    def _get_report(report_path):
        with open(report_path, "r") as file:
            return json.load(file)

    def on_enqueue(self):
        super(MarketFrontApiCheck, self).on_enqueue()
        setup_container(self)

        self._change_gh_status(GitHubStatus.PENDING)

    def on_execute(self):
        super(MarketFrontApiCheck, self).on_execute()

        with MetatronEnv(self, nodejs_version=self.Parameters.node_version):
            root_dir = tempfile.mkdtemp()
            report_path = os.path.join(str(self.log_path()), REPORT_FILE_NAME)
            actual_root_dir = os.path.join(root_dir, ACTUAL_SRC_DIR)
            check_root_dir = os.path.join(root_dir, CHECK_SRC_DIR)
            actual_version_dir = actual_root_dir
            check_version_dir = check_root_dir

            if self.Parameters.app_src_dir:
                actual_version_dir = os.path.join(actual_root_dir, self.Parameters.app_src_dir)
                check_version_dir = os.path.join(check_root_dir, self.Parameters.app_src_dir)

            self._prepare_checker(root_dir)

            self._prepare_repo(self.Parameters.head_branch, actual_root_dir, actual_version_dir)
            self._prepare_repo(self.Parameters.check_branch, check_root_dir, check_version_dir)

            self._call_check(root_dir, report_path, actual_version_dir, check_version_dir)

            self._report = MarketFrontApiCheck._get_report(report_path)

            self._send_report()

    def on_success(self, prev_status):
        super(MarketFrontApiCheck, self).on_success(prev_status)

        if self._report["compatible"] is True:
            self._change_gh_status(GitHubStatus.SUCCESS, self._report_url)
        else:
            self._change_gh_status(GitHubStatus.ERROR, self._report_url)

            if self.Parameters.strict:
                raise TaskError("The version being tested is not compatible")

    def on_failure(self, prev_status):
        super(MarketFrontApiCheck, self).on_failure(prev_status)

        self._change_gh_status(GitHubStatus.FAILURE)

    def on_break(self, prev_status, status):
        super(MarketFrontApiCheck, self).on_break(prev_status, status)

        self._change_gh_status(GitHubStatus.FAILURE)

    def _change_gh_status(self, status, report_url=None):
        if not self.Parameters.send_hook:
            return

        return change_status(
            owner=self.Parameters.github_owner,
            repo=self.Parameters.github_repo,
            context=self._github_context,
            sha=self.Parameters.github_sha,
            url=report_url or self._task_url,
            state=status,
            description=GitHubStatusDescription[status]
        )

    def _prepare_checker(self, root_dir):
        rich_check_call(
            ["npm", "i",
             "--registry=http://npm.yandex-team.ru",
             "@yandex-market/compatible-type-checker@{}".format(self.Parameters.checker_version)],
            task=self,
            alias="install-checker",
            cwd=root_dir
        )

    def _prepare_repo(self, branch, repo_root, app_src_path):
        clone_repo(
            pipes.quote(self.Parameters.github_owner),
            pipes.quote(self.Parameters.github_repo),
            pipes.quote(branch),
            repo_root
        )

        rich_check_call(
            ["npm", "run", "bootstrap"],
            task=self,
            alias="bootstrap",
            cwd=app_src_path
        )

    def _call_check(self, root_dir, report_path, actual_version, check_version):
        config_path = check_version if self.Parameters.check_version_config else actual_version

        rich_check_call(
            [
                "./node_modules/.bin/compatible-type-checker",
                "-c", os.path.join(config_path, self.Parameters.project_conf),
                "-o", report_path,
                "--av", actual_version,
                "--cv", check_version,
            ],
            task=self,
            alias="compatible-type-checker",
            cwd=root_dir
        )

        if not os.path.isfile(report_path):
            raise TaskError("Report not found from path {}".format(report_path))

    def _send_report(self):
        ticket_id = extract_ticket_id(self.Parameters.pr_title)

        if ticket_id is None:
            logging.debug("The ticket number is not found")
            return

        oauth_token = sdk2.Vault.data("robot-metatron-st-token")

        from startrek_client import Startrek

        st = Startrek(useragent="robot-metatron", token=oauth_token)

        comment = self._build_ticket_comment()

        issue = st.issues[ticket_id]
        issue.comments.create(text=comment)

        logging.debug("Report was sent to {}".format(ticket_id))

    def _build_ticket_comment(self):
        is_success = self._report["compatible"] is True

        caption = "==== {}\n".format(self.Parameters.github_context or GITHUB_CONTEXT)
        caption += "===== Результат проверки совместимости API: {}"
        caption_text = "!!(green)СОВМЕСТИМО!!!" if is_success else "!!(red)НЕ СОВМЕСТИМО!!!"

        comment = caption.format(caption_text)
        comment += "\n\n- <{Отчет\n#|"
        comment += "**||Статус|Совместимый|Актуальный файл|Проверяемый файл|Изменения||**\n"

        for file in self._report["files"]:
            if file['status'] == 'NOT_CHANGED':
                continue

            validate_result = file["validateResult"]
            actual_file_info = file["actualFile"] or "Отсутствует"

            if validate_result and validate_result["actualFileError"]:
                actual_file_info += "\n<{{Ошибка парсинга\n%%{}%%}}>".format(validate_result["actualFileError"])

            check_file_info = file["checkFile"] or "Отсутствует"

            if validate_result and validate_result["checkFileError"]:
                check_file_info += "\n<{{Ошибка парсинга\n%%{}%%}}>".format(validate_result["checkFileError"])

            if validate_result and validate_result["diff"]:
                diff_info = "<{{JSON\n%%(json){}%%}}>".format(json.dumps(validate_result["diff"], indent=4))
            elif validate_result and validate_result["error"]:
                diff_info = "<{{Ошибка валидации\n%%{}%%}}>".format(validate_result["error"])
            else:
                diff_info = "Отсутствуют"

            comment += "||{}|{}|{}|{}|{}||".format(
                FILE_STATUS[file["status"]],
                "!!(green)Да!!" if file["compatible"] else "!!(red)Нет!!",
                actual_file_info,
                check_file_info,
                diff_info
            )

        comment += '|#}>'
        comment += "\n- (({} Таск.))".format(self._task_url)
        comment += '\n{}'.format(hidden_id('fapi-check-report'))

        return comment
