# -*- coding: utf-8 -*-

import json
import logging
import os
import shutil
import tempfile
from typing import Union, List, Set

from sandbox import sdk2
from sandbox.agentr.errors import ResourceNotAvailable
from sandbox.common.errors import InvalidResource
from sandbox.common.types import misc as ctm
from sandbox.common.types import task as ctt
from sandbox.projects.common import task_env
from sandbox.projects.market.front.MarketAutotestsHermione import MarketAutotestsHermione
from sandbox.projects.market.front.helpers.MetatronEnv import MetatronEnv
from sandbox.projects.market.front.helpers.sandbox_helpers import get_resource_http_proxy_link
from sandbox.projects.market.front.helpers.tsum import send_telegram_message_via_tsum_bot
from sandbox.projects.market.front.helpers.ubuntu import create_ubuntu_selector, setup_container
from sandbox.projects.market.resources import MARKET_AUTOTEST_REPORT, MARKET_FRONT_AUTOTEST_COMPARATOR_RESULT
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.sdk2.internal.common import Query

DISK_SPACE = 3 * 1024  # 3 Gb
SUITES_FILE_NAME = "suites.json"
SUITES_PATH = os.path.join("data", SUITES_FILE_NAME)
COMPARISON_RESULT_FILENAME = "result.json"
OK_STATUS = "OK"
WARN_STATUS = "WARN"
CRIT_STATUS = "CRIT"
NEW_FAILED = "new_failed"
OLD_FAILED = "old_failed"
BROKEN_TEST_STATUS_LIST = ["failed", "broken"]
DEFAULT_ST_CALL_TEXT_TEMPLATE = "Тикет оторвали от релиза, т.к. в нём не проходили автотесты или упало больше %d тестов"
DEFAULT_TG_NOTIFY_TEMPLATE = """От релиза {action} тикеты:
{feature_list}

Т.к. в них не проходили автотесты или упало больше 100 тестов.
"""


class MarketFrontAutotestComparator(sdk2.Task):
    """
    Сравнивает между собой прогоны автотестов в тикетах, попавших в релиз - и в предыдущем релизе.
    Результат складывает в файлик отчёта
    """
    root_dir = None
    feature_issue_keys = []
    test_types_parsed = []
    result = {}
    result_resource = None
    removed_feature_list = []

    class Parameters(sdk2.Task.Parameters):

        with sdk2.parameters.Group("Номера релизов") as release_versions:
            previous_release_version = sdk2.parameters.String(
                "Предыдущий",
                description="Фикс-версия последнего успешного релиза",
            )

            current_release_version = sdk2.parameters.String(
                "Текущий",
                description="Фикс-версия текущего релиза",
            )

        with sdk2.parameters.Group("Конфиги") as configs:
            st_queue = sdk2.parameters.String(
                "Очередь релизных тикетов",
                default="MARKETFRONT",
            )

            test_types = sdk2.parameters.String(
                "Типы тестов через запятую",
                description="Какие тесты сравниваем. Поле project_build_context в тасках автотестов."
            )

            crit_diff = sdk2.parameters.Integer(
                "Разница для сигнала",
                default=10,
                description="Какой дифф в количестве упавших считать критичным"
            )

            verbose = sdk2.parameters.Bool(
                "Писать отчёт в тикет",
                default=True,
                description="Отправляет в тикет ссылку на отчёт и форматированную версию отчёта"
            )

            notification_chat_id = sdk2.parameters.String(
                "ID чата в TG",
                default="-1001483860024",  # Чат "Релизы фронта Маркета"
                description="Чат для отправки уведомлений об отрываемых тикетах. Оставить пустым если уведомление не требуется."
            )

            enable_feature_removing = sdk2.parameters.Bool(
                "Включить отрыв",
                default=False,
                description="Отрывать тикеты без тестов или с большим числом падений от релиза"
            )

            ubuntu_version = create_ubuntu_selector()

    class Requirements(task_env.TinyRequirements):
        dns = ctm.DnsType.DNS64
        disk_space = DISK_SPACE
        environments = [
            PipEnvironment("yandex_tracker_client", version="1.3", custom_parameters=["--upgrade-strategy only-if-needed"]),
            PipEnvironment("startrek_client", version="2.3.0", custom_parameters=["--upgrade-strategy only-if-needed"])
        ]

    def on_enqueue(self):
        super(MarketFrontAutotestComparator, self).on_enqueue()
        setup_container(self)

    def on_prepare(self):
        self.root_dir = tempfile.mkdtemp()

    def on_execute(self):
        self._find_current_release_feature_tickets()
        self._load_comparator_config()
        self._copy_previous_release_feature_results()
        self._compare_every_feature_with_previous_release()
        self._create_a_resource_with_comaprison_result()
        self._remove_bad_tickets_from_release()
        self._send_telegram_report_through_api()
        self._send_startrek_report_to_release_ticket()

    def _find_current_release_feature_tickets(self):
        with MetatronEnv(self):
            from startrek_client import Startrek
            oauth_token = sdk2.Vault.data("robot-metatron-st-token")
            st = Startrek(useragent="robot-metatron", token=oauth_token)

            st_search_query = "Type: !\"Релиз\" \"Fix Version\": \"%s\" Queue: \"%s\"" % (self.Parameters.current_release_version, self.Parameters.st_queue)
            logging.info("Ищем тикеты в ST. Запрос: %s" % st_search_query)
            release_feature_issues = st.issues.find(st_search_query)

            self.feature_issue_keys = [issue.key for issue in release_feature_issues]
            logging.info("Нашли %d тикетов в релизе" % len(self.feature_issue_keys))

    def _load_comparator_config(self):
        self.test_types_parsed = [test_type.strip() for test_type in self.Parameters.test_types.split(",")]
        for test_type in self.test_types_parsed:
            path = os.path.join(self.root_dir, test_type)
            os.mkdir(path)
        logging.info("Будем смотреть на %d вида(ов) тестов" % len(self.test_types_parsed))

    def _copy_previous_release_feature_results(self):
        for test_type in self.test_types_parsed:
            autotest_tasks = sdk2.Task.find(
                task_type=MarketAutotestsHermione,
                status=(ctt.Status.SUCCESS, ctt.Status.FAILURE),
                input_parameters={
                    "project_build_context": test_type,
                    "testpalm_run_version": str(self.Parameters.previous_release_version)
                },
                children=True
            ).order(-sdk2.Task.id).limit(20)

            last_test_task = None
            for task in autotest_tasks:
                if task.Parameters.broken_tests_resource_id:
                    # Нашли таску перепрогона, она нам и нужна. Дальше не ищем, таски отсортированы.
                    last_test_task = task
                    break

            if not last_test_task:
                logging.info("Не нашли тесты %s в релизе %s" % (test_type, self.Parameters.previous_release_version))
                continue
            logging.info("В прошлом релизе выбрали таску %d для %s" % (last_test_task.id, test_type))

            report = sdk2.Resource.find(
                resource_type=MARKET_AUTOTEST_REPORT,
                task=last_test_task
            ).first()

            if not report:
                logging.info("Не нашли ресурс автотестов %s в релизе %s" % (test_type, self.Parameters.previous_release_version))
                continue

            try:
                resource_data = sdk2.ResourceData(report)
            except (InvalidResource, ResourceNotAvailable) as ex:
                logging.info("Ресурс автотестов %s в релизе %s не удалось получить" % (test_type, self.Parameters.previous_release_version))
                logging.info(str(ex))
                continue
            result_json_path = os.path.join(str(resource_data.path), SUITES_PATH)

            try:
                with open(result_json_path) as result_json:
                    result_data = json.load(result_json)

                if isinstance(result_data["children"], list):
                    dest = os.path.join(self.root_dir, test_type)
                    filesize = os.path.getsize(result_json_path) / 1024.0
                    logging.info("Копируем %s в %s . Размер %f KB." % (result_json_path, dest, filesize))

                    shutil.copy(result_json_path, dest)
                    logging.info("Ресурс %d успешно скопировали" % report.id)
            # Не копируем отчёт если его нет, его не удалось его распарсить или если он пустой
            except (ValueError, KeyError, IOError) as e:
                logging.info("Не удалось скопировать отчёт из ресурса %d. Ошибка: %s" % (report.id, str(e)))

    def _compare_every_feature_with_previous_release(self):
        for feature_key in self.feature_issue_keys:
            self.result[feature_key] = {}

        for test_type in self.test_types_parsed:
            logging.info("Проверяем %s" % test_type)

            old_report_path = os.path.join(self.root_dir, test_type, SUITES_FILE_NAME)
            try:
                with open(old_report_path) as f:
                    old_report_data = json.load(f)
            except IOError:
                logging.info("Нет отчёта прошлого релиза")
                # Не скипаем остальную логику - собираем все упавшие в список новых упавших
                old_report_data = {}

            old_failed_tests = set(find_all_test_names_recursive(old_report_data, BROKEN_TEST_STATUS_LIST))  # type: Set[str]

            for feature_key in self.feature_issue_keys:
                logging.info("Проверяем автотесты тикета %s" % feature_key)
                self.result[feature_key][test_type] = {}

                last_test_task, feature_report_data = self._load_feature_test_report(test_type, feature_key)
                if feature_report_data is None:
                    continue

                logging.info("Оба отчёта нашлись, сравниваем.")

                # Если отчёт пустой и таска в FAILURE - похоже, что-то сломалось в тикете
                if len(feature_report_data["children"]) == 0 and last_test_task.status == ctt.Status.FAILURE:
                    self.result[feature_key][test_type]["status"] = CRIT_STATUS + " Похоже, тесты в тикете сломаны совсем - отчёт пустой и таска упала"
                    continue

                feature_failed_tests = set(find_all_test_names_recursive(feature_report_data, BROKEN_TEST_STATUS_LIST))  # type: Set[str]

                # Сравнить два отчёта, записать результат сравнения в result
                self.result[feature_key][test_type] = self._compare_test_sets(old_failed_tests, feature_failed_tests)
                logging.info("Сравнили отчёты.")

    def _compare_test_sets(self, old_failed_tests, new_failed_test):
        # type: (Set[str], Set[str]) -> dict
        # Тут приведение обратно к list, потому что set - не сериализуется json
        old_failed = list(new_failed_test.intersection(old_failed_tests))
        new_failed = list(new_failed_test.difference(old_failed_tests))

        diff_size = len(new_failed)

        if diff_size == 0:
            return {"status": OK_STATUS, NEW_FAILED: new_failed, OLD_FAILED: old_failed}

        if diff_size < int(self.Parameters.crit_diff):
            return {"status": WARN_STATUS + " упало %d новых тестов" % diff_size, NEW_FAILED: new_failed, OLD_FAILED: old_failed}

        return {"status": CRIT_STATUS + " В тикете упало %d новых тестов по сравнению с предыдущим релизом" % diff_size, NEW_FAILED: new_failed, OLD_FAILED: old_failed}

    def _create_a_resource_with_comaprison_result(self):
        logging.info("Записываем результат в json-файл")
        with open(COMPARISON_RESULT_FILENAME, "w") as f:
            json.dump(self.result, f)

        # 3 раза пытаемся создать ресурс с отчётом
        max_retry_number = 3
        for i in range(0, max_retry_number):
            try:
                logging.info("Пытаемся создать ресурс с результатом таски")
                self.result_resource = MARKET_FRONT_AUTOTEST_COMPARATOR_RESULT(
                    task=self,
                    description="Результат сравнения автотестов в тикетах релиза %s с тестами в релизе %s" % (self.Parameters.current_release_version, self.Parameters.previous_release_version),
                    path=COMPARISON_RESULT_FILENAME,
                )
                sdk2.ResourceData(self.result_resource).ready()
                logging.info("Ресурс создан")
                # Выходим из цикла ретраев
                break
            except InvalidResource as ex:
                if i + 1 < max_retry_number:
                    logging.info(str(ex))
                    continue
                else:
                    raise ex

    def _remove_bad_tickets_from_release(self):
        with MetatronEnv(self):
            from startrek_client import Startrek

            oauth_token = sdk2.Vault.data("robot-metatron-st-token")
            st = Startrek(useragent="robot-metatron", token=oauth_token)

            for feature_key in self.result:
                should_remove_feature_from_release = False
                for test_type in self.result[feature_key]:
                    if CRIT_STATUS in self.result[feature_key][test_type]["status"]:
                        should_remove_feature_from_release = True

                if should_remove_feature_from_release:
                    self.removed_feature_list.append(feature_key)
                    if self.Parameters.enable_feature_removing:
                        self._remove_feature_issue_from_release(st, feature_key)

    def _send_telegram_report_through_api(self):
        if len(self.removed_feature_list) == 0:
            return
        if len(str(self.Parameters.notification_chat_id)) == 0:
            return

        removed_feature_links = ["https://st.yandex-team.ru/" + feature_key for feature_key in self.removed_feature_list]
        chat_id = str(self.Parameters.notification_chat_id)
        text = DEFAULT_TG_NOTIFY_TEMPLATE.format(feature_list="\n".join(removed_feature_links),
                                                 action="оторвали" if self.Parameters.enable_feature_removing else "нужно оторвать")

        logging.info("Пытаемся отправить сообщение в чат %s:\n%s" % (chat_id, text))

        with MetatronEnv(self):
            # Используем любой валидный OAuth токен для авторизации
            token = os.environ["ST_OAUTH_TOKEN"]

            response = send_telegram_message_via_tsum_bot(
                token,
                chat_id,
                text
            )

            logging.info("Ответ ЦУМа: {}, {}".format(response.status_code, response.text))

    def _send_startrek_report_to_release_ticket(self):
        if not self.Parameters.verbose:
            return

        logging.info("Пишем результат прогона в релизный тикет")
        with MetatronEnv(self):

            from startrek_client import Startrek
            oauth_token = sdk2.Vault.data("robot-metatron-st-token")
            st = Startrek(useragent="robot-metatron", token=oauth_token)

            st_search_query = "Type: \"Релиз\" \"Fix Version\": \"%s\" Queue: \"%s\"" % (self.Parameters.current_release_version, self.Parameters.st_queue)
            logging.info("Ищем релизный тикет в ST. Запрос: %s" % st_search_query)

            current_release_issues = st.issues.find(st_search_query)
            if len(current_release_issues) == 0:
                logging.info("Релизный тикет не найден. Отчёт не пишем.")
                return

            current_release_issue = current_release_issues[0]
            current_release_issue.comments.create(text=self._comment_text_from_result())
            logging.info("Записали отчёт в релизный тикет")

    def _comment_text_from_result(self):
        comment_strings = [
            "==Проблемы в прогонах автотестов\n",
            "((%s Полный отчёт))\n" % get_resource_http_proxy_link(self.result_resource)
        ]

        for issue_key in self.result.keys():
            key_title = "===((https://st.yandex-team.ru/%s \"\"%s\"\"))\n" % (issue_key, issue_key)
            comment_strings.append(key_title)
            for test_type in self.result[issue_key]:

                test_type_result = self.result[issue_key][test_type]
                status = test_type_result["status"].strip()

                if status.find(OK_STATUS) >= 0:
                    # Не пишем об этом в отчёт
                    pass
                elif status.find(WARN_STATUS) >= 0:
                    comment_strings.append("##%s##\n" % test_type)
                    comment_strings.append("!!(yellow)%s!!\n" % status)
                elif status.find(CRIT_STATUS) >= 0:
                    comment_strings.append("##%s##\n" % test_type)
                    comment_strings.append("!!(red)%s!!\n" % status)
                else:
                    comment_strings.append("##%s##\n" % test_type)
                    comment_strings.append("%s\n" % status)

                try:
                    new_failed = test_type_result[NEW_FAILED]
                    if len(new_failed) > 0:
                        new_failed = ["%%" + name + "%%" for name in new_failed]
                        comment_strings.append("<{diff:\n%s\n}>\n" % "\n\n".join(new_failed))
                except KeyError:
                    # Если нет new_failed - ничего не делаем
                    pass

            # Если все отчёты ок - не пишем этот тикет в отчёт
            if comment_strings[-1] == key_title:
                comment_strings.remove(key_title)
            else:
                comment_strings.append("\n")
        return "".join(comment_strings)

    def _load_feature_test_report(self, test_type, feature_key):
        # type: (str, str) -> Union[sdk2.Task, object]
        # Найти последнюю завершённую таску этого типа тестов в этом тикете
        autotest_tasks = sdk2.Task.find(
            task_type=MarketAutotestsHermione,
            status=(ctt.Status.SUCCESS, ctt.Status.FAILURE),
            input_parameters={
                "project_build_context": test_type,
                "testpalm_run_issue": feature_key
            },
            children=True
        ).order(-sdk2.Task.id).limit(20)  # type: Query

        # Если тасок автотестов нет совсем - это крит
        if autotest_tasks.count == 0:
            self.result[feature_key][test_type]["status"] = CRIT_STATUS + " В тикете не проходили тесты этого типа"
            logging.info("Нет таски этого типа в тикете")
            return None, None

        last_test_task = None
        for task in autotest_tasks:
            if task.Parameters.broken_tests_resource_id:
                # Нашли таску перепрогона, она нам и нужна. Дальше не ищем, таски отсортированы.
                last_test_task = task
                break

        # Если таски есть, но нет перепрогонов - смотреть такой тикет не будем, в статус напишем WARN
        if not last_test_task:
            self.result[feature_key][test_type]["status"] = WARN_STATUS + " В тикете был только основной паке тестов, не было перепрогона. Нельзя составить список упавших."
            logging.info("Нет перепрогона этого типа в тикете")
            return None, None
        logging.info("Нашли таску %d" % last_test_task.id)

        # Попытаться получить отчёт для тикета
        report = sdk2.Resource.find(
            resource_type=MARKET_AUTOTEST_REPORT,
            task=last_test_task
        ).first()

        if not report:
            logging.info("Таска %d не сгенерировала ресурс с отчётом" % last_test_task.id)
            return last_test_task, None

        logging.info("Нашли ресурс с отчётом %d" % report.id)
        try:
            resource_data = sdk2.ResourceData(report)
        except (InvalidResource, ResourceNotAvailable) as ex:
            self.result[feature_key][test_type]["status"] = WARN_STATUS + " Таска была, но отчёт не удалось получить."
            logging.info("Ресурс автотестов %s в тикете %s не удалось получить" % (test_type, feature_key))
            logging.info(str(ex))
            return last_test_task, None
        new_report_path = os.path.join(str(resource_data.path), SUITES_PATH)

        try:
            with open(new_report_path) as f:
                new_report_data = json.load(f)
        except IOError:
            # Обработать кейс когда отчёта для тикета нет
            self.result[feature_key][test_type]["status"] = WARN_STATUS + " Таска была, но отчёта нет. Подозрительно, лучше перепроверить"
            logging.info("Таска %d не сгенерировала отчёт" % last_test_task.id)
            return last_test_task, None
        except (ValueError, KeyError):
            self.result[feature_key][test_type]["status"] = WARN_STATUS + " Не удалось распарсить отчёт"
            logging.info("Таска %d сгенерировала не JSON или JSON без поля children" % last_test_task.id)
            return last_test_task, None

        logging.info("Прочитали результаты из файла")
        return last_test_task, new_report_data

    def _remove_feature_issue_from_release(self, st, feature_key):
        from startrek_client.exceptions import NotFound, StartrekError

        issue = st.issues[feature_key]
        issue.update(fixVersions={"remove": [self.Parameters.current_release_version], "add": []})

        text = DEFAULT_ST_CALL_TEXT_TEMPLATE % self.Parameters.crit_diff
        if issue["assignee"] is not None:
            summonees = [issue["assignee"].login]
        else:
            summonees = []

        issue.comments.create(text=text, summonees=summonees)

        try:
            transition = issue.transitions["checkFailed"]
            transition.execute()
        except NotFound:
            logging.error("Перехода не существует для тикета %s. Доступные переходы %s" % (feature_key, str([t.id for t in issue.transitions])))
        except StartrekError as e:
            logging.error(e)


def find_all_test_names_recursive(data, status_list):
    # type: (dict, List[str]) -> List[str]
    test_names = []
    if "children" in data:
        for child in data["children"]:
            test_names += find_all_test_names_recursive(child, status_list)
        return test_names

    # Мы дошли до листа дерева, и надо его записать в список скипов
    try:
        status = data["status"]  # type: str
        name = data["name"]  # type: str
    except KeyError as e:
        logging.warning("Не удалось получить атрибуты `name` и `status` у объекта. Объект:\n%s" % json.dumps(data), exc_info=e)
        return test_names

    if status in status_list:
        test_names.append(name)

    return test_names
