# -*- coding: utf-8 -*-

import os
import re
import codecs
import pipes
import tempfile
import logging
import json
import time
import sandbox.sdk2.path as spath

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.types import misc as ctm
from sandbox.projects.market.front.helpers.MetatronEnv import MetatronEnv
from sandbox.projects.market.front.helpers.github import clone_repo
from sandbox.projects.market.front.helpers.sandbox_helpers import rich_check_call, \
    get_resource_http_proxy_link as get_resource_link
from sandbox.projects.market.front.helpers.ubuntu import create_ubuntu_selector, setup_container
from sandbox.projects.market.front.helpers.node import create_node_selector
from sandbox.sandboxsdk.environments import PipEnvironment

UNKNOWN_TASK_ID = 0

# Мапа конфиг genisys -> файл скипов
AUTOTEST_CONFIG_SKIP_JS_PATH_MAP = {
    "white-h2-kadavr-desktop": os.path.join("skipped", "hermione2", "desktop.js"),
    "white-h2-kadavr-touch": os.path.join("skipped", "hermione2", "touch.js"),
    "white-testing-desktop": os.path.join("skipped", "market", "desktop.js"),
    "white-testing-touch": os.path.join("skipped", "market", "touch.js"),
}

# Мапа конфиг genisys -> JSON файл скипов
AUTOTEST_CONFIG_SKIP_JSON_PATH_MAP = {
    "white-h2-kadavr-desktop": os.path.join("skipped", "hermione2", "desktop.json"),
    "white-h2-kadavr-touch": os.path.join("skipped", "hermione2", "touch.json"),
    "white-testing-desktop": os.path.join("skipped", "market", "desktop.json"),
    "white-testing-touch": os.path.join("skipped", "market", "touch.json"),
}

COMPONENTS = {
    "@Desktop": 61463,
    "@Touch": 61464,
    "@Market": 84838,
}

AUTOTEST_CONFIG_COMPONENTS_MAP = {
    "white-h2-kadavr-desktop": [COMPONENTS["@Desktop"], COMPONENTS["@Market"]],
    "white-h2-kadavr-touch": [COMPONENTS["@Touch"], COMPONENTS["@Market"]],
    "white-testing-desktop": [COMPONENTS["@Desktop"], COMPONENTS["@Market"]],
    "white-testing-touch": [COMPONENTS["@Touch"], COMPONENTS["@Market"]],
}

# Available modes
SKIP_JS = "skip .js"
SKIP_JSON = "skip .json"
UNSKIP = "unskip"

JSON_MODES = [SKIP_JSON, UNSKIP]
JS_MODES = [SKIP_JS]

ST_MAX_RETRY = 3
ST_SECONDS_BETWEEN_RETRIES = 10
LOG_ISSUE_KEY = "MARKETFRONT-55313"
DEFAULT_QA_CALL_TEXT = "==ВАЖНО!\n" \
                       "ОБЯЗАТЕЛЬНО проставь теги контура, отвественного за скипнутые тесты\n"


class MarketFrontAutotestSkip(sdk2.Task):
    # Ресурс с allure-отчётом
    report_resource = None
    autotest_run_data = None
    new_skip_names = []
    prepared_skips = []
    release_version_name = None
    genisys_config_name = None
    skip_file_full_path = None
    root_dir = None
    skips_removed_count = 0
    skip_records_removed_count = 0
    config_object = None

    """
    Таска, обновляющая скипы автотестов.
    Берёт названия тестов со статусам failed и broken в ресурсе переданной задачи и добавляет коммит в репозиторий скипов.
    """

    class Parameters(sdk2.Task.Parameters):
        latest_autotest_task_id = sdk2.parameters.Integer(
            "Id таски автотестов",
            description="Таска, на чей прогон тестов надо смотреть (MARKET_AUTOTESTS_HERMIONE)",
            required=True,
        )

        max_skip_limit = sdk2.parameters.Integer(
            "Ограничение на число скипов",
            description="Таска упадёт, если кто-то попытается скипнуть больше тестов, чем установлено тут",
            default=20,
            required=True,
        )
        with sdk2.parameters.RadioGroup("Job mode") as mode:
            mode.values[SKIP_JS] = mode.Value(SKIP_JS, default=True)
            mode.values[SKIP_JSON] = mode.Value(SKIP_JSON)
            mode.values[UNSKIP] = mode.Value(UNSKIP)

        with sdk2.parameters.Group("Параметры отчёта") as report_config:
            report_resource_type = sdk2.parameters.String(
                "Тип Sandbox ресурса",
                description="Ресурс, в котором хранится отчёт тестов",
                default="MARKET_AUTOTEST_REPORT",
            )

            report_path = sdk2.parameters.String(
                "Путь к JSON-файлу allure в ресурсе",
                default=os.path.join("data", "suites.json"),
            )

        with sdk2.parameters.Group('Startrek') as st_config:
            st_queue = sdk2.parameters.String(
                "Очередь для создания тикета на починку",
                default="MARKETFRONT",
            )

            st_tags = sdk2.parameters.String(
                "Теги тикета через запятую",
                description='Пробелы вокруг запятых будут очищены',
                default="автотест, починка, скип",
            )

            task_author = sdk2.parameters.String(
                "Автор скипов",
                description="От чьего имени будет создан тикет",
                required=False,
            )

        with sdk2.parameters.Group('GitHub репозиторий скипов') as github_repo_block:
            app_owner = sdk2.parameters.String(
                'GitHub owner',
                description='Логин владельца репозитория или название организации',
                default='market',
                required=True,
            )
            app_repo = sdk2.parameters.String(
                "Репозиторий",
                default='marketplace-hermione-tests-config',
                required=True,
            )
            app_branch = sdk2.parameters.String(
                "Ветка",
                default='master',
                required=True,
            )

            ubuntu_version = create_ubuntu_selector()
            node_version = create_node_selector()

    class Requirements(sdk2.Task.Requirements):
        dns = ctm.DnsType.DNS64
        environments = [
            PipEnvironment('yandex_tracker_client', version="1.3", custom_parameters=["--upgrade-strategy only-if-needed"]),
            PipEnvironment('startrek_client', version="2.3.0", custom_parameters=["--upgrade-strategy only-if-needed"])
        ]

    def on_enqueue(self):
        super(MarketFrontAutotestSkip, self).on_enqueue()
        setup_container(self)

    def on_prepare(self):
        self.root_dir = tempfile.mkdtemp()

    def on_execute(self):
        self._load_sandbox_resource()
        self._load_parent_task_parameters()
        self._clone_skip_repo()
        self._select_mode()

    def on_failure(self, prev_status):
        self._close_st_issues_on_failure()

    def on_break(self, prev_status, status):
        self._close_st_issues_on_failure()

    def _load_sandbox_resource(self):
        """Пытается найти в переданной таске ресурс указанного типа и выкачать из него данные по скипам по указанному пути"""
        logging.info("Начинаем загрузку ресурса с тестами")

        if self.Parameters.latest_autotest_task_id == UNKNOWN_TASK_ID:
            msg = "В параметрах таски передали невалидный ID таски автотестов.\nСкорее всего ЦУМ-джоба не смогла найти ни одной подходящей джобы автотестов."
            raise TaskFailure(msg)

        self.report_resource = sdk2.Resource[self.Parameters.report_resource_type].find(task_id=self.Parameters.latest_autotest_task_id).first()

        if not self.report_resource:
            msg = "У таски %d нет ресурса %s. Где-то передали неправильный тип таски для поиска отчёта." % (int(self.Parameters.latest_autotest_task_id), self.Parameters.report_resource_type)
            raise TaskFailure(msg)

        resource_data = sdk2.ResourceData(self.report_resource)
        report_path = os.path.join(str(resource_data.path), self.Parameters.report_path)

        if not os.path.exists(report_path):
            msg = "Ожидалось что в ресурсе %d будет файл %s с JSON-объектом с данными о прогоне. Файл не найден." % (self.report_resource.id, self.Parameters.report_path)
            raise TaskFailure(msg)

        self.autotest_run_data = spath.Path(report_path).read_bytes()
        logging.info("Закончили загрузку ресурса с тестами")

    def _load_parent_task_parameters(self):
        logging.info("Вытаскиваем параметры из таски автотестов")
        latest_autotest_task = sdk2.Task.find(
            id=self.Parameters.latest_autotest_task_id
        ).first()

        try:
            self.release_version_name = latest_autotest_task.Parameters.testpalm_run_version
            if not self.release_version_name:
                self.release_version_name = latest_autotest_task.Parameters.app_branch.split("release/")[1]
        except (AttributeError, IndexError):
            self.release_version_name = ""

        try:
            self.genisys_config_name = latest_autotest_task.Parameters.project_build_context
        except AttributeError:
            msg = "В родительской задаче %s отсутствует поле project_build_context. Нельзя определить, в какой конфиг писать скип." % self.Parameters.latest_autotest_task_id
            raise TaskFailure(msg)

        path_map = AUTOTEST_CONFIG_SKIP_JSON_PATH_MAP if self.Parameters.mode in JSON_MODES else AUTOTEST_CONFIG_SKIP_JS_PATH_MAP
        try:
            self.skip_file_full_path = os.path.join(self.root_dir, path_map[self.genisys_config_name])
        except AttributeError:
            msg = "У таска автотестов неизвестный genisys-конфиг `%s`.\nЕсли нужного конфига нет в списке - добавьте в код этой таски.\nСписок известных конфигов: [%s]. " \
                  % (self.genisys_config_name, ", ".join(path_map.keys()))
            raise TaskFailure(msg)
        logging.info("Параметры из таски автотестов получены")

    def _get_tests(self, status):
        status_list = [status] if not isinstance(status, list) else status
        logging.info("Получаем список тестов")
        if not self.autotest_run_data:
            msg = "Отчёт о прогоне - пустой. В нём нет тестов, которые можно скипнуть."
            raise TaskFailure(msg)

        data = json.loads(self.autotest_run_data)
        self._find_test_recursive(data, status_list)

        logging.info("Получили список тестов:\n%s" % "\n".join(self.new_skip_names))

        if len(self.new_skip_names) == 0:
            msg = "Не удалось найти ни одного теста в отчёте. Скип не будет добавлен."
            raise TaskFailure(msg)

    def _check_limits(self):
        if len(self.new_skip_names) > int(self.Parameters.max_skip_limit):
            msg = ("Кто-то пытается скипнуть слишком много тестов. Стоит ограничение - можно скипнуть не более %d тестов.\n" % self.Parameters.max_skip_limit) \
                  + "Если вам ДЕЙСТВИТЕЛЬНО нужно скипнуть %d тестов, склонируйте эту таску и исправьте ограничени в поле max_skip_limit." % len(self.new_skip_names)
            raise TaskFailure(msg)

    def _clone_skip_repo(self):
        clone_repo(
            pipes.quote(self.Parameters.app_owner),
            pipes.quote(self.Parameters.app_repo),
            pipes.quote(self.Parameters.app_branch),
            self.root_dir
        )

    def _filter_duplicate_skips(self):
        all_skipped_test_names = []
        for skip in self.config_object['actual']:
            all_skipped_test_names += skip['fullNames']
        all_skipped_test_names = set(all_skipped_test_names)

        # Нужен только для логирования - что выкидываем
        duplicate_skips = [skip for skip in self.new_skip_names if skip in all_skipped_test_names]
        if len(duplicate_skips) > 0:
            logging.info("Следующие скипы уже есть в конфиге и не будут скипнуты повторно:\n%s" % "\n".join(duplicate_skips))

        self.new_skip_names = [skip for skip in self.new_skip_names if skip not in duplicate_skips]

        if len(self.new_skip_names) == 0:
            msg = "Все тесты уже были скипнуты. Нет новых тестов для скипа."
            raise TaskFailure(msg)

    def _prepare_skip_ticket_for_every_test(self):
        with MetatronEnv(self, nodejs_version=self.Parameters.node_version):
            # startreck
            from startrek_client import Startrek

            oauth_token = sdk2.Vault.data("robot-metatron-st-token")
            st = Startrek(useragent="robot-metatron", token=oauth_token)

            for test_name in self.new_skip_names:
                skip_issue = self._create_startrek_ticket(st, test_name)
                if skip_issue is None:
                    logging.info("Не удалось создать тикет для скипа \"%s\"" % test_name)
                    continue
                self.prepared_skips.append({
                    "issue": skip_issue,
                    "test_name": test_name,
                })

    def _create_startrek_ticket(self, st, skip_test_name):
        actual_queue = self.Parameters.st_queue
        actual_tags = [tag.strip() for tag in str(self.Parameters.st_tags).split(",")]
        actual_description = "В релизе %s были скипнут тест:\n\n" % self.release_version_name
        actual_description += "%%(bash)\n"
        actual_description += skip_test_name
        actual_description += "\n%%\n"
        actual_description += "Ссылка на отчёт: %s/index.html\n" % get_resource_link(self.report_resource)

        actual_summary = "[Починка АТ] %s" % skip_test_name

        actual_author = self.Parameters.task_author if self.Parameters.task_author else self.author

        ticket_params = {
            "createdBy": actual_author,
            "description": actual_description,
            "priority": "normal",
            "queue": actual_queue,
            "summary": actual_summary,
            "tags": actual_tags,
            "fixVersions": "% Поддержка",
            "components": AUTOTEST_CONFIG_COMPONENTS_MAP[self.genisys_config_name],
            "type": {
                "name": "Test"
            }
        }

        from startrek_client.exceptions import StartrekError
        for attempt in range(ST_MAX_RETRY):
            try:
                issue = st.issues.create(**ticket_params)
                logging.info("Создан тикет %s" % issue.key)
                return issue
            except StartrekError as e:
                logging.error("Попытка %d. Не удалось создать тикет для скипа в ST" % attempt, exc_info=e)
                logging.info("Ждём 10 секунд до следующей попытки создать тикет")
                time.sleep(ST_SECONDS_BETWEEN_RETRIES)

        return None

    def _update_skip_resource_ttl(self):
        # Если новых скипов нет - обновлять TTL ресурса не нужно
        if len(self.prepared_skips) == 0 or self.report_resource is None:
            return

        self.report_resource.ttl = 180

    def _enrich_skip_issues(self):
        with MetatronEnv(self):
            # startreck
            from startrek_client import Startrek

            oauth_token = sdk2.Vault.data("robot-metatron-st-token")
            st = Startrek(useragent="robot-metatron", token=oauth_token)
            release_issues = []
            if self.release_version_name:
                release_issues = st.issues.find('Type: \"Релиз\" \"Fix Version\": \"%s\" Queue: \"%s\"' % (self.release_version_name, self.Parameters.st_queue))

        for skip in self.prepared_skips:
            issue = skip['issue']
            # Пытаемся привязать релизный тикет к созданному тикету
            if self.release_version_name:
                try:
                    if len(release_issues) > 0:
                        issue.links.create(issue=release_issues[0].id, relationship='relates')
                except Exception as e:
                    logging.error("Ошибка при создании связи с релизном тикетом в ST. Не обзяательный шаг, пропускаем.", exc_info=e)

            # Пытаемся позвать автора в комментарии
            if self.Parameters.task_author:
                try:
                    issue.comments.create(
                        text=DEFAULT_QA_CALL_TEXT,
                        summonees=self.Parameters.task_author,
                    )
                except Exception as e:
                    logging.error("Ошибка при призыве автора в ST. Не обзяательный шаг, пропускаем.", exc_info=e)

    def _select_mode(self):
        if self.Parameters.mode == SKIP_JS:
            logging.info("Работаем в режиме \"Скипнуть, скипы лежат в JS\"")
            self._get_tests(["failed", "broken"])
            self._check_limits()
            self._prepare_skip_ticket_for_every_test()
            self._add_skips_js()
            self._check_linter()
            self._push_new_skips("Skip added: release {}".format(self.release_version_name))
            self._update_skip_resource_ttl()
            self._enrich_skip_issues()
            return

        if self.Parameters.mode == SKIP_JSON:
            logging.info("Работаем в режиме \"Скипнуть, скипы лежат в JSON\"")
            self._get_tests(["failed", "broken"])
            self._check_limits()
            self._load_config()
            self._filter_duplicate_skips()
            self._prepare_skip_ticket_for_every_test()
            self._add_skips_json()
            self._check_linter()
            self._push_new_skips("Skip added: release {}".format(self.release_version_name))
            self._update_skip_resource_ttl()
            self._enrich_skip_issues()
            return

        # Работает только с JSON-конфигом
        if self.Parameters.mode == UNSKIP:
            logging.info("Работаем в режиме \"Расскипать тесты в JSON\"")
            self._get_tests(["passed"])
            self._load_config()
            self._remove_skips_json()
            self._push_new_skips("Removed %d unactal skips and closed %d tickets" % (self.skips_removed_count, self.skip_records_removed_count))
            self._log_to_st()
            return

        logging.error("Таска была создана с неизвестным режимом mode: %s" % self.Parameters.mode)

    def _add_skips_js(self):
        skip_insert_place = "actual: [\n"
        ident = "    "  # 4 пробела

        with open(self.skip_file_full_path, 'r') as skip_file:
            raw_config = skip_file.read()
            [begin, current_skips] = raw_config.split(skip_insert_place)
            default_ident = re.search(r'[ ]*', current_skips).group()

        new_skips = []
        for skip in self.prepared_skips:
            # Специально не заменил на format - в нём совсем невозможно разобраться в происходящем.
            new_skip = default_ident + "{\n" \
                       + default_ident + ident + ("issue: '%s',\n" % skip['issue'].key) \
                       + default_ident + ident + ("reason: 'Скип автотестов в релизе %s',\n" % self.release_version_name) \
                       + default_ident + ident + "fullNames: [\n" \
                       + default_ident + ident + ident + ("'%s',\n" % skip['test_name']) \
                       + default_ident + ident + "],\n" \
                       + default_ident + "},\n"
            new_skips.append(new_skip)

        updated_config = begin + skip_insert_place + "".join(new_skips) + current_skips

        with open(self.skip_file_full_path, 'w') as skip_file:
            skip_file.write(updated_config)

    def _add_skips_json(self):
        new_skips = []
        for skip in self.prepared_skips:
            new_skips.append({
                'issue': skip['issue'].key,
                'reason': 'Скип автотестов в релизе %s' % self.release_version_name,
                'fullNames': [skip['test_name']],
            })

        self.config_object['actual'] = new_skips + self.config_object['actual']

        self._save_config()

    def _check_linter(self):
        # TODO
        # Или запускать подзадачу MARKET_FRONT_CI и дожидаться выполнения - или ставить вручную нод_модули,
        # запускать линтер и проверять его результат из этого таска
        pass

    def _remove_skips_json(self):
        skips_to_remove = self.new_skip_names

        with MetatronEnv(self, nodejs_version=self.Parameters.node_version):
            # startreck
            from startrek_client import Startrek
            from startrek_client.exceptions import StartrekError

            oauth_token = sdk2.Vault.data("robot-metatron-st-token")
            st = Startrek(useragent="robot-metatron", token=oauth_token)

            logging.info("Начинаем удалять лишние скипы из конфига")
            for skip_record in self.config_object['actual']:
                unactual_skips = [name for name in skip_record['fullNames'] if name in skips_to_remove]
                actual_skips = [name for name in skip_record['fullNames'] if name not in unactual_skips]

                self.skips_removed_count += len(unactual_skips)

                if len(unactual_skips) > 0:
                    skip_record['fullNames'] = actual_skips
                    try:
                        st.issues[skip_record['issue']].comments.create(
                            text="Следующие тесты были расскипаны, т.к. стабильно проходят:\n%%(bash)\n"
                            + "\n".join(unactual_skips)
                            + "%%\n"
                        )
                    except StartrekError as e:
                        logging.error(e)
                        # ignored
                        pass

                if len(skip_record['fullNames']) == 0:
                    logging.info("В тикете %s не осталось тестов на починку." % skip_record['issue'])
                    close_st_issue(st, skip_record['issue'])
                    self.skip_records_removed_count += 1

        self.config_object['actual'] = list(filter(lambda skip: len(skip['fullNames']) != 0, self.config_object['actual']))

        self._save_config()

    def _push_new_skips(self, message):
        command_add = [
            "git",
            "add",
            self.skip_file_full_path
        ]
        command_commit = [
            "git",
            "commit",
            "-m",
            message
        ]
        push_url = "ssh://git@github.yandex-team.ru/{owner}/{repo}.git".format(
            owner=self.Parameters.app_owner,
            repo=self.Parameters.app_repo,
        )

        command_push = [
            "git",
            "push",
            push_url,
            self.Parameters.app_branch
        ]
        with MetatronEnv(self, nodejs_version=self.Parameters.node_version):
            rich_check_call(command_add, task=self, alias="git-add", cwd=self.root_dir)
            rich_check_call(command_commit, task=self, alias="git-commit", cwd=self.root_dir)
            rich_check_call(command_push, task=self, alias="git-push", cwd=self.root_dir)

    def _log_to_st(self):
        with MetatronEnv(self, nodejs_version=self.Parameters.node_version):
            # startreck
            from startrek_client import Startrek

            oauth_token = sdk2.Vault.data("robot-metatron-st-token")
            st = Startrek(useragent="robot-metatron", token=oauth_token)
            st.issues[LOG_ISSUE_KEY].comments.create(
                text="Removed %d unactal skips and closed %d tickets" % (self.skips_removed_count, self.skip_records_removed_count)
            )

    def _find_test_recursive(self, data, expected_status_list):
        if "children" in data:
            for child in data["children"]:
                self._find_test_recursive(child, expected_status_list)
            return

        # Мы дошли до листа дерева, и надо его записать в список скипов
        try:
            status = data["status"]
            name = data["name"]
        except KeyError as e:
            logging.warning("Не удалось получить атрибуты `name` и `status` у объекта. Объект:\n%s" % json.dumps(data), exc_info=e)
            return

        if status in expected_status_list:
            self.new_skip_names.append(name)
            return

    def _load_config(self):
        logging.info("Загружаем конфиг текущих скипов")
        with open(self.skip_file_full_path, 'r') as skip_file:
            self.config_object = json.load(skip_file)
            logging.info("Конфиг загрузили")

    def _save_config(self):
        logging.info("Сохраняем конфиг скипов")
        with codecs.open(self.skip_file_full_path, 'w', encoding='utf8') as skip_file:
            json.dump(
                self.config_object,
                skip_file,
                ensure_ascii=False,
                indent=4,
                sort_keys=True,
                # Если есть ident, то по умолчанию сепараторы (', ', ': '), из-за чего в конце строк с запятой - лишние пробелы
                separators=(',', ': '),
            )

    def _close_st_issues_on_failure(self):
        for skip in self.prepared_skips:
            try:
                skip["issue"].transitions['close'].execute(comment='Произошла ошибка при добавлении скипа, скип не добавлен, тикет закрываю.', resolution='fixed')
            except Exception as e:
                logging.error("Не удалось закрыть открытый тикет", exc_info=e)


def close_st_issue(st, issue_key):
    logging.info("Пытаемся закрыть тикет %s" % issue_key)
    from startrek_client.exceptions import NotFound, StartrekError
    try:
        transition = st.issues[issue_key].transitions['close']
        transition.execute(comment='Все тесты из этой задачи - починились или больше не существуют', resolution='fixed')
        logging.info("Закрыли тикет %s" % issue_key)
    except NotFound:
        logging.error("Перехода не существует для тикета %s. Доступные переходы %s" % (issue_key, str([t.id for t in st.issues[issue_key].transitions])))
        try:
            st.issues[issue_key].comments.create(text='Все тесты из этой задачи - починились или больше не существуют. Скипы удалены из конфига')
        except StartrekError as e:
            logging.error(e)
            # ignored
            pass


def pluralize(number, nom, gen, plu):
    num = number % 100

    if num >= 11 and num <= 19:
        return plu

    num = num % 10
    num_to_plural = {
        1: nom,
        2: gen,
        3: gen,
        4: gen,
    }

    try:
        return num_to_plural[num]
    except KeyError:
        return plu
