# coding=utf-8

import os.path
import re
import tempfile
import shutil
import sandbox.common.types.misc as ctm
from sandbox import sdk2
from sandbox.projects.market.front.helpers.MetatronEnv import MetatronEnv
from sandbox.projects.market.front.helpers.sandbox_helpers import rich_check_call
from sandbox.projects.market.front.helpers.node import create_node_selector
from sandbox.projects.market.front.helpers.ubuntu import create_ubuntu_selector, setup_container
from sandbox.sdk2.vcs.git import Git
import sandbox.sdk2.path as spath
from sandbox.projects.market.resources import MARKET_PARTNER_FRONT_ADV_APP


class MarketFrontBuildAdvStatic(sdk2.Task):
    """
    Task for export-static build
    https://github.yandex-team.ru/market/adv-www/
    """

    __PACKAGE_REPO = "https://github.yandex-team.ru/market/adv-www"
    __PACKAGE_DIR = "adv-www"
    __APP_FILES = ["package.json", "node_modules", "configs", "server", "out", "www", "scripts"]

    class Parameters(sdk2.Task.Parameters):
        packages_branch = sdk2.parameters.String(
            "Git ветка проекта",
            default_value="dev",
            required=True
        )

        ubuntu_version = create_ubuntu_selector()
        node_version = create_node_selector()

    class Requirements(sdk2.Task.Requirements):
        dns = ctm.DnsType.DNS64

    def on_enqueue(self):
        super(MarketFrontBuildAdvStatic, self).on_enqueue()
        setup_container(self)

    def on_execute(self):
        super(MarketFrontBuildAdvStatic, self).on_execute()

        with MetatronEnv(self, nodejs_version=self.Parameters.node_version):
            self._prepare_repos()
            self._build_project()
            self._filter_app()

            pack_app_archive_path = self._create_tar()

            self._app_create_resource(pack_app_archive_path)

    def _prepare_repos(self):
        Git(self.__PACKAGE_REPO).clone(
            os.path.join(os.getcwd(), self.__PACKAGE_DIR),
            self.Parameters.packages_branch
        )

    def _build_project(self):
        rich_check_call(
            ["npm", "run", "deps"],
            self,
            "npm run deps",
            cwd=self.__PACKAGE_DIR
        )

        rich_check_call(
            ["npm", "run", "build:production"],
            self,
            "npm run build:production",
            cwd=self.__PACKAGE_DIR
        )

    def _filter_app(self):
        directory_list = os.listdir(os.path.join(os.getcwd(), self.__PACKAGE_DIR))

        for dir_entry in directory_list:
            if dir_entry not in self.__APP_FILES:
                path_to_target = os.path.join(os.getcwd(), self.__PACKAGE_DIR, dir_entry)
                if os.path.isdir(path_to_target):
                    shutil.rmtree(path_to_target)
                else:
                    os.unlink(path_to_target)

    def _create_tar(self):
        app_repo_match = re.search(".+/([^/.]+?)(:?.git)?$", self.__PACKAGE_REPO)
        assert app_repo_match is not None
        pack_app_dir = app_repo_match.group(1)
        pack_app_archive_path = tempfile.mktemp(suffix=".tar.gz", prefix="app")
        rich_check_call(
            ["tar", "-czf", pack_app_archive_path, pack_app_dir],
            self,
            "tar"
        )

        return pack_app_archive_path

    def _app_create_resource(self, pack_app_archive_path):
        resource_type = MARKET_PARTNER_FRONT_ADV_APP
        app_res = sdk2.ResourceData(resource_type(
            self, "App tarball", "app.tar.gz"
        ))
        if app_res.path.exists():
            app_res.path.unlink()
        app_res.path.write_bytes(spath.Path(pack_app_archive_path).read_bytes())
        app_res.ready()
