# coding=utf-8

import logging
import os
import pipes
import tempfile

import sandbox.common.types.misc as ctm
import sandbox.sdk2.path as spath
from sandbox import sdk2

from sandbox.projects.market.front.helpers.MetatronEnv import MetatronEnv
from sandbox.projects.market.front.helpers.github import clone_repo, clone_repo_for_merge_commit, \
    find_opened_release_branch
from sandbox.projects.market.front.helpers.node import NODE_DEFAULT
from sandbox.projects.market.front.helpers.sandbox_helpers import rich_check_call
from sandbox.projects.market.front.helpers.node import create_node_selector
from sandbox.projects.market.front.helpers.ubuntu import create_ubuntu_selector, setup_container
from sandbox.projects.market.resources import MARKET_APP
from sandbox.projects.sandbox_ci.utils import env

KILL_TIMEOUT = 30 * 60  # 30 min
DISK_SPACE = 3 * 1024  # 3 Gb
GH_OWNER = 'MARKET'

# todo: выпилить сборку из таски с автотестами (MarketAutotests, в сути -- копипаста оттуда)
class MarketFrontBuildDev(sdk2.Task):
    """
    Сборка приложения в Dev окружении. Для автотестов и CI
    """

    APP_SRC_PATH = ''
    APP_REPO_PATH = ''
    root_dir = ''
    app_branch_final = None

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = KILL_TIMEOUT
        ubuntu_version = create_ubuntu_selector()
        node_version = create_node_selector()

        with sdk2.parameters.Group('GitHub репозиторий проекта') as github_repo_block:
            app_owner = sdk2.parameters.String(
                'GitHub owner',
                description='Логин владельца репозитория или название организации',
                default=GH_OWNER,
                required=True
            )
            app_repo = sdk2.parameters.String(
                "Репозиторий",
                default='market',
                required=True
            )
            app_branch = sdk2.parameters.String(
                "Ветка",
                default='master',
                required=True
            )
            app_src_dir = sdk2.parameters.String(
                "Кастомный путь корня приложения внутри репозитория"
            )
            use_release_branch = sdk2.parameters.Bool(
                'Использовать текущую релизную ветку',
                description='Отметьте, если нужно поискать ветку с текущим релизом, и прогнать АТ из неё '
                            '(если не будет найдена, прогонится из ветки, указанной в app_branch)',
                default=False
            )
            with use_release_branch.value[True]:
                app_release_branch_starts_with = sdk2.parameters.String(
                    'Начало названия релизной ветки для поиска её на гитхабе',
                    default='release/',
                    required=True
                )
            app_merge_commit = sdk2.parameters.String(
                "Merge-коммит",
                default='',
                required=False
            )
            app_status_check_commit = sdk2.parameters.String(
                "Коммит для отправки статус-чеков",
                default='',
                required=False
            )

        with sdk2.parameters.Group('Environment') as environ_block:
            environ = sdk2.parameters.Dict('Environment variables')

    class Requirements(sdk2.Task.Requirements):
        dns = ctm.DnsType.DNS64
        disk_space = DISK_SPACE

    def on_enqueue(self):
        super(MarketFrontBuildDev, self).on_enqueue()
        setup_container(self)

    @property
    def project_name(self):
        return self.Parameters.app_repo

    @property
    def node_version(self):
        return self.Parameters.node_version or NODE_DEFAULT

    @property
    def app_resource_id(self):
        return self._app_resource_id

    @app_resource_id.setter
    def app_resource_id(self, value):
        self._app_resource_id = value

    def _clone_repo(self):
        if self.Parameters.app_merge_commit:
            clone_repo_for_merge_commit(
                pipes.quote(self.Parameters.app_owner),
                pipes.quote(self.Parameters.app_repo),
                pipes.quote(self.Parameters.app_merge_commit),
                self.APP_REPO_PATH
            )
        else:
            clone_repo(
                pipes.quote(self.Parameters.app_owner),
                pipes.quote(self.Parameters.app_repo),
                pipes.quote(self.app_branch_final),
                self.APP_REPO_PATH
            )

    def _configure(self):
        rich_check_call(
            ["make", "configure"],
            task=self, alias="configure", cwd=self.APP_SRC_PATH
        )

    def _build(self):
        logging.info('Running project build with environment: {}'.format(os.environ))

        rich_check_call(
            ["make", "bootstrap"],
            task=self, alias="bootstrap", cwd=self.APP_SRC_PATH
        )

    def _create_app_pack(self):
        pack_app_archive_path = tempfile.mktemp(suffix=".tar.gz", prefix=self.Parameters.app_repo)

        rich_check_call(
            ["tar", "-C", os.path.join(self.APP_REPO_PATH, '..'), "-czf", pack_app_archive_path, self.Parameters.app_repo],
            task=self, alias="create_app_pack"
        )

        self._app_create_resource(pack_app_archive_path)

    def _app_create_resource(self, pack_app_archive_path):
        resource = MARKET_APP(
            self, "App tarball", "{}.tar.gz".format(self.Parameters.app_repo),
            app_repo=self.Parameters.app_repo,
            app_branch=self.app_branch_final,
            app_merge_commit=self.Parameters.app_merge_commit,
            app_status_check_commit=self.Parameters.app_status_check_commit,
        )
        app_res = sdk2.ResourceData(resource)

        app_res.path.write_bytes(spath.Path(pack_app_archive_path).read_bytes())
        app_res.ready()

        self.app_resource_id = resource.id

    def _prepare_env(self):
        # NB: last export overrides previous
        env.export(self.Parameters.environ)
        return

    def _prepare_app_branch(self):
        release_branch = find_opened_release_branch(
            owner=self.Parameters.app_owner,
            repo=self.Parameters.app_repo,
            branch_starts_with=self.Parameters.app_release_branch_starts_with,
        ) if self.Parameters.use_release_branch else None

        self.app_branch_final = release_branch or self.Parameters.app_branch

    def on_execute(self):
        # NB: MetatronEnv clears env on exit
        with MetatronEnv(self, nodejs_version=self.Parameters.node_version), self.memoize_stage.preparation(max_runs=1):
            self.root_dir = tempfile.mkdtemp()
            self.APP_REPO_PATH = os.path.join(self.root_dir, self.Parameters.app_repo)

            if self.Parameters.app_src_dir:
                self.APP_SRC_PATH = os.path.join(self.APP_REPO_PATH, self.Parameters.app_src_dir)
            else:
                self.APP_SRC_PATH = self.APP_REPO_PATH

            self._prepare_env()
            self._prepare_app_branch()
            self._clone_repo()
            self._configure()
            self._build()
            self._create_app_pack()
