# coding=utf-8
from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.sdk2.vcs.git import Git
from sandbox.projects.common.debpkg import DebRelease

from sandbox.projects.market.front.helpers.MetatronEnv import MetatronEnv
from sandbox.projects.market.front.helpers.sandbox_helpers import rich_check_call
from sandbox.projects.market.front.helpers.node import create_node_selector
from sandbox.projects.market.front.helpers.ubuntu import create_ubuntu_selector, setup_container

import os
import logging
import json


class MarketFrontBuildExportStatic(sdk2.Task):
    """
    Task for export-static build

    https://github.yandex-team.ru/market/export-static/
    """

    class Parameters(sdk2.Task.Parameters):
        app_branch = sdk2.parameters.String(
            "Git ветка проекта",
            default_value="master"
        )

        packages_branch = sdk2.parameters.String(
            "Git ветка packages",
            default_value="master"
        )

        ubuntu_version = create_ubuntu_selector()
        node_version = create_node_selector()

    __PACKAGES_REPO = "https://github.yandex-team.ru/market/packages"
    __PACKAGES_DIR = "packages"
    __MAKEDEBPKG_CONF = "makedebpkg.conf"
    __CARCH = "all"
    __RESOURCES_DIR = os.path.join(os.path.dirname(__file__), "..", "helpers")

    MAKEDEBPKG_PATH = "/opt/makedebpkg"
    MAKEDEBPKG = ".makedebpkg"

    __EXPORT_WWW_STATIC_DIR = "yandex-market-export-www-static"

    def on_enqueue(self):
        super(MarketFrontBuildExportStatic, self).on_enqueue()
        setup_container(self)

    def on_execute(self):
        super(MarketFrontBuildExportStatic, self).on_execute()

        with MetatronEnv(self, nodejs_version=self.Parameters.node_version):
            self.__prepare_repos()
            self.__build()
            self.__dupload()

    def __prepare_repos(self):
        Git(self.__PACKAGES_REPO).clone(
            os.path.join(os.getcwd(), self.__PACKAGES_DIR),
            self.Parameters.packages_branch
        )

    def __build(self):
        packages_path = os.path.join(os.getcwd(), self.__PACKAGES_DIR)
        makedebpkg_config_path = os.path.join(packages_path, self.__MAKEDEBPKG_CONF)

        os.symlink(self.MAKEDEBPKG_PATH, os.path.join(packages_path, self.MAKEDEBPKG))

        package_dir = os.path.join(packages_path, self.__EXPORT_WWW_STATIC_DIR)

        os.environ.update({
            "rev": self.Parameters.app_branch,
            "PATH": makedebpkg_config_path + ":" + os.environ["PATH"],
            "CARCH": self.__CARCH,
        })

        logging.info("makedebpkg_env %s", os.environ)

        rich_check_call(
            [
                self.MAKEDEBPKG_PATH + "/makedebpkg", "--config", makedebpkg_config_path, "--log", "--sign",
                "--nocheck", "--key", os.environ["GPGKEY"]
            ],
            task=self, alias="makedebpkg", cwd=package_dir
        )

    def __dupload(self):
        packages_path = os.path.join(os.getcwd(), self.__PACKAGES_DIR)
        package_dir = os.path.join(packages_path, self.__EXPORT_WWW_STATIC_DIR)

        with sdk2.helpers.ProcessLog(self, "perl") as process_log:
            perl = sp.Popen(
                [
                    "perl",
                    os.path.join(self.__RESOURCES_DIR, "dupload_config_parser.pl"),
                    os.path.join(packages_path, "dupload.conf")
                ],
                stdout=sp.PIPE, stderr=process_log.stderr
            )

            (stdout, stderr) = perl.communicate()

            perl.wait()

            logging.info("perl stdout: %s", stdout)

            dupload_conf = json.loads(stdout)

        changes_ext = "_" + self.__CARCH + ".changes"
        changes_path = os.path.join(package_dir, "pkg")

        logging.debug(os.listdir(package_dir))

        for filename in os.listdir(changes_path):
            if filename.endswith(changes_ext):
                changes_path = os.path.join(changes_path, filename)

                with DebRelease(dupload_conf, login="robot-metatron"):
                    rich_check_call(
                        ["dupload", "--to", "verstka", str(changes_path)],
                        task=self, alias="dupload"
                    )

                self.set_info("duploaded: {}".format(filename))

                break
