# coding=utf-8

import tempfile
import os
import os.path
import pipes
import sandbox.sdk2.path as spath

from sandbox import sdk2
from sandbox.sdk2.vcs.git import Git
from sandbox.projects.common.nanny import nanny
from sandbox.projects.market.resources import MARKET_FULFILLMENT_YARD_FRONT
from sandbox.projects.market.front.MarketFrontNodejsBuildBase import MarketFrontNodejsBuildBase, RESOURCE_TYPES
from sandbox.projects.market.front.helpers.sandbox_helpers import rich_check_call
from sandbox.projects.market.front.helpers.ubuntu import setup_container
from sandbox.projects.market.front.helpers.MetatronEnv import MetatronEnv
from sandbox.projects.market.front.helpers.tsum import send_report_to_tsum


class MarketFrontBuildFulfillmentYard(nanny.ReleaseToNannyTask2, sdk2.Task):
    """
    Task to build market fullfilment front
    """

    RESOURCES_DIR = os.path.join(os.path.dirname(__file__), "..", "helpers")
    YENV = "production"
    NODE_ENV = "production"
    NPM_REGISTRY = "http://npm.yandex-team.ru"
    APP_SRC_DIR = "app_src"
    STATIC_CARCH = "all"
    MAKEDEBPKG_PATH = "/opt/makedebpkg"
    S3_BUCKET_NAME = "market-static"

    class Requirements(MarketFrontNodejsBuildBase.Requirements):
        pass

    class Parameters(MarketFrontNodejsBuildBase.Parameters):
        dist_dir = sdk2.parameters.String("Папка дистрибутива сборки", required=True, default="dist")
        app_repo = sdk2.parameters.String("Ссылка на git репозиторий", required=True, default="https://github.yandex-team.ru/market/fulfillment-yard-front.git")

    def on_enqueue(self):
        super(MarketFrontBuildFulfillmentYard, self).on_enqueue()
        setup_container(self)

    @property
    def app_repo(self):
        return self.Parameters.app_repo if self.Parameters.app_repo else "https://github.yandex-team.ru/market/fulfillment-yard-front.git"

    @property
    def resource_type(self):
        return MARKET_FULFILLMENT_YARD_FRONT

    @property
    def dist_dir(self):
        """
        Папка дистрибутива приложения, относительно корня репозитория
        """
        return self.Parameters.dist_dir if self.Parameters.dist_dir else None

    def make_dist_path(self, app_src_path):
        return app_src_path if not self.dist_dir else os.path.join(app_src_path, self.dist_dir)

    @property
    def make_env(self):
        make_env = os.environ.copy()
        make_env["YENV"] = self.YENV
        make_env["NODE_ENV"] = self.NODE_ENV

        if self.Parameters.common_branch:
            make_env.update({
                "COMMON_SCHEME": self.Parameters.common_scheme,
                "COMMON_REPO": self.Parameters.common_repo,
                "COMMON_REV": self.Parameters.common_branch,
            })

        if self.dist_dir:
            make_env["DIST_DIR"] = self.dist_dir

        if self.Parameters.prune:
            make_env["PRUNE"] = "1"

        return make_env

    def _git_clone(self, url, ref, target_dir='.'):
        git = Git(url)
        git.clone(str(target_dir), ref)

    def _prepare_repos(self, app_src_path):
        MarketFrontNodejsBuildBase._git_clone(
            pipes.quote(self.app_repo),
            pipes.quote(self.Parameters.app_branch),
            app_src_path
        )

    def _install_packages(self, app_src_path):
        rich_check_call(
            ["npm", "install"],
            task=self, alias="npm_install", cwd=app_src_path, env=self.make_env
        )

    def _app_build(self, app_src_path):
        rich_check_call(
            ["npm", "run", "build"],
            task=self, alias="npm_build", cwd=app_src_path, env=self.make_env
        )

    def _prune_deps(self, app_src_path):
        if self.Parameters.prune:
            rich_check_call(
                ["npm", "prune", "--production"],
                task=self, alias="npm_prune", cwd=app_src_path
            )

    def _create_archive_resource(self, pack_app_archive_path):
        resource_type = RESOURCE_TYPES[self.resource_type]
        app_res = sdk2.ResourceData(resource_type(
            self, "App tarball", "app.tar.gz"
        ))

        if app_res.path.exists():
            app_res.path.unlink()

        app_res.path.write_bytes(spath.Path(pack_app_archive_path).read_bytes())
        app_res.ready()

    def _create_html_resource(self, dist_path):
        resource_type = RESOURCE_TYPES[self.resource_type]
        html_res = sdk2.ResourceData(resource_type(
            self, "HTML", "index.html"
        ))

        if html_res.path.exists():
            html_res.path.unlink()

        html_res.path.write_bytes(spath.Path(dist_path / "index.html").read_bytes())
        html_res.ready()

    def _create_app_pack(self, dist_path):
        pack_app_archive_path = tempfile.mktemp(suffix=".tar.gz", prefix="app")

        rich_check_call(
            ["tar", "-czf", pack_app_archive_path, dist_path],
            task=self, alias="create_app_pack"
        )

        self._create_archive_resource(pack_app_archive_path)

        return pack_app_archive_path

    def on_execute(self):
        with MetatronEnv(self, nodejs_version=self.Parameters.node_version):
            root_dir = tempfile.mkdtemp()
            packages_path = os.path.join(root_dir, "packages")
            app_src_path = os.path.join(packages_path, self.APP_SRC_DIR)

            self._prepare_repos(app_src_path)
            self._install_packages(app_src_path)
            self._app_build(app_src_path)
            self._prune_deps(app_src_path)

            dist_path = self.make_dist_path(app_src_path)

            pack_app_archive_path = self._create_app_pack(dist_path)
            if self.Parameters.tsum_send_report:
                resource = sdk2.Resource.find(
                    resource_type=self.resource_type,
                    task_id=self.id
                ).first()
                with sdk2.helpers.ProcessLog(self, "tsum-report"):
                    send_report_to_tsum(
                        archive=pack_app_archive_path,
                        sandbox_resource_type=self.resource_type,
                        sandbox_ticket_num=self.id,
                        sandbox_resource_id=getattr(resource, "id", None),
                        tsum_api_hosts=self.Parameters.tsum_api_hosts
                    )
