# coding=utf-8
import os

from sandbox import sdk2
from sandbox import common
import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt

from sandbox.projects.sandbox import sandbox_lxc_image, LXC_CONTAINER
from sandbox.projects.sandbox_ci import SANDBOX_CI_LXC_IMAGE


class MarketFrontBuildLxc(sdk2.Task):
    """
    Task to build LXC image for MarketFrontBuildUnified
    """

    RESOURCES_DIR = os.path.join(os.path.dirname(__file__), "resources")

    CUSTOM_PACKAGES_PATH = dict(
        trusty=os.path.join(RESOURCES_DIR, "custom_packages.trusty"),
        xenial=os.path.join(RESOURCES_DIR, "custom_packages.xenial"),
    )

    CUSTOM_REPOS_PATH = dict(
        trusty=os.path.join(RESOURCES_DIR, "custom_repos.trusty"),
        xenial=os.path.join(RESOURCES_DIR, "custom_repos.xenial"),
    )

    CUSTOM_SCRIPT = "custom_script.sh"
    CUSTOM_SCRIPT_PATH = os.path.join(RESOURCES_DIR, CUSTOM_SCRIPT)

    class Parameters(sdk2.Parameters):
        with sdk2.parameters.String("Версия убунту", required=True, multiline=True) as ubuntu_release:
            ubuntu_release.values[sandbox_lxc_image.UbuntuRelease.TRUSTY] = ubuntu_release.Value('TRUSTY (14)', default=True)
            ubuntu_release.values[sandbox_lxc_image.UbuntuRelease.XENIAL] = ubuntu_release.Value('XENIAL (16)')
            ubuntu_release.values[sandbox_lxc_image.UbuntuRelease.BIONIC] = ubuntu_release.Value('BIONIC (18)')

        test_result_lxc = sandbox_lxc_image.SandboxLxcImage.Parameters.test_result_lxc(default=True)
        build_restarts = sdk2.parameters.Integer("Max build restarts", default=3, required=True)
        kill_timeout = build_restarts.default * 15 * 60
        max_restarts = 1
        with sdk2.parameters.String("Тип ресурса контейнера", required=True, multiline=True) as resource_type:
            resource_type.values[LXC_CONTAINER.name] = resource_type.Value(LXC_CONTAINER.name, default=True)
            resource_type.values[SANDBOX_CI_LXC_IMAGE.name] = resource_type.Value(SANDBOX_CI_LXC_IMAGE.name)

    class Context(sdk2.Context):
        child_tasks_ids = []

    class Requirements(sdk2.Requirements):
        dns = ctm.DnsType.DNS64
        disk_space = 100

    @common.utils.singleton_property
    def _notifications(self):
        return self.server.task[self.id].read()["notifications"]

    def on_execute(self):
        subtasks = list(self.find().limit(0))
        if subtasks:
            failed_subtasks = [subtask for subtask in subtasks if subtask.status not in ctt.Status.Group.SUCCEED]
            if failed_subtasks:
                raise common.errors.TaskFailure("Failed subtasks: {}".format(failed_subtasks))

            return  # success

        parameters = {}
        with open(self.CUSTOM_PACKAGES_PATH[self.Parameters.ubuntu_release]) as custom_packages:
            parameters[sandbox_lxc_image.SandboxLxcImage.Parameters.custom_packages.name] = custom_packages.read()

        with open(self.CUSTOM_REPOS_PATH[self.Parameters.ubuntu_release]) as custom_repos:
            parameters[sandbox_lxc_image.SandboxLxcImage.Parameters.custom_repos.name] = custom_repos.read()

        with open(self.CUSTOM_SCRIPT_PATH) as custom_script:
            parameters[sandbox_lxc_image.SandboxLxcImage.Parameters.custom_script.name] = custom_script.read()

        build_task = sandbox_lxc_image.SandboxLxcImage(
            self,
            description="Child of {} task {}".format(self.__class__.__name__, self.id),
            max_restarts=self.Parameters.build_restarts,
            custom_image=True,
            ubuntu_release=self.Parameters.ubuntu_release,
            test_result_lxc=self.Parameters.test_result_lxc,
            resource_type=self.Parameters.resource_type,
            **parameters
        )

        raise sdk2.WaitTask(
            build_task.enqueue().id,
            tuple(ctt.Status.Group.FINISH) + tuple(ctt.Status.Group.BREAK)
        )
