# coding=utf-8

import os
import logging

from sandbox import sdk2
from sandbox.projects.market.resources import MARKET_FRONT_APP_BUNDLE_STATS
from sandbox.projects.market.front.helpers.github import clone_repo
from sandbox.projects.market.front.helpers.sandbox_helpers import rich_check_call
from sandbox.projects.market.front.helpers.node import create_node_selector
from sandbox.projects.market.front.helpers.ubuntu import create_ubuntu_selector, setup_container
from sandbox.projects.market.front.helpers.path import raise_if_path_not_exists
from sandbox.projects.market.front.helpers.MetatronEnv import MetatronEnv
import sandbox.common.types.misc as ctm

DISK_SPACE = 6 * 1024

class MarketFrontBundleStats(sdk2.Task):
    """
    Таска для создания ресурса со статистикой по webpack бандлам
    """

    APP_REPO_DIR = "app_src"

    class Parameters(sdk2.Task.Parameters):
        ubuntu_version = create_ubuntu_selector()
        node_version = create_node_selector()

        github_owner = sdk2.parameters.String(
            "Github owner",
            required=True
        )

        github_repo = sdk2.parameters.String(
            "Github repo",
            required=True
        )

        github_ref = sdk2.parameters.String(
            "Github branch",
            required=True
        )

        app_id = sdk2.parameters.String(
            "Идентификатор приложения",
            description="Идентификатор нужен для поиска ресурса со статистикой и "
                        "дальнейшего сравнения его в статоскопе с актуальными данными. "
                        "Например, market_touch",
            required=True
        )

        app_root_dir = sdk2.parameters.String(
            "Кастомный путь корня приложения внутри репозитория"
        )

        stats_dir = sdk2.parameters.String(
            "Путь до папки с отчетом статистики бандла от корня приложения",
            required=True
        )

        stats_filename = sdk2.parameters.String(
            "Название файла с отчетом статистики бандла",
            required=True
        )

        make_stats_command = sdk2.parameters.String(
            "Команда для генерации статистики бандла",
            description="Статистику можно собирать разными способами, "
                        "с помощью плагина https://github.com/webpack-contrib/webpack-bundle-analyzer, "
                        "либо просто вебпаком https://webpack.js.org/api/stats/",
            required=True
        )

    class Requirements(sdk2.Task.Requirements):
        dns = ctm.DnsType.DNS64
        disk_space = DISK_SPACE

    def on_enqueue(self):
        super(MarketFrontBundleStats, self).on_enqueue()
        setup_container(self)

    @property
    def _root_path(self):
        return str(self.path())

    @property
    def _app_repo_path(self):
        return os.path.join(self._root_path, self.APP_REPO_DIR)

    @property
    def _app_root_path(self):
        if self.Parameters.app_root_dir:
            return os.path.join(self._app_repo_path, self.Parameters.app_root_dir)

        return self._app_repo_path

    @property
    def _stats_path(self):
        return os.path.join(self._app_root_path, self.Parameters.stats_dir, self.Parameters.stats_filename)

    def _clone_repo(self):
        logging.info('Cloning repo')

        clone_repo(
            self.Parameters.github_owner,
            self.Parameters.github_repo,
            self.Parameters.github_ref,
            self._app_repo_path
        )

    def _create_stats(self):
        logging.info('Creating stats')

        rich_check_call(
            ["bash", "-c", self.Parameters.make_stats_command],
            self, "create_stats", cwd=self._app_root_path
        )

    def _create_stats_resource(self):
        logging.info('Creating stats resource')

        raise_if_path_not_exists(
            self._stats_path,
            "Can't create stats resource, {} not exists".format(self._stats_path)
        )

        resource = MARKET_FRONT_APP_BUNDLE_STATS(
            self, "Bundle stats", self._stats_path,
            app_id=self.Parameters.app_id,
            pack_tar = 2
        )
        resource_data = sdk2.ResourceData(resource)
        resource_data.ready()

    def on_execute(self):
        super(MarketFrontBundleStats, self).on_execute()

        with MetatronEnv(self, nodejs_version=self.Parameters.node_version):
            self._clone_repo()
            self._create_stats()
            self._create_stats_resource()
