# coding=utf-8

import os
import logging

from sandbox import sdk2
from sandbox.common.utils import get_task_link
from sandbox.common.types import resource as ctr
from sandbox.projects.market.resources import MARKET_FRONT_BUNDLE_STATS, MARKET_FRONT_BUNDLE_COMPARISON_REPORT
from sandbox.projects.market.front.MarketFrontBuildBundleStats import MarketFrontBuildBundleStats
from sandbox.projects.market.front.helpers.github import change_status, GitHubStatus, GitHubStatusDescription
from sandbox.projects.market.front.helpers.MetatronEnv import MetatronEnv
from sandbox.projects.sandbox_ci.utils.context import GitRetryWrapper
from sandbox.projects.market.front.helpers.sandbox_helpers import rich_check_call, get_resource_http_proxy_link
from sandbox.projects.market.front.helpers.path import makedirs_if_not_exists


class MarketFrontCIBundleComparison(MarketFrontBuildBundleStats):
    """
    Таска для сравнения статистик сборки в CI
    Для сравнения используется утилита https://statoscope.tech/
    """

    APP_REPO_DIR = "app_src"
    COMPARISON_DIR = "comparison_results"

    BASE_BUNDLE_STATS_DIR = "base_bundle_stats"
    ACTUAL_BUNDLE_STATS_DIR = "actual_bundle_stats"

    class Parameters(MarketFrontBuildBundleStats.Parameters):
        comparison_command = sdk2.parameters.String(
            "Command to compare",
            description="A command which will be executed to compare stats."
                        "It must includes three placeholders: {base_path}, {actual_path} and {output_path}."
                        "For example: 'npx statoscope generate -i {base_path} -i {actual_path} -o {output_path}'",
            required=True
        )

        github_context = sdk2.parameters.String(
            "Github Context",
            description="Context is using as a title in CI",
            required=True
        )

        commit_hash = sdk2.parameters.String(
            "Git commit hash",
            description="Commit hash which is used to set the status",
            required=True
        )

        comparison_report_path = sdk2.parameters.String(
            "Path to comparison report",
            required=True
        )

        skip_conflicting_stats = sdk2.parameters.Bool(
            'If False, the task will not raise exception on conflicting stats'
        )

    @property
    def _base_bundle_stats_dir(self):
        """
        Absolute path to a base bundle stats folder
        """
        return os.path.join(self._repo_path, self.BASE_BUNDLE_STATS_DIR)

    @property
    def _base_bundle_stats_path(self):
        """
        Absolute path to a base bundle stats JSON
        """
        return os.path.join(self._base_bundle_stats_dir, self.Parameters.bundle_stats_filename)

    @property
    def _actual_bundle_stats_path(self):
        """
        Absolute path to a actual bundle stats
        """
        return os.path.join(self._repo_path, self.ACTUAL_BUNDLE_STATS_DIR, self.Parameters.bundle_stats_filename)

    def _change_gh_status(self, status, report_url=None):
        if self.Parameters.commit_hash and self.Parameters.github_context:
            change_status(
                owner=self.Parameters.github_owner,
                repo=self.Parameters.github_repo,
                context="[Sandbox CI] {}".format(self.Parameters.github_context),
                sha=self.Parameters.commit_hash,
                url=report_url or get_task_link(self.id),
                state=status,
                description=GitHubStatusDescription[status]
            )

    def on_prepare(self):
        with MetatronEnv(self, nodejs_version=self.Parameters.node_version), GitRetryWrapper():
            self._change_gh_status(GitHubStatus.PENDING)

    def on_failure(self, prev_status):
        with MetatronEnv(self, nodejs_version=self.Parameters.node_version), GitRetryWrapper():
            self._change_gh_status(GitHubStatus.FAILURE)

    def on_execute(self):
        # super(MarketFrontCIBundleComparison, self).on_execute()

        with MetatronEnv(self, nodejs_version=self.Parameters.node_version), GitRetryWrapper():
            self._clone_repo()
            # self._update_repo()
            self._build_stats_artefacts()
            self._check_artefacts_exist()
            self._load_base_bundle_stats()
            self._move_actual_bundle_stats()
            self._create_comparison_report()

    def _update_repo(self):
        logging.info("Merging 'master' into branch")

        rich_check_call(
            ["bash", "-c", "git merge master --commit --no-edit"],
            task=self, alias="Updating branch", cwd=self._app_path
        )

    def _load_base_bundle_stats(self):
        logging.info("Loading base bundle stats")

        base_stats_resource = MARKET_FRONT_BUNDLE_STATS\
            .find(
                state=ctr.State.READY,
                attrs=dict(
                    app_ref=self.Parameters.app_ref,
                    bundle_ref=self.Parameters.bundle_ref
                )
            )\
            .order(-sdk2.Resource.id)\
            .first()

        if base_stats_resource is None:
            return

        makedirs_if_not_exists(self._base_bundle_stats_dir)

        resource_data = sdk2.ResourceData(base_stats_resource)

        logging.info("Unpacking base bundle stats resource")

        rich_check_call(
            ["tar", "-xzf", "{}".format(resource_data.path), "-C", self._base_bundle_stats_dir],
            task=self, alias="Unpack base bundle stats resource"
        )

    def _move_actual_bundle_stats(self):
        logging.info("Moving actual bundle stats artefacts")

        rich_check_call(
            ["mv", "-f", self._artefact_path, self._actual_bundle_stats_path],
            task=self, alias="Move actual bundle stats"
        )

    def _create_comparison_report_resource(self):
        resource = MARKET_FRONT_BUNDLE_COMPARISON_REPORT(
            task=self,
            description="Bundle comparison reports",
            path=self.Parameters.comparison_report_path
        )

        resource_data = sdk2.ResourceData(resource)
        resource_data.ready()

        return resource

    def _create_comparison_report(self):
        makedirs_if_not_exists(self._comparison_output_path)

        gh_status = GitHubStatus.SUCCESS

        if not os.path.isfile(self._base_bundle_stats_path):
            logging.info("{} is missing! Maybe a webpack config has been added?", self._base_bundle_stats_path)

            if not self.Parameters.skip_conflicting_stats:
                raise Exception("Conflicting stat was found! {} was not found!".format(self._base_bundle_stats_path))

        if not os.path.isfile(self._actual_bundle_stats_path):
            logging.info("{} is missing! Maybe a webpack config has been removed?", self._actual_bundle_stats_path)

            if not self.Parameters.skip_conflicting_stats:
                raise Exception("Conflicting stat was found! {} was not found!".format(self._actual_bundle_stats_path))

        logging.info("Base stats dir {}".format(self._base_bundle_stats_path))
        makedirs_if_not_exists(self._base_bundle_stats_path)
        os.rename(self._base_bundle_stats_path, "base_{}".format(self.Parameters.bundle_stats_filename))
        base_path = os.path.join(
            self._base_bundle_stats_dir, "base_{}".format(self.Parameters.bundle_stats_filename)
        )

        logging.info("Actual stats dir {}".format(self._actual_bundle_stats_path))
        makedirs_if_not_exists(self._actual_bundle_stats_path)
        os.rename(self._actual_bundle_stats_path, "actual_{}".format(self.Parameters.bundle_stats_filename))
        actual_path = os.path.join(
            self._actual_bundle_stats_path, "actual_{}".format(self.Parameters.bundle_stats_filename)
        )

        try:
            command = self.Parameters.comparison_command.format(
                base_path=base_path,
                actual_path=actual_path,
                output_path=self.Parameters.comparison_report_path
            )

            rich_check_call(
                ["bash", "-c", command],
                task=self, alias="Compare stats", cwd=self._app_path
            )
        except:
            gh_status = GitHubStatus.ERROR
        finally:
            resource = self._create_comparison_report_resource()
            self._change_gh_status(gh_status, get_resource_http_proxy_link(resource))
