# coding=utf-8

import logging
import json

from sandbox import common
import sandbox.common.types.misc as ctm
from sandbox import sdk2
from sandbox.sdk2 import Task, Vault
from sandbox.projects.sandbox_ci.utils.request import send_request
from sandbox.projects.market.front.helpers.ubuntu import create_ubuntu_selector, setup_container
from sandbox.projects.market.front.helpers.node import create_node_selector
from sandbox.projects.market.front.helpers.github import change_status
from sandbox.projects.market.front.helpers.ci import propagate_github_event_data, extract_ticket_id, is_release


class GitHubStatus(common.utils.Enum):
    SUCCESS = ("success",  "Перейти к пайплайну по ссылке →")
    PENDING = ("pending", "Задача в очереди")
    ERROR = ("error", "Ошибка при выполнении задачи")


class MarketFrontCiStartMt(Task):
    """
    Таска для запуска и перезапуска мультитестинга.


    Для добавления проверки в репозиторий, нужно добавить в конфиг sandbox-ci
    в секцию github_event_handlers/pull_request/sandbox_tasks подобный хэндлер:
    Для добавления проверки в репозиторий, нужно добавить JSON-файл с подобным содержимым:
    {
        "type": "MARKET_FRONT_CI_START_MT",
        "custom_fields": [
            {"name": "tsum_project", "value": "front-touch"},
            {"name": "tsum_pipeline_type", "value": "mt-touch-front"},
            {"name": "common_branch", "value": "develop"}   // для проектов с common
        ]
    }

    Помните, что нужен вебхук https://sandbox-ci.si.yandex-team.ru/v1/hooks/github на событие Pull request.
    И Content type у него должен быть application/json
    """

    GITHUB_TOKEN_VAULT_KEY = "robot-metatron-github-token"
    TSUM_TOKEN_VAULT_KEY = "robot-metatron-tsum-token"

    TSUM_MT_PAGE_URL = "https://tsum.yandex-team.ru/pipe/projects/{project}/multitestings/environments/{name}"
    TSUM_MT_API_URL = "https://tsum.yandex-team.ru/api/multitestings/project/{project}/environments/{name}/createOrRestart"

    class Parameters(Task.Parameters):
        github_owner = sdk2.parameters.String(
            "Github owner",
            default_value="market"
        )

        github_repo = sdk2.parameters.String(
            "Github repo"
        )

        commit_hash = sdk2.parameters.String(
            "Хэш коммита, которому проставляется статус"
        )

        pr_title = sdk2.parameters.String(
            "Заголовок PR"
        )

        head_branch = sdk2.parameters.String(
            "Ветка для сборки"
        )

        common_branch = sdk2.parameters.String(
            "Ветка common для сборки",
            default="(unused)"
        )

        tsum_project = sdk2.parameters.String(
            "Название проекта"
        )

        tsum_pipeline_type = sdk2.parameters.String(
            "Тип пайплайна"
        )

        ubuntu_version = create_ubuntu_selector()
        node_version = create_node_selector()

    class Requirements(Task.Requirements):
        dns = ctm.DnsType.DNS64

    class Context(Task.Context):
        ticket_id = None
        is_release = None

    def on_save(self):
        super(MarketFrontCiStartMt, self).on_save()
        propagate_github_event_data(self)

        self.Context.is_release = is_release(self.Parameters.pr_title)
        self.Context.ticket_id = extract_ticket_id(self.Parameters.pr_title)

    def on_enqueue(self):
        super(MarketFrontCiStartMt, self).on_enqueue()
        setup_container(self)

        if self.Context.is_release:
            # Не можем вычислить ссылку на релизный пайплайн, поэтому пропускаем.
            self.set_info("Релизный тикет, игнорируем")
            return

        if self.Context.ticket_id is None:
            self.set_info("Номер тикета не найден, игнорируем")
            return

        self._change_gh_status(GitHubStatus.PENDING)
        self.set_info("Пайплайн: {}".format(self._tsum_page_url))

    def on_execute(self):
        super(MarketFrontCiStartMt, self).on_execute()

        if self.Context.is_release:
            logging.debug("Found a release ticket")
            return

        if self.Context.ticket_id is None:
            logging.debug("The ticket number is not found")
            return

        self._create_or_restart_mt()

    def on_success(self, prev_status):
        super(MarketFrontCiStartMt, self).on_success(prev_status)

        self._change_gh_status(GitHubStatus.SUCCESS)

    def on_failure(self, prev_status):
        super(MarketFrontCiStartMt, self).on_failure(prev_status)

        self._change_gh_status(GitHubStatus.ERROR)

    def on_break(self, prev_status, status):
        super(MarketFrontCiStartMt, self).on_break(prev_status, status)

        self._change_gh_status(GitHubStatus.ERROR)

    @property
    def _github_context(self):
        return "[Sandbox CI] MT pipeline"

    @property
    def _task_url(self):
        return common.utils.get_task_link(self.id)

    @property
    def _tsum_mt_name(self):
        assert self.Context.ticket_id

        return self.Context.ticket_id.lower()

    @property
    def _tsum_page_url(self):
        assert not self.Context.is_release
        assert self.Context.ticket_id

        return self.TSUM_MT_PAGE_URL.format(
            project=self.Parameters.tsum_project,
            name=self._tsum_mt_name
        )

    @property
    def _tsum_api_url(self):
        return self.TSUM_MT_API_URL.format(
            project=self.Parameters.tsum_project,
            name=self._tsum_mt_name
        )

    def _change_gh_status(self, status):
        if not self.Parameters.commit_hash or len(str(self.Parameters.commit_hash)) == 0:
            logging.warning("В параметрах джобы не указан хэш коммита. Не выставляем статус.")
            return

        return change_status(
            owner=self.Parameters.github_owner,
            repo=self.Parameters.github_repo,
            context=self._github_context,
            sha=self.Parameters.commit_hash,
            url=self._tsum_page_url if status == GitHubStatus.SUCCESS else self._task_url,
            state=status[0],
            description=status[1]
        )

    def _create_or_restart_mt(self):
        token = Vault.data(self.TSUM_TOKEN_VAULT_KEY)

        headers = {
            "Authorization": token,
            "Content-Type": "application/json"
        }

        data = {
            "name": self._tsum_mt_name,
            "type": "USE_EXISTING_PIPELINE",
            "pipelineId": self.Parameters.tsum_pipeline_type,
            "resources": {
                "fd4f152f-9f11-4b91-8a35-e58503b6bdf6": {
                    "tickets": [self.Context.ticket_id]
                }
            }
        }

        if self.Parameters.tsum_pipeline_type in ["mt-partner-front", "mt-partner-front-prestable"]:
            data["resources"]["ffa8df00-2d1a-4a35-a0f4-b3f6887f7193"] = {
                "appBranch": {
                    "name": self.Parameters.head_branch
                }
            }

            data["resources"]["64b0e129-af5a-46b1-bd90-61b362b6de89"] = {
                "type": "JAVA",
                "key": "cocon_commit_id",
                "value": None
            }
        else:
            data["resources"]["ee218388-942a-4823-a07c-19fc2d18352f"] = {
                "appBranch": {
                    "name": self.Parameters.head_branch
                },
                "commonBranch": {
                    "name": self.Parameters.common_branch
                },
                "nodeVersion": str(self.Parameters.node_version)
            }

        res = send_request("post", self._tsum_api_url, headers=headers, data=json.dumps(data))

        res.raise_for_status()

        try:
            assert res.json()['name'] == self._tsum_mt_name
        except Exception:
            logging.error(u"Bad TSUM response: {}".format(res.text))

            raise Exception("Something is wrong, see logs")
