# coding=utf-8

import json
import os
import tempfile
import logging
import pipes
import urllib
import requests

from sandbox import sdk2
from sandbox.common.errors import TaskError
from sandbox.common.utils import get_task_link
import sandbox.common.types.misc as ctm
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.sandboxsdk.errors import SandboxTaskFailureError

from sandbox.projects.market.front.helpers.sandbox_helpers import rich_check_call, \
    get_resource_http_proxy_link as get_resource_link
from sandbox.projects.market.front.helpers.node import create_node_selector
from sandbox.projects.market.front.helpers.ubuntu import create_ubuntu_selector, setup_container
from sandbox.projects.market.front.helpers.MetatronEnv import MetatronEnv
from sandbox.projects.market.front.helpers.startrack import ReleaseTicket, build_result_cms
from sandbox.projects.market.front.helpers.github import GitHubStatus, change_status, \
    clone_repo, GitHubStatusDescription

APP_REPO_OWNER = "market"
APP_SRC_DIR = "app_src"

CMS_URL = "http://mbo-cms-api.vs.market.yandex.net/schema/save"
CMS_FLOW_GRABBER_CONFIG_PATH = "configs/cms-flow-grabber/{platform}.json"
CMS_TIMEOUT = 120.0

GITHUB_CONTEXT = "JSON Schema"

USER_ID = 592370828


class GenericParameters(sdk2.Task.Parameters):
    with sdk2.parameters.Group("Приложение") as app:
        github_owner = sdk2.parameters.String(
            "Организация на GitHub",
            default_value=APP_REPO_OWNER,
            required=True
        )

        github_repo = sdk2.parameters.String(
            "Имя репозитория",
            required=True
        )

        head_branch = sdk2.parameters.String(
            "Ветка",
            default_value="master",
            required=True
        )

        app_root_path = sdk2.parameters.String(
            "Кастомный путь корня приложения внутри репозитория",
            default_value="",
            required=False
        )

        service = sdk2.parameters.String(
            "Идентификатор сервиса в CMS",
            description='Пример: "MarketNode"'
        )

        platform = sdk2.parameters.String(
            "Платформа",
            default_value="desktop",
            required=True
        )

        issue = ReleaseTicket(
            'Тикет',
            description='Пример: "MARKETVERSTKA-12345"'
        )

        ubuntu_version = create_ubuntu_selector()
        node_version = create_node_selector()

    with sdk2.parameters.Group("GitHub") as Github:
        send_hook = sdk2.parameters.Bool(
            "Посылать отчет",
            default_value=False,
            required=True
        )

        with send_hook.value[True]:
            github_sha = sdk2.parameters.String(
                "Хеш коммита",
                default_value="",
                required=True
            )

            github_context = sdk2.parameters.String(
                "Контекст",
                default_value=GITHUB_CONTEXT,
                required=True
            )

    with sdk2.parameters.Group("Отладка") as Dev:
        debug = sdk2.parameters.Bool(
            "Я программист",
            default=False,
            required=True
        )

        with debug.value[True]:
            cms_url = sdk2.parameters.String(
                "CMS URL",
                default_value=CMS_URL,
                required=True
            )

            cms_request_timeout = sdk2.parameters.String(
                "HTTP Request timeout (s)",
                default_value=CMS_TIMEOUT,
                required=True
            )

            project_conf = sdk2.parameters.String(
                "CMS Flow Grabber config path",
                default_value=CMS_FLOW_GRABBER_CONFIG_PATH,
                required=True
            )

            user_id = sdk2.parameters.String(
                "Passport UID",
                default_value=USER_ID,
                required=True
            )


class MarketFrontCMS(sdk2.Task):
    """
    Task to save JSON Schema to CMS
    """
    YENV = "production"
    _report_url = None

    class Parameters(GenericParameters):
        pass

    class Requirements(sdk2.Task.Requirements):
        dns = ctm.DnsType.DNS64
        environments = [
            PipEnvironment('yandex_tracker_client', version="1.3", custom_parameters=["--upgrade-strategy only-if-needed"]),
            PipEnvironment('startrek_client', version="2.3.0", custom_parameters=["--upgrade-strategy only-if-needed"])
        ]

    @property
    def _task_url(self):
        return get_task_link(self.id)

    @property
    def _github_context(self):
        return "[Sandbox CI] {}".format(self.Parameters.github_context or GITHUB_CONTEXT)

    def on_enqueue(self):
        super(MarketFrontCMS, self).on_enqueue()
        setup_container(self)

        self._change_gh_status(GitHubStatus.PENDING)

    def on_execute(self):
        super(MarketFrontCMS, self).on_execute()

        with MetatronEnv(self, nodejs_version=self.Parameters.node_version):
            root_dir = tempfile.mkdtemp()
            app_src_path = os.path.join(root_dir, APP_SRC_DIR)
            app_root_path = app_src_path

            if self.Parameters.app_root_path:
                app_root_path = os.path.join(app_src_path, self.Parameters.app_root_path)

            self._prepare_repos(app_src_path, app_root_path)

            schema = self._build_schema(app_root_path)
            result = self._send_schema_to_cms(schema)

            log_file = self._save_result_to_logs(result)

            self._report_url = self._get_report_url(log_file)

            report = build_result_cms(result, self._task_url, self._report_url)

            self._send_report(report)

            if not result["success"]:
                self._change_gh_status(GitHubStatus.ERROR)

    def on_success(self, prev_status):
        super(MarketFrontCMS, self).on_success(prev_status)

        self._change_gh_status(GitHubStatus.SUCCESS)

    def on_failure(self, prev_status):
        super(MarketFrontCMS, self).on_failure(prev_status)

        self._change_gh_status(GitHubStatus.FAILURE)

    def on_break(self, prev_status, status):
        super(MarketFrontCMS, self).on_break(prev_status, status)

        self._change_gh_status(GitHubStatus.FAILURE)

    def _prepare_repos(self, app_src_path, app_root_path):
        # TODO: убрать после выкладки https://github.yandex-team.ru/market-infra/tsum/pull/3610
        repo = self.Parameters.github_repo
        user_branch = self.Parameters.head_branch
        branch = "develop" if repo == "mobile" and user_branch == "master" else user_branch
        clone_repo(
            pipes.quote(self.Parameters.github_owner),
            pipes.quote(repo),
            pipes.quote(branch),
            app_src_path
        )

        rich_check_call(
            ["npm", "run", "bootstrap"],
            task=self, alias="bootstrap", cwd=app_root_path
        )

    def _build_schema(self, app_root_path):
        schema_path = os.path.join(str(self.log_path()), "cms-flow-grabber.json")
        command = [
            "npm",
            "run",
            "ci:cms",
            "--",
            "-c",
            self.Parameters.project_conf.format(platform=self.Parameters.platform),
            "-o",
            schema_path
        ]

        logging.debug("Start grubbing types: {}".format(" ".join(command)))

        rich_check_call(command, task=self, alias="cms-flow-grabber", cwd=app_root_path)

        if not os.path.isfile(schema_path):
            raise TaskError("JSON Schema not found from path {}".format(schema_path))

        with open(schema_path, "r") as file:
            data = file.read()

        return json.loads(data)

    def _send_schema_to_cms(self, data):
        uniq_service_id = self.Parameters.service if self.Parameters.service else self.Parameters.github_repo

        params = {
            "userId": self.Parameters.user_id,
            "service": uniq_service_id,
            "platform": self.Parameters.platform,
            "branch": self.Parameters.head_branch
        }

        logging.debug("Post cms-flow-grabber results to CMS from URL: {}?{}".format(self.Parameters.cms_url,
                                                                                    urllib.urlencode(params)))

        response = requests.post(
            url=self.Parameters.cms_url,
            json=data,
            params=params,
            timeout=float(self.Parameters.cms_request_timeout)
        )

        try:
            response_data = response.json()
        except ValueError:
            self.set_info("Unable to parse CMS response as JSON")
            self.set_info("Code: {}\nResponse: {}".format(response.status_code, response.text))
            raise SandboxTaskFailureError("CMS responded with invalid JSON")

        if response.status_code != requests.codes.ok:
            self.set_info("CMS responded with errors:")
            try:
                if response_data["errors"]:
                    for error in response_data["errors"]:
                        message = ""
                        if error["message"]:
                            message += error["message"]
                        if error["ref"]:
                            message += "\nref: " + error["ref"]
                        if message:
                            self.set_info(message)
            except KeyError:
                self.set_info("Code: {}\nResponse: {}".format(response.status_code, response.text))
            raise SandboxTaskFailureError("CMS responded with errors")

        return response_data

    def _save_result_to_logs(self, report):
        log_file = "cms_report.json"
        report_path = os.path.join(str(self.log_path()), log_file)

        with open(report_path, 'w') as file:
            json.dump(report, file)
            file.close()

        logging.debug("Report was saved to {}".format(report_path))

        return log_file

    def _send_report(self, report):
        if not len(self.Parameters.issue):
            return

        oauth_token = sdk2.Vault.data("robot-metatron-st-token")

        from startrek_client import Startrek

        st = Startrek(useragent="robot-metatron", token=oauth_token)
        issue = st.issues[self.Parameters.issue]

        issue.comments.create(text=report)

        logging.debug("Report was sent to {}".format(self.Parameters.issue))

    def _change_gh_status(self, status):
        if not self.Parameters.send_hook:
            return

        return change_status(
            owner=self.Parameters.github_owner,
            repo=self.Parameters.github_repo,
            context=self._github_context,
            sha=self.Parameters.github_sha,
            url=self._report_url or self._task_url,
            state=status,
            description=GitHubStatusDescription[status]
        )

    def _get_report_url(self, log_file):
        resource_url = get_resource_link(self.log_resource)

        return "{}/{}".format(resource_url, log_file)
