# coding=utf-8

from sandbox.projects.market.front.helpers.MetatronEnv import MetatronEnv
from sandbox.projects.market.front.helpers.sandbox_helpers import rich_check_call
from sandbox.projects.market.front.helpers.ubuntu import create_ubuntu_selector, setup_container
from sandbox.projects.market.front.helpers.node import create_node_selector
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.sdk2.vcs.git import Git
import sandbox.common.types.misc as ctm

import datetime
import json
import logging
import os.path
from sandbox import sdk2


class MarketFrontCollectCodeStats(sdk2.Task):
    """
    Собирает метрики по коду проекта и отправляет их в Statface.
    В качестве параметров надо указать репо и ветку проекта из github.yandex-team.ru.
    Так же надо указать путь внутри исходников проекта до манифеста, описывающего
    собираемые метрики. Манифест должен быть json-файлом со следующей стрктурой:

    [
        {
            "statfaceReport": "Market_Partner/CodeHealth/Coverage",
            "collectScript": "make configure && make bootstrap && npm run report:coverage",
            "resultFile": "reports/coverage.json"
        },
    ]

    , где statfaceReport - путь до отчета в statface, collectScript - скрипт из репо проекта,
    собирающей метрики, resultFile - файл с результатами работы collectScript-а. resultFile должен
    быть json-файлом с мапой вида "поле statface-очета" -> "значение поля", или массивом таких мап.
    Так для отчета Market_Partner/CodeHealth/Coverage, resultFile выглядит селующим образом:

    {"lines":0.77,"functions":1.6,"branches":0.6,"statements":0.71}

    а для отчета Market_Partner/CodeHealth/CLOC, resultFile выглядит селующим образом:
    [
    {"repository":"URL","lang":"JavaScript","files":1000,"blank":2,"comment":3,"code":100500},
    {"repository":"URL","lang":"JSON","files":20,"blank":0,"comment":0,"code":100},
    ]

    """

    APP_SRC_DIR = "app_src"

    GITHUB_TOKEN_VAULT_KEY = "robot-metatron-github-token"

    STATFACE_TOKEN_VAULT_KEY = "robot-metatron-statbox-token"

    class Parameters(sdk2.Task.Parameters):
        github_url = sdk2.parameters.String(
            "URL репозитория, вида https://github.yandex-team.ru/market/partnernode.git"
        )

        github_branch = sdk2.parameters.String(
            "Ветка в github-репозитории",
            default_value="master"
        )

        app_root_path = sdk2.parameters.String(
            "Кастомный путь корня приложения внутри репозитория",
            default_value="",
            required=False
        )

        stats_manifest = sdk2.parameters.String(
            "Путь до манифеста с описанием собираемых метрик"
        )

        ubuntu_version = create_ubuntu_selector()
        node_version = create_node_selector()

    class Requirements(sdk2.Task.Requirements):
        dns = ctm.DnsType.DNS64
        environments = [
            PipEnvironment("python-statface-client", "0.78.0", use_wheel=True),
        ]

    def on_enqueue(self):
        super(MarketFrontCollectCodeStats, self).on_enqueue()
        setup_container(self)

    def on_execute(self):
        super(MarketFrontCollectCodeStats, self).on_execute()

        with MetatronEnv(self, nodejs_version=self.Parameters.node_version):
            self._checkout_repo()
            self._collect_metrics()

    def _checkout_repo(self):
        logging.info("Checking out sources from {}".format(self.Parameters.github_url))
        git = Git(self.Parameters.github_url)
        git.clone(self.APP_SRC_DIR, self.Parameters.github_branch)

    def _collect_metrics(self):
        if self.Parameters.app_root_path:
            config_path = os.path.join(self.APP_SRC_DIR, self.Parameters.app_root_path, self.Parameters.stats_manifest)
        else:
            config_path = os.path.join(self.APP_SRC_DIR, self.Parameters.stats_manifest)

        with open(config_path) as file:
            config = json.load(file)
            for metric_config in config:
                self._collect_code_metric(
                    metric_config["statfaceReport"],
                    metric_config["collectScript"],
                    metric_config["resultFile"],
                )

    def _collect_code_metric(self, statface_report, collect_script, result_file):
        logging.info("Collecting data for report {}".format(statface_report))
        alias = "collect_{}".format(statface_report.lower().replace("/", "_"))

        if self.Parameters.app_root_path:
            app_root_dir = os.path.join(self.APP_SRC_DIR, self.Parameters.app_root_path)
        else:
            app_root_dir = self.APP_SRC_DIR

        rich_check_call(["bash", "-c", collect_script], self, alias, cwd=app_root_dir)

        result_file_path = os.path.join(app_root_dir, result_file)

        with open(result_file_path) as result_file:
            result = json.load(result_file)
            if not isinstance(result, list):
                result = [result]

            today = datetime.datetime.now().strftime('%Y-%m-%d')
            for row in result:
                row['fielddate'] = today
            self._send_to_statface(statface_report, result)

    def _send_to_statface(self, report_id, data):
        from statface_client import StatfaceClient, STATFACE_PRODUCTION
        logging.info("Uploading data to {}".format(report_id))
        oauth_token = sdk2.Vault.data(self.STATFACE_TOKEN_VAULT_KEY)
        client = StatfaceClient(oauth_token=oauth_token, host=STATFACE_PRODUCTION)
        report = client.get_report(report_id)
        report.upload_data("daily", data)
