# coding=utf-8

import sandbox.common.types.misc as ctm
from sandbox import sdk2
import os

from sandbox.projects.common import task_env
from sandbox.projects.market.front.helpers.MetatronEnv import MetatronEnv
from sandbox.projects.market.front.helpers.sandbox_helpers import rich_check_call
from sandbox.projects.market.front.helpers.ubuntu import create_ubuntu_selector, setup_container
from sandbox.projects.market.front.helpers.node import create_node_selector
from sandbox.projects.market.front.helpers.github import clone_repo
from sandbox.projects.market.resources import MARKET_FRONT_ENVY_SUMMARY


class MarketFrontEnvy(sdk2.Task):
    """
    Таска для запуска envy.
    Используется для обновления версии библиотек во всех продуктовых репозиториях
    """

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 1200

        packages = sdk2.parameters.Dict("Список библиотек и их версий на которые требуется обновить", required=True)
        ticket = sdk2.parameters.String("Номер задачи", required=True)
        contour = sdk2.parameters.String("Контур", default_value="platform")
        component = sdk2.parameters.String("Компонент", default_value="@Common")
        ref_name = sdk2.parameters.String("ref в репозхитории envy", default_value="master")

        ubuntu_version = create_ubuntu_selector()
        node_version = create_node_selector()

        with sdk2.parameters.Output:
            output = sdk2.parameters.Resource("Отчёт envy об обновлении", resource_type=MARKET_FRONT_ENVY_SUMMARY)

    class Requirements(task_env.TinyRequirements):
        dns = ctm.DnsType.DNS64
        disk_space = 20 * 1024

    def on_enqueue(self):
        super(MarketFrontEnvy, self).on_enqueue()
        setup_container(self)

    def on_execute(self):
        with MetatronEnv(task=self, nodejs_version=self.Parameters.node_version):
            root_dir = os.path.join(str(self.path().absolute()), 'envy')

            clone_repo(
                owner="market",
                name="envy",
                ref=self.Parameters.ref_name,
                target_dir=root_dir
            )

            rich_check_call(
                command=["npm", "ci"],
                task=self,
                alias="npmci",
                cwd=root_dir
            )

            # У ЦУМа левый сертификат
            os.environ["NODE_TLS_REJECT_UNAUTHORIZED"] = "0"

            res = MARKET_FRONT_ENVY_SUMMARY(self, "Envy summary", "summary.json")
            data = sdk2.ResourceData(res)

            params = [
                "-t", self.Parameters.ticket,
                "--contour", self.Parameters.contour,
                "--component", self.Parameters.component,
                "--summary", str(data.path),
            ]

            for (package, ver) in self.Parameters.packages.items():
                params.append('-i')
                params.append('{}@{}'.format(package, ver))

            rich_check_call(
                command=["npm", "run", "update", '--'] + params,
                task=self,
                alias="run_envy",
                cwd=root_dir
            )

            data.ready()
            self.Parameters.output = res
