# -*- coding: utf-8 -*-
import logging
import os
import pipes
import sandbox.common.types.misc as ctm

from sandbox import sdk2
from sandbox.projects.market.front.helpers.MetatronEnv import MetatronEnv
from sandbox.projects.market.front.helpers.github import clone_repo
from sandbox.projects.market.front.helpers.sandbox_helpers import rich_check_call
from sandbox.projects.market.front.helpers.ubuntu import create_ubuntu_selector, setup_container
from sandbox.projects.market.front.helpers.node import NODE_DEFAULT
from sandbox.projects.market.front.helpers.node import create_node_selector
from sandbox.projects.sandbox_ci.utils import env

GH_OWNER = 'MARKET'

class MarketFrontGenerateManualSkipRun(sdk2.Task):
    """
    Таска для фомирования ручного рана со скипнутыми АТ
    """

    ROOT_DIR = ''
    APP_SRC_PATH = ''

    class Requirements(sdk2.Task.Requirements):
        dns = ctm.DnsType.DNS64

    class Parameters(sdk2.Task.Parameters):
        ubuntu_version = create_ubuntu_selector()
        node_version = create_node_selector()

        with sdk2.parameters.Group('GitHub репозиторий проекта') as github_repo_block:
            app_owner = sdk2.parameters.String(
                'GitHub owner',
                description='Логин владельца репозитория или название организации',
                default=GH_OWNER,
                required=True
            )
            app_repo = sdk2.parameters.String(
                "Репозиторий",
                default='partnernode',
                required=True
            )
            app_branch = sdk2.parameters.String(
                "Ветка",
                default='master',
                required=True
            )

        with sdk2.parameters.Group('Testpalm') as testpalm_block:
            project_id = sdk2.parameters.String(
                "Проект",
                default='marketmbi',
                required=True
            )
            testcase_id = sdk2.parameters.String(
                "Test suite",
                default='5c63d73dbb580f09261cba51',
                required=True
            )

        with sdk2.parameters.Group('Environment') as environ_block:
            environ = sdk2.parameters.Dict('Environment variables')

    def on_enqueue(self):
        super(MarketFrontGenerateManualSkipRun, self).on_enqueue()
        setup_container(self)

    @property
    def node_version(self):
        return self.Parameters.node_version or NODE_DEFAULT

    @property
    def project_name(self):
        return self.Parameters.app_repo

    @property
    def report_description(self):
        return self.project_name

    @property
    def TESTPALM_API_TOKEN(self):
        return os.environ["TESTPALM_OAUTH_API_TOKEN"]

    @sdk2.header()
    def header(self):
        url = 'https://testpalm.yandex-team.ru/{projectId}/testsuite/{testcaseId}'.format(
            projectId=self.Parameters.project_id,
            testcaseId=self.Parameters.testcase_id
        )
        return '<a href="{url}" target="_blank">{url}</a>'.format(url=url)

    def _clone_repo(self):
        clone_repo(
            pipes.quote(self.Parameters.app_owner),
            pipes.quote(self.Parameters.app_repo),
            pipes.quote(self.Parameters.app_branch),
            self.APP_SRC_PATH
        )

    def _prepare_dirs(self):
        self.ROOT_DIR = str(self.path())
        self.APP_SRC_PATH = os.path.join(
            self.ROOT_DIR, self.Parameters.app_repo)

    def _prepare_env(self):
        os.environ['TESTPALM_ACCESS_TOKEN'] = self.TESTPALM_API_TOKEN
        os.environ['TESTPALM_PROJECT_ID'] = self.Parameters.project_id
        os.environ['TESTPALM_TESTCASE_ID'] = self.Parameters.testcase_id
        env.export(self.Parameters.environ)

    def _configure(self):
        rich_check_call(
            ["make", "configure"],
            task=self, alias="configure", cwd=self.APP_SRC_PATH
        )

    def _build(self):
        logging.info(
            'Running project build with environment: {}'.format(os.environ))

        rich_check_call(
            ["make", "bootstrap"],
            task=self, alias="bootstrap", cwd=self.APP_SRC_PATH
        )

    def _generate_manual_run(self):
        logging.info(
            'Generating manual testpalm run with skipped tests'
        )

        rich_check_call(
            ["npm", "run", "generate-skip-run"],
            task=self, alias="generate-skip-run", cwd=self.APP_SRC_PATH
        )

    def on_execute(self):
        with MetatronEnv(self, nodejs_version=self.Parameters.node_version):
            self._prepare_dirs()
            self._prepare_env()
            self._clone_repo()
            self._configure()
            self._build()
            self._generate_manual_run()
