# coding=utf-8

import json
import logging

from sandbox import sdk2
from sandbox.sdk2 import Task, Vault
from sandbox.projects.common import task_env
from sandbox.projects.market.front.helpers.ubuntu import create_ubuntu_selector, setup_container
from sandbox.projects.market.front.helpers.node import create_node_selector
import sandbox.common.types.misc as ctm
from sandbox.projects.sandbox_ci.utils.request import send_request


def safe_parse_json(serialized_data):
    try:
        return json.loads(serialized_data)
    except Exception:
        return serialized_data


def safe_deserialize_dict_values(unknown_dict):
    updated_dict = dict()
    for k, v in unknown_dict.items():
        updated_dict[k] = safe_parse_json(v)

    return updated_dict


class MarketFrontLaunchTsumMT(Task):
    """
    Таска для запуска и перезапуска мультитестинга.

    Абсолютно абстрактная: умеет запускать любой МТ ЦУМа с любыми параметрами.
    """

    TSUM_TOKEN_VAULT_KEY = "robot-metatron-tsum-token"
    TSUM_MT_API_URL = "https://tsum.yandex-team.ru/api/multitestings/project/{project}/environments/{name}/createOrRestart"

    class Parameters(Task.Parameters):
        ubuntu_version = create_ubuntu_selector()
        node_version = create_node_selector()

        tsum_project_id = sdk2.parameters.String("Название проекта")
        tsum_pipeline_id = sdk2.parameters.String("Тип пайплайна")
        tsum_pipeline_name = sdk2.parameters.String("Ключ для группировки и перезапуска")
        tsum_pipeline_resources = sdk2.parameters.Dict("Ресурсы")

    class Requirements(task_env.TinyRequirements):
        dns = ctm.DnsType.DNS64

    def on_enqueue(self):
        super(MarketFrontLaunchTsumMT, self).on_enqueue()
        setup_container(self)

    def on_execute(self):
        super(MarketFrontLaunchTsumMT, self).on_execute()

        self._launch_or_relaunch_pipeline()

    @property
    def _tsum_api_url(self):
        return self.TSUM_MT_API_URL.format(
            project=self.Parameters.tsum_project_id,
            name=self.Parameters.tsum_pipeline_name
        )

    def _launch_or_relaunch_pipeline(self):
        headers = {
            "Authorization": Vault.data(self.TSUM_TOKEN_VAULT_KEY),
            "Content-Type": "application/json"
        }

        resources = safe_deserialize_dict_values(self.Parameters.tsum_pipeline_resources)

        if "ee218388-942a-4823-a07c-19fc2d18352f" in resources:
            resources["ee218388-942a-4823-a07c-19fc2d18352f"]['nodeVersion'] = str(self.Parameters.node_version)

        data = {
            "name": self.Parameters.tsum_pipeline_name,
            "type": "USE_EXISTING_PIPELINE",
            "pipelineId": self.Parameters.tsum_pipeline_id,
            "resources": resources
        }

        res = send_request(
            "post",
            self._tsum_api_url,
            headers=headers,
            data=json.dumps(data)
        )

        logging.debug(u"Request body: {}".format(data))
        try:
            res.raise_for_status()

            assert res.json()['name'] == self.Parameters.tsum_pipeline_name
        except Exception:
            logging.error(u"Bad TSUM response: {}".format(res.text))
            self.set_info(u"Bad TSUM response: {}".format(res.text))

            raise Exception("Something is wrong, see logs")

        url = "https://tsum.yandex-team.ru/pipe/projects/{}/multitestings/environments/{}".format(
            self.Parameters.tsum_project_id, self.Parameters.tsum_pipeline_name
        )
        status_message = 'Ссылка на пайплайн: <a href="{url}" target="_blank">{url}</a>'.format(url=url)
        self.set_info(status_message, do_escape=False)
