# coding=utf-8

import json
import logging
import os
import os.path
import pipes
import re
import shutil
import tempfile

from sandbox.sandboxsdk import environments
import sandbox.common.types.misc as ctm
from sandbox import sdk2
import sandbox.sdk2.path as spath
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.sdk2.vcs.git import Git
from sandbox.projects.common.debpkg import DebRelease
from sandbox.projects.common.nanny import nanny

import sandbox.projects.market.front.helpers.s3 as s3
from sandbox.projects.market.front.helpers.MetatronEnv import MetatronEnv
from sandbox.projects.market.front.helpers.startrack import FixVersion
from sandbox.projects.market.front.helpers.sandbox_helpers import rich_check_call, \
    rich_check_start, rich_check_report
from sandbox.projects.market.front.helpers.node import create_node_selector
from sandbox.projects.market.front.helpers.ubuntu import create_ubuntu_selector, setup_container, UBUNTU_TRUSTY
from sandbox.projects.market.front.helpers.tsum import send_report_to_tsum
from sandbox.projects.market.front.resources import MarketFrontBuildUnifiedStaticResource


class GenericParameters(sdk2.Task.Parameters):
    copy_static = sdk2.parameters.Bool("Копировать фриз статики в приложение", required=True, default=True)
    build_static = sdk2.parameters.Bool("Проводить сборку статического пакета", required=True, default=True)
    with build_static.value[True]:
        fix_version = FixVersion("Фикс-версия релиза в формате YYYY.R.PS", required=True)

    with sdk2.parameters.Group("Приложение") as app:
        need_merge_commit = sdk2.parameters.Bool("Нужен merge-коммит", required=False, default=True)
        app_repo = sdk2.parameters.String("Репозиторий")
        app_src_dir = sdk2.parameters.String("Кастомный путь корня приложения внутри репозитория")
        app_branch = sdk2.parameters.String(
            "Ветка репозитория приложения",
            default_value="",
            required=False
        )
        app_commit = sdk2.parameters.String(
            "Коммит",
            default_value="",
            required=False
        )
        dist_dir = sdk2.parameters.String("Папка дистрибутива сборки")

    """
    Протокол для работы с репозиторием common
    """
    common_scheme = ""

    """
    URI репозитория common (без протокола)
    """
    common_repo = ""

    with sdk2.parameters.Group("common") as common:
        common_branch = sdk2.parameters.String(
            "Ветка репозитория common",
            default_value="",
            required=False
        )

    with sdk2.parameters.Group("ЦУМ") as tsum:
        tsum_send_report = sdk2.parameters.Bool(
            "Отсылать отчет о приложении",
            required=True,
            default=False
        )

        with tsum_send_report.value[True]:
            tsum_api_hosts = sdk2.parameters.List(
                'API hosts',
                sdk2.parameters.String,
                default=[
                    'https://tsum-agent-api.market.yandex.net:4209',
                    'https://tsum-agent-api-testing.market.yandex.net:4209'
                ],
                required=True
            )

    with sdk2.parameters.Group("Штуки") as misc:
        ubuntu_version = create_ubuntu_selector()
        node_version = create_node_selector()
        prune = sdk2.parameters.Bool("Делать prune проекта", default=True, required=True)
        with prune.value[True]:
            prune_old_style = sdk2.parameters.Bool("Old style npm prune", default=True, required=True)

    environ = sdk2.parameters.Dict('Environment variables')


class MarketFrontNodejsBuildAbstract(nanny.ReleaseToNannyTask2, sdk2.Task):
    """
    Унифицированная сборка маркетфронтовых приложений на nodejs
    """

    PACKAGES_REPO = "git://github.yandex-team.ru/market/packages.git"
    RESOURCES_DIR = os.path.join(os.path.dirname(__file__), "..", "helpers")
    YENV = "production"
    NPM_REGISTRY = "http://npm.yandex-team.ru"
    APP_SRC_DIR = "app_src"
    STATIC_CARCH = "all"
    MAKEDEBPKG_PATH = "/opt/makedebpkg"

    @property
    def static_package_name(self):
        """
        Имя статического пакета
        """
        raise NotImplementedError

    @property
    def static_root(self):
        """
        Путь статики внутри пакета (н.п market-skubi/_)
        """
        raise NotImplementedError

    @property
    def resource_type(self):
        """
        Тип ресурса для архива с приложением
        """
        raise NotImplementedError

    @property
    def app_repo(self):
        """
        Адрес репозитория приложения
        """
        raise NotImplementedError

    @property
    def dist_dir(self):
        """
        Папка дистрибутива приложения, относительно корня репозитория
        """
        return self.Parameters.dist_dir if self.Parameters.dist_dir else None

    @property
    def s3_prefix(self):
        return s3.s3_path_by_repo(self.app_repo)

    def make_dist_path(self, app_src_path):
        return app_src_path if not self.dist_dir else os.path.join(app_src_path, self.dist_dir)

    @property
    def make_env(self):
        make_env = os.environ.copy()
        make_env["YENV"] = self.YENV

        if self.Parameters.common_branch:
            make_env.update({
                "COMMON_SCHEME": self.Parameters.common_scheme,
                "COMMON_REPO": self.Parameters.common_repo,
                "COMMON_REV": self.Parameters.common_branch,
            })

        if self.dist_dir:
            make_env["DIST_DIR"] = self.dist_dir

        if self.Parameters.prune and not self.Parameters.prune_old_style:
            make_env["PRUNE"] = "1"

        if self.Parameters.environ:
            make_env.update(self.Parameters.environ)

        return make_env

    """
    Ветка репозитория https://github.yandex-team.ru/market/packages
    """
    packages_branch = "master"

    class Parameters(GenericParameters):
        pass

    class Requirements(sdk2.Task.Requirements):
        dns = ctm.DnsType.DNS64
        environments = (
            environments.PipEnvironment('boto', use_wheel=True),
        )

    def on_enqueue(self):
        super(MarketFrontNodejsBuildAbstract, self).on_enqueue()
        setup_container(self)

    def on_execute(self):
        super(MarketFrontNodejsBuildAbstract, self).on_execute()

        with MetatronEnv(self, nodejs_version=self.Parameters.node_version):
            root_dir = tempfile.mkdtemp()
            packages_path = os.path.join(root_dir, "packages")
            unified_static_path = os.path.join(packages_path, "yandex-market-unified-static")
            app_src_path = os.path.join(packages_path, self.APP_SRC_DIR)
            logging.info('app_src_path: ${}'.format(app_src_path))
            dist_path = self.make_dist_path(app_src_path)
            logging.info('dist_path: ${}'.format(dist_path))

            self._prepare_repos(app_src_path, packages_path)

            static_version = None
            if self.Parameters.build_static:
                static_version = "{}.t{}".format(self.Parameters.fix_version, self.id)

            self._app_build(app_src_path, dist_path, static_version)

            # todo: refactor this
            #  @see: https://st.yandex-team.ru/MARKETFRONTECH-1045
            #  @see: https://st.yandex-team.ru/MARKETFRONTECH-1042
            if self.Parameters.build_static:
                self._upload_static_to_s3(app_src_path, dist_path)

            if self.Parameters.copy_static:
                self._static_copy(app_src_path, dist_path)

            if self.Parameters.build_static:
                changes_path_o = self._static_build(packages_path, app_src_path, dist_path, unified_static_path,
                                                    static_version)
                self._upload_static(packages_path, changes_path_o)

            pack_app_archive_path = self._create_app_pack(app_src_path, dist_path)

            if self.Parameters.tsum_send_report:
                resource = sdk2.Resource.find(
                    resource_type=self.resource_type.__name__,
                    task_id=self.id
                ).first()
                with sdk2.helpers.ProcessLog(self, "tsum-report"):
                    send_report_to_tsum(
                        archive=pack_app_archive_path,
                        sandbox_resource_type=self.resource_type.__name__,
                        sandbox_ticket_num=self.id,
                        sandbox_resource_id=getattr(resource, "id", None),
                        tsum_api_hosts=self.Parameters.tsum_api_hosts
                    )

    @staticmethod
    def _git_clone(url, ref, target_dir='.'):
        git = Git(url)
        git.clone(str(target_dir), ref)

    def _git_clone_from_revision(self, revision, target_dir='.'):
        git = Git(pipes.quote(self.app_repo))
        git.clone(str(target_dir), pipes.quote(self.Parameters.app_branch), commit=revision)

    @staticmethod
    def _git_clone_merge_commit(url, commit, target_dir='.'):
        git = Git(url)
        git.clone(str(target_dir), "master")
        git.update_cache_repo("refs/pull/*")
        git.execute("checkout", "-f", commit, cwd=target_dir)

    def _upload_static(self, packages_path, changes_path_o):
        with sdk2.helpers.ProcessLog(self, "perl") as process_log:
            perl = sp.Popen(
                ["perl", os.path.join(self.RESOURCES_DIR, "dupload_config_parser.pl"),
                 os.path.join(packages_path, "dupload.conf")],
                stdout=sp.PIPE, stderr=process_log.stderr
            )

            (stdout, stderr) = perl.communicate()
            perl.wait()
            logging.info("perl stdout: %s", stdout)

            dupload_conf = json.loads(stdout)

        with DebRelease(dupload_conf, login="robot-metatron"):
            rich_check_call(
                ["dupload", "--to", "verstka", str(changes_path_o)],
                task=self, alias="dupload"
            )

    def _app_build(self, app_src_path, dist_path, static_version):
        if self.Parameters.app_src_dir:
            working_dir = os.path.join(app_src_path, self.Parameters.app_src_dir)
        else:
            working_dir = app_src_path

        logging.info('working_dir for make: {}'.format(working_dir))
        rich_check_call(
            ["make"],
            task=self, alias="make", cwd=working_dir, env=self.make_env
        )

        self._prune_deps(dist_path)

        if self.Parameters.build_static:
            if self.Parameters.app_src_dir and self.platform:
                version_dir = os.path.join(app_src_path, self.Parameters.app_src_dir, self.platform)
            else:
                version_dir = dist_path
            with open(os.path.join(version_dir, "version.txt"), "a") as f:
                f.write("package_version={}".format(static_version))

    def _create_app_pack(self, app_src_path, dist_path):
        app_repo_match = re.search(".+/([^/.]+?)(:?.git)?$", self.app_repo)
        assert app_repo_match is not None

        # todo: remove after white and blue are unified
        #  @see: https://st.yandex-team.ru/MARKETFRONTECH-506
        #  @see: https://st.yandex-team.ru/MARKETFRONT-15211
        if self.Parameters.app_src_dir and self.platform:
            working_dir = os.path.join(app_src_path, self.Parameters.app_src_dir, self.platform)
        else:
            working_dir = dist_path

        pack_app_path_o = spath.Path(tempfile.gettempdir(), "application")

        self._prepare_pack_app(pack_app_path_o)

        rsync_filter_path = os.path.join(working_dir, ".rsync-filter")

        rich_check_call(
            [
                "rsync", "-az", "--delete", "--force", "--exclude-from=" + rsync_filter_path,
                                                       working_dir + "/", str(pack_app_path_o / "app")
            ],
            task=self, alias="rsync"
        )

        pack_app_archive_path = tempfile.mktemp(suffix=".tar.gz", prefix="app")

        rich_check_call(
            ["tar", "-C", str(pack_app_path_o / "../"), "-czf", pack_app_archive_path, "application"],
            task=self, alias="create_app_pack"
        )

        self._app_create_resource(pack_app_archive_path)

        return pack_app_archive_path

    def _app_create_resource(self, pack_app_archive_path):
        app_res = sdk2.ResourceData(self.resource_type(
            self, "App tarball", "app.tar.gz"
        ))
        if app_res.path.exists():
            app_res.path.unlink()
        app_res.path.write_bytes(spath.Path(pack_app_archive_path).read_bytes())
        app_res.ready()

    @staticmethod
    def _prepare_pack_app(pack_app_path_o):
        if pack_app_path_o.exists():
            shutil.rmtree(str(pack_app_path_o))

        pack_app_path_o.mkdir()

        for appdir in ["app", "run"]:
            (pack_app_path_o / appdir).mkdir()

    def _prune_deps(self, app_src_path):
        if self.Parameters.prune and self.Parameters.prune_old_style:
            rich_check_call(
                ["npm", "prune", "--production"],
                task=self, alias="npm_prune", cwd=app_src_path
            )

    def _prepare_repos(self, app_src_path, packages_path):
        self._git_clone(self.PACKAGES_REPO, pipes.quote(self.packages_branch), packages_path)
        if self.Parameters.app_commit and self.Parameters.need_merge_commit:
            self._git_clone_merge_commit(
                pipes.quote(self.app_repo),
                pipes.quote(self.Parameters.app_commit),
                app_src_path
            )
        elif self.Parameters.app_commit and not self.Parameters.need_merge_commit:
            self._git_clone_from_revision(self.Parameters.app_commit, app_src_path)
        else:
            self._git_clone(pipes.quote(self.app_repo), pipes.quote(self.Parameters.app_branch), app_src_path)

    @staticmethod
    def _package_name_version(version):
        return re.sub(r"[.]", "-", version)

    def _upload_static_to_s3(self, app_src_path, dist_path):
        # todo: remove after white and blue are unified
        #  @see: https://st.yandex-team.ru/MARKETFRONTECH-506
        #  @see: https://st.yandex-team.ru/MARKETFRONT-15211
        if self.Parameters.app_src_dir and self.platform:
            working_dir = os.path.join(app_src_path, self.Parameters.app_src_dir, self.platform)
        else:
            working_dir = dist_path

        with MetatronEnv(self):
            conn = s3.get_connection()

        static_path = os.path.join(working_dir, "s3", "market-static")

        if not os.path.exists(static_path):
            static_path = os.path.join(working_dir, "freeze", "static")

        filename_blacklist = ["manifest.json"]

        processes = []

        for root, dirs, files in os.walk(static_path):
            for filename in files:
                if filename in filename_blacklist:
                    continue
                file_path = os.path.join(root, filename)

                # zopfli/brotli crash on emtpy files
                if os.path.getsize(file_path) == 0:
                    with open(file_path, "w") as f:
                        f.write("\n")

                (zopProc, zopLog) = rich_check_start(
                    ["zopfli", "-i1000", file_path],
                    task=self, alias="compress_gz_{}".format(filename)
                )

                brotli_params = ["brotli", "--force", "--output", file_path + ".br", "--input", file_path] \
                    if self.Parameters.ubuntu_version == UBUNTU_TRUSTY \
                    else ["brotli", "--force", "--output=" + file_path + ".br", file_path]

                (broProc, broLog) = rich_check_start(
                    brotli_params,
                    task=self, alias="compress_br_{}".format(filename)
                )

                processes.append((filename, file_path, zopProc, zopLog, broProc, broLog))

        for (filename, file_path, zopProc, zopLog, broProc, broLog) in processes:
            uploading_path = os.path.join(self.s3_prefix, os.path.relpath(file_path, static_path))
            logging.info('Uploading to: {}'.format(uploading_path))
            s3.upload_file(conn, s3.S3_BUCKET_NAME, file_path, uploading_path)

        for (filename, file_path, zopProc, zopLog, broProc, broLog) in processes:
            rich_check_report(zopProc, zopLog, task=self, alias="compress_gz_{}".format(filename))
            rich_check_report(broProc, broLog, task=self, alias="compress_br_{}".format(filename))

            uploading_path = os.path.join(self.s3_prefix, os.path.relpath(file_path, static_path))
            logging.info('Uploading gz/br to: {}'.format(uploading_path))

            s3.upload_file(conn, s3.S3_BUCKET_NAME, file_path + ".gz", "{}.gz".format(uploading_path))
            s3.upload_file(conn, s3.S3_BUCKET_NAME, file_path + ".br", "{}.br".format(uploading_path))

    def _static_copy(self, app_src_path, dist_path):
        # todo: remove after white and blue are unified
        #  @see: https://st.yandex-team.ru/MARKETFRONTECH-506
        #  @see: https://st.yandex-team.ru/MARKETFRONT-15211
        if self.Parameters.app_src_dir and self.platform:
            working_dir = os.path.join(app_src_path, self.Parameters.app_src_dir, self.platform)
        else:
            working_dir = dist_path

        logging.info('working dir path for static copy: {}'.format(working_dir))

        freeze_static_path = os.path.join(working_dir, "freeze/static")
        s3_static_path = os.path.join(working_dir, "s3", pipes.quote(s3.S3_BUCKET_NAME), pipes.quote(self.s3_prefix))
        conductor_static_path = os.path.join(working_dir, pipes.quote(self.static_root))

        rich_check_call(["mkdir", "-vp", s3_static_path], self, 'static-s3-mkdir')
        rich_check_call(["rm", "-vrf", s3_static_path], self, 'static-s3-remove')
        rich_check_call(["mv", "-vfT", freeze_static_path, s3_static_path], self, 'static-s3-move')

        # TODO: убрать после окончательного перезда на s3
        rich_check_call(["mkdir", "-vp", conductor_static_path], self, 'static-conductor-mkdir')
        rich_check_call(["rm", "-vrf", conductor_static_path], self, 'static-conductor-remove')
        rich_check_call(["cp", "-vrfT", s3_static_path, conductor_static_path], self, 'static-conductor-copy')

    def _static_build(self, packages_path, app_src_path, dist_path, unified_static_path, static_version):
        # todo: remove after white and blue are unified
        #  @see: https://st.yandex-team.ru/MARKETFRONTECH-506
        #  @see: https://st.yandex-team.ru/MARKETFRONT-15211
        if self.Parameters.app_src_dir and self.platform:
            working_dir = os.path.join(app_src_path, self.Parameters.app_src_dir, self.platform)
        else:
            working_dir = dist_path
        makedebpkg_config_path = os.path.join(packages_path, "makedebpkg.conf")

        static_path = os.path.join(working_dir, "s3", pipes.quote(s3.S3_BUCKET_NAME), pipes.quote(self.s3_prefix))

        if not os.path.exists(static_path):
            static_path = os.path.join(working_dir, pipes.quote(self.static_root))

        os.symlink(self.MAKEDEBPKG_PATH, os.path.join(packages_path, ".makedebpkg"))

        static_fullname = "{}-{}".format(
            self.static_package_name,
            self._package_name_version(static_version)
        )

        os.environ.update({
            "PATH": self.MAKEDEBPKG_PATH + ":" + os.environ["PATH"],
            "PACKAGE": self.static_package_name,
            "VERSION": static_version,
            "PACKAGE_FULLNAME": static_fullname,
            "SOURCES": static_path,
            "CARCH": self.STATIC_CARCH
        })

        rich_check_call(
            [
                self.MAKEDEBPKG_PATH + "/makedebpkg", "--config", makedebpkg_config_path, "--log", "--sign",
                "--nocheck", "--key", pipes.quote(os.environ["GPGKEY"])
            ],
            task=self, alias="build_static", cwd=unified_static_path
        )

        return self._static_create_resource(static_fullname, static_version, unified_static_path)

    def _static_create_resource(self, static_fullname, static_version, unified_static_path):
        with open(os.path.join(unified_static_path, "PKGBUILD")) as f:
            match = re.search("pkgrel=(\d+)", f.read())
            assert match is not None
            pkgrel = match.group(1)

        static_version_with_pkgrel = static_version + "-" + pkgrel

        static_res = sdk2.ResourceData(MarketFrontBuildUnifiedStaticResource(
            self, "Static package", "static",
            fix_version=self.Parameters.fix_version,
            version=static_version_with_pkgrel
        ))

        if not static_res.path.exists():
            static_res.path.mkdir()

        def make_pkg_path(pkg_path_suffix):
            path_o = spath.Path(
                unified_static_path,
                static_fullname + "_" + static_version_with_pkgrel + pkg_path_suffix
            )
            return path_o

        changes_suffix = "_" + self.STATIC_CARCH + ".changes"
        for suffix in ["_" + self.STATIC_CARCH + ".deb", changes_suffix, ".dsc"]:
            file_path_o = make_pkg_path(suffix)

            logging.info("static file path %s", str(file_path_o))
            assert file_path_o.exists()
            dest = (static_res.path / file_path_o.name)
            if dest.exists():
                dest.unlink()

            dest.write_bytes(file_path_o.read_bytes())

        static_res.ready()

        return make_pkg_path(changes_suffix)
