# -*- coding: utf-8 -*-

import logging
import os
import shutil

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.types import task as ctt
from sandbox.common.types import misc as ctm
from sandbox.projects.market.front.helpers.sandbox_helpers \
    import rich_check_call, report_data, format_header
from sandbox.projects.market.front.helpers.node import create_node_selector
from sandbox.projects.market.front.helpers.ubuntu import create_ubuntu_selector, setup_container
from sandbox.projects.market.front.helpers.MetatronEnv import MetatronEnv
from sandbox.projects.market.resources import MARKET_PORCHMARK_REPORT, MARKET_PORCHMARK_WORKDIR
from sandbox.projects.sandbox_ci.utils import env
from sandbox.projects.market.front.helpers.github import GitHubHelper
from sandbox.projects.market.front.constants.porchmark import \
    HUMAN_REPORT_FILENAME, HTML_REPORT_FILENAME, JSON_REPORT_FILENAME, PORCHMARK_CONFIG_FILENAME, WORKDIR_NAME

DISK_SPACE = 3 * 1024  # 3 Gb
TEST_MODES = ("test", "light", "normal", "hard")
DEFAULT_TEST_MODE = "normal"
SUBTASK_TIMEOUT = 30 * 60  # 30 min


class MarketFrontPorchmarkCompare(sdk2.Task):
    """
    Маркетные perf-тесты с помощью porchmark
    """

    root_dir = None
    app_repo_path = None
    app_src_path = None
    config_path = None
    porchmark_workdir = None
    github = None

    class Context(sdk2.Context):
        report_resource_id = None
        save_result_task_id = None
        git_commit = None

    class Parameters(sdk2.Task.Parameters):
        ubuntu_version = create_ubuntu_selector()
        node_version = create_node_selector()

        with sdk2.parameters.Group('GitHub репозиторий проекта') as github_repo_block:
            app_owner = sdk2.parameters.String(
                "Github owner",
                default_value="market",
            )

            app_repo = sdk2.parameters.String(
                "Github repo",
                required=True
            )

            app_branch = sdk2.parameters.String(
                "Тестируемая ветка",
                required=True
            )

            app_src_dir = sdk2.parameters.String(
                "Кастомный путь корня приложения внутри репозитория"
            )

        with sdk2.parameters.Group('Porchmark') as porchmark_block:
            porchmark_bin_path = sdk2.parameters.String(
                "Porchmark bin path",
                default_value="node_modules/.bin/porchmark",
                required=True,
            )

            porchmark_config_path = sdk2.parameters.String(
                "Porchmark config path",
                default_value="platform.touch/configs/testing/porchmark.conf.js",
                required=True,
            )

            porchmark_raw_config = sdk2.parameters.String(
                "Porchmark raw config",
                required=False,
                multiline=True,
            )

        with sdk2.parameters.Group('Compare') as compare_block:
            test_host = sdk2.parameters.String(
                "Test host (env.TEST_HOST)",
                required=True,
            )

            with sdk2.parameters.String(
                "Test mode (env.TEST_MODE)",
                required=True,
                multiline=True,
            ) as test_mode:
                for mode in TEST_MODES:
                    is_default = mode == DEFAULT_TEST_MODE
                    test_mode.values[mode] = test_mode.Value(mode, default=is_default)

            test_page = sdk2.parameters.String(
                "Test page from porchmark.conf.js (env.TEST_PAGE)",
                required=True,
            )

            sample_host = sdk2.parameters.String(
                "Sample host (env.SAMPLE_HOST)",
                required=True,
            )

        with sdk2.parameters.Group('Report') as report_block:
            create_porchmark_workdir_resource = sdk2.parameters.Bool(
                "Create MARKET_PORCHMARK_WORKDIR resource",
                default_value=True,
                required=False,
                description="собирать ресурс MARKET_PORCHMARK_WORKDIR\n"
                            "в ресурс собирается workdir porchmark'a\n"
                            "там находятся WPR-архивы, логи и скриншоты\n"
                            "ресурс полезен для отладки\n"
                            "можно воспроизвести сравнение с теми же WPR что и в таске"
            )

        with sdk2.parameters.Group("Environment") as environ_block:
            environ = sdk2.parameters.Dict("Environment variables")

    class Requirements(sdk2.Task.Requirements):
        dns = ctm.DnsType.DNS64
        disk_space = DISK_SPACE

    def on_enqueue(self):
        super(MarketFrontPorchmarkCompare, self).on_enqueue()
        setup_container(self)

    @sdk2.header()
    def header(self):
        resource_id = self.Context.report_resource_id

        if resource_id:
            resource = self.server.resource[resource_id].read()
            data = report_data(resource)
            report = {'<h3 id="checks-reports">Report</h3>': [format_header(**data)]}
            return report

    def on_prepare(self):
        pass

    def on_execute(self):
        self._porchmark_compare()
        self._verify_save_report_task()

    def _porchmark_compare(self):
        with MetatronEnv(self, nodejs_version=self.Parameters.node_version), \
                self.memoize_stage.porchmark_compare(max_runs=1):

            self.root_dir = str(self.path())
            self.app_repo_path = os.path.join(self.root_dir, self.Parameters.app_repo)

            if self.Parameters.app_src_dir:
                self.app_src_path = os.path.join(self.app_repo_path, self.Parameters.app_src_dir)
            else:
                self.app_src_path = self.app_repo_path

            self._clone_repo()
            self._install_deps()

            self.config_path = self._prepare_config()
            self.porchmark_workdir = self._prepare_workdir()

            self._prepare_environ()

            try:
                self._run_porchmark_compare()
                self._create_report_resource()
                self._save_report()
            finally:
                self._create_workdir_resource()

    def _verify_save_report_task(self):
        with self.memoize_stage.verify_save_report_task(max_runs=1):
            subtask = sdk2.Task[self.Context.save_result_task_id]

            logging.info("verify save report task status={}".format(subtask.status))

            if subtask.status not in ctt.Status.Group.SUCCEED:
                raise TaskFailure('Subtask id: {} failed with status: {}'.format(subtask.id, subtask.status))

    def _clone_repo(self):
        self.github = GitHubHelper(
            owner=self.Parameters.app_owner,
            repo=self.Parameters.app_repo,
            branch=self.Parameters.app_branch,
            path=self.app_repo_path,
        )
        self.github.clone()
        self.Context.git_commit = self.github.commit

    def _install_deps(self):
        veendor_config_path = os.path.join(self.app_src_path, ".veendor.js")
        package_json_path = os.path.join(self.app_src_path, "package.json")

        if os.path.isfile(veendor_config_path):
            rich_check_call(
                ["veendor", "install"],
                task=self, alias="veendor_install", cwd=self.app_src_path
            )
        elif os.path.isfile(package_json_path):
            rich_check_call(
                ["npm", "install"],
                task=self, alias="npm_install", cwd=self.app_src_path
            )
        else:
            raise Exception(
                "no veendor config or package.json: {}, {}".format(
                    veendor_config_path,
                    package_json_path,
                )
            )

    def _prepare_config(self):
        config_path = os.path.join(self.app_src_path, str(self.Parameters.porchmark_config_path))

        if self.Parameters.porchmark_raw_config:
            # save raw config
            config_path = os.path.join(self.app_src_path, "porchmark.raw-config.js")
            with open(config_path, "w") as f:
                f.write(str(self.Parameters.porchmark_raw_config))

        logging.debug("porchmark config path: {}".format(config_path))

        if not os.path.isfile(config_path):
            raise Exception("porchmark config not exists: {}".format(config_path))

        return config_path

    def _prepare_workdir(self):
        porchmark_workdir = os.path.join(self.root_dir, WORKDIR_NAME)
        os.mkdir(porchmark_workdir)
        return porchmark_workdir

    def _prepare_environ(self):
        env.export({
            "TEST_HOST": self.Parameters.test_host,
            "TEST_MODE": self.Parameters.test_mode,
            "TEST_PAGE": self.Parameters.test_page,
            "SAMPLE_HOST": self.Parameters.sample_host,
        })

        env.export(self.Parameters.environ)

        logging.info('environ = {}'.format(os.environ))

    def _run_porchmark_compare(self):
        # check porchmark bin
        bin_path = os.path.join(self.app_src_path, str(self.Parameters.porchmark_bin_path))

        if not os.path.isfile(bin_path):
            raise Exception("porchmark bin not exists: {}".format(bin_path))

        # execute porchmark compare
        rich_check_call(
            ["node", bin_path, "compare", "--config", self.config_path],
            task=self, alias="porchmark_compare", cwd=self.porchmark_workdir
        )

    def _create_workdir_resource(self):
        if not self.Parameters.create_porchmark_workdir_resource:
            return

        if not os.path.exists(self.porchmark_workdir):
            logging.error(
                "can't create MARKET_PORCHMARK_WORKDIR resource, {} not exists".format(
                    self.porchmark_workdir
                )
            )
            return

        logging.info('creating MARKET_PORCHMARK_WORKDIR resource')

        workdir_resource = MARKET_PORCHMARK_WORKDIR(
            self,
            "porchmark compare workdir",
            self.porchmark_workdir,
            type='porchmark-compare-workdir',
            project=self.Parameters.app_repo,
            report_description=self.Parameters.app_repo,
            status=ctt.Status.SUCCESS,
            root_path="{}/{}".format(self.Parameters.test_page, HUMAN_REPORT_FILENAME)
        )

        workdir_resource_data = sdk2.ResourceData(workdir_resource)
        logging.info("workdir_resource path: {}".format(workdir_resource_data.path))
        workdir_resource_data.ready()

    def _copy_json_report(self, porchmark_report_dir):
        json_report_filepath = "{}/{}/{}".format(
            self.porchmark_workdir,
            self.Parameters.test_page,
            JSON_REPORT_FILENAME,
        )

        shutil.copy(json_report_filepath, porchmark_report_dir)

    def _copy_human_report(self, porchmark_report_dir):
        human_report_filepath = "{}/{}/{}".format(
            self.porchmark_workdir,
            self.Parameters.test_page,
            HUMAN_REPORT_FILENAME,
        )

        shutil.copy(human_report_filepath, porchmark_report_dir)

    def _copy_html_report(self, porchmark_report_dir):
        html_report_filepath = "{}/{}/{}".format(
            self.porchmark_workdir,
            self.Parameters.test_page,
            HTML_REPORT_FILENAME,
        )

        if os.path.exists(html_report_filepath):
            shutil.copy(html_report_filepath, porchmark_report_dir)
        else:
            logging.error("can't copy html report file, file not exists: {}".format(html_report_filepath))

    def _copy_porchmark_config(self, porchmark_report_dir):
        filepath = "{}/{}".format(
            self.porchmark_workdir,
            PORCHMARK_CONFIG_FILENAME,
        )

        if os.path.exists(filepath):
            shutil.copy(filepath, porchmark_report_dir)
        else:
            logging.error("can't copy porchmark config, file not exists: {}".format(filepath))

    def _create_report_resource(self):
        logging.info('creating MARKET_PORCHMARK_REPORT resource')

        porchmark_report_dir = os.path.join(self.root_dir, "porchmark_report")
        os.mkdir(porchmark_report_dir)

        self._copy_json_report(porchmark_report_dir)
        self._copy_human_report(porchmark_report_dir)
        self._copy_html_report(porchmark_report_dir)
        self._copy_porchmark_config(porchmark_report_dir)

        report_resource = MARKET_PORCHMARK_REPORT(
            self,
            "porchmark report",
            porchmark_report_dir,
            type='porchmark-compare-report',
            project=self.Parameters.app_repo,
            report_description=self.Parameters.app_repo,
            status=ctt.Status.SUCCESS,
            root_path=HUMAN_REPORT_FILENAME
        )

        report_resource_data = sdk2.ResourceData(report_resource)
        report_resource_data.ready()

        self.Context.report_resource_id = report_resource.id

    def _save_report(self):
        from sandbox.projects.market.front.MarketFrontPorchmarkSaveReport import MarketFrontPorchmarkSaveResult

        task = MarketFrontPorchmarkSaveResult(
            self,
            report_resource_id=self.Context.report_resource_id,
            app_owner=self.Parameters.app_owner,
            app_repo=self.Parameters.app_repo,
            app_branch=self.Parameters.app_branch,
            git_commit=self.Context.git_commit,
            porchmark_config_path=self.Parameters.porchmark_config_path,
            test_mode=self.Parameters.test_mode,
            test_page=self.Parameters.test_page,

        )
        task.enqueue()
        self.Context.save_result_task_id = task.id
        raise sdk2.WaitTask(
            self.Context.save_result_task_id,
            ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
            wait_all=True,
            timeout=SUBTASK_TIMEOUT
        )
