# coding=utf-8

import json
import urllib
import logging
import requests

from sandbox.projects.AddrsDeploy.tools.misc import retry


class CalculateMessage(object):

    TANK_STATUSES = {
        "0": "completed",
        "1": "interrupted_generic_interrupt",
        "2": "interrupted",
        "3": "interrupted_active_task_not_found ",
        "4": "interrupted_no_ammo_file",
        "5": "interrupted_address_not_specified",
        "6": "interrupted_cpu_or_disk_overload",
        "7": "interrupted_unknown_config_parameter",
        "8": "interrupted_stop_via_web",
        "9": "interrupted",
        "11": "interrupted_job_number_error",
        "12": "interrupted_phantom_error",
        "13": "interrupted_job_metainfo_error",
        "14": "interrupted_target_monitoring_error",
        "15": "interrupted_target_info_error",
        "21": "autostop_time",
        "22": "autostop_http",
        "23": "autostop_net",
        "24": "autostop_instances",
        "25": "autostop_total_time",
        "26": "autostop_total_http",
        "27": "autostop_total_net",
        "28": "autostop_negative_http",
        "29": "autostop_negative_net",
        "30": "autostop_http_trend",
        "31": "autostop_metric_higher",
        "32": "autostop_metric_lower"
    }

    def __init__(self, job_id, sla):
        self.job_id = job_id
        self.sla = sla
        self.message = ''

    @retry(tries=3, delay=5)
    def query_lpdb(self, method, params=None):
        url = "https://lunapark.yandex-team.ru/api/job/{}/{}.json?{}".format(
            self.job_id, method, urllib.urlencode(params) if params else '')
        logging.info('Request to lunapark: %s', url)
        response = requests.get(url)
        response.raise_for_status()
        logging.debug('Response from lunapark: %s', response.content)
        return response.content

    def get_percentile(self, case, percents=None):
        mapping = {}
        if percents is None:
            percents = [95]

        for percent in percents:
            if not 0 <= percent <= 100:
                raise RuntimeError("percent is out of bounds")

        query = {"case": case}
        if percents is not None:
            query["percents"] = ",".join(map(lambda e: str(e), percents))

        try:
            json_content = json.loads(self.query_lpdb('dist/percentiles', query))
        except (TypeError, ValueError):
            return {'95': 'n/a'}

        if len(json_content) != 1 or "ms" not in json_content[0]:
            raise RuntimeError('Malformed object')

        for data in json_content:
            mapping[data["percentile"]] = data["ms"]

        return mapping

    def get_status(self):
        try:
            json_content = json.loads(self.query_lpdb(method='summary'))
        except (ValueError, TypeError):
            raise RuntimeError('Malformed object')

        if len(json_content) != 1 or 'quit_status' not in json_content[0]:
            raise RuntimeError('Malformed object')

        return json_content[0]['quit_status']

    @staticmethod
    def check_status(status):
        return status == 0

    def create_message(self):
        quit_status = self.get_status()

        if not self.check_status(quit_status):
            status_str = str(quit_status)
            text_status = self.TANK_STATUSES[status_str] if status_str in self.TANK_STATUSES else status_str
            self.message = u'Quit status: {}\n'.format(text_status)
        else:
            self.message = u'#|\n||страница|персентиль (текущий / sla)||\n'
            for page, percents in self.sla.items():
                percentiles_ms = self.get_percentile(page, map(lambda i: int(i), percents.keys()))
                self.message += u'||{}|'.format(page)
                for percentile, ms in percentiles_ms.items():
                    value_cell = u'{} / {}'.format(ms, percents[percentile])
                    if 'n/a' in str(ms):
                        value_cell = u'!!{}!!'.format(value_cell)
                    elif ms > percents[percentile]:
                        value_cell = u'!!{}!!'.format(value_cell)
                    else:
                        value_cell = u'!!(зел){}!!'.format(value_cell)
                    self.message += u'{} ({}); '.format(percentile, value_cell)
                self.message += u'||\n'

            self.message += u'|#\n'
            logging.info('Comparison message is %s', self.message)
