# coding=utf-8

import os.path
import os

from sandbox import sdk2
import sandbox.common.types.misc as ctm

from sandbox.projects.market.front.helpers.MetatronEnv import MetatronEnv
from sandbox.projects.market.front.helpers.sandbox_helpers import rich_check_call
from sandbox.projects.market.front.helpers.ubuntu import create_ubuntu_selector, setup_container
from sandbox.projects.market.front.helpers.node import create_node_selector
from sandbox.sdk2.vcs.git import Git
from sandbox.projects.market.front.helpers.sandbox_helpers import rich_check_call, get_resource_http_proxy_link
from sandbox.projects.market.front.resources import MarketCIReport
from sandbox.projects.market.resources import MARKET_AUTOTEST_REPORT

class MarketFrontSimpleBashSaveHtmlReport(sdk2.Task):
    """
    Таска для выполнения проектных скриптов в СБ c генерацией Html  отчёта
    """

    APP_SRC_DIR = "app_src"

    class Parameters(sdk2.Task.Parameters):
        github_owner = sdk2.parameters.String(
            "Github owner",
            default_value="market"
        )

        github_repo = sdk2.parameters.String(
            "Github repo"
        )

        app_branch = sdk2.parameters.String(
            "App branch",
            default_value="master"
        )

        commit_hash = sdk2.parameters.String(
            "Хэш коммита, на который нужно ресетнуться перед выполнением скрипта"
        )

        execution_command = sdk2.parameters.String(
            "Команда для выполнения"
        )

        report_dir = sdk2.parameters.String(
            "Report dir",
            default_value="html_reports"
        )

        report_file_name = sdk2.parameters.String(
            "Report file name",
            default_value="index.html"
        )

        check_env = sdk2.parameters.Dict("Переменные окружения для запуска проверки")

        vault_env = sdk2.parameters.Dict("Добавить значения из Vault в переменные окружения, ключ - переменная окружения, значение - owner:key")

        ubuntu_version = create_ubuntu_selector()
        node_version = create_node_selector()

    class Requirements(sdk2.Task.Requirements):
        dns = ctm.DnsType.DNS64

    @property
    def check_env(self):
        env = os.environ.copy()
        if self.Parameters.check_env:
            env.update(self.Parameters.check_env)
        if self.Parameters.vault_env:
            env.update(dict(map(lambda kv: (kv[0], sdk2.Vault.data(*kv[1].split(':'))), self.Parameters.vault_env.items())))
        return env

    def on_enqueue(self):
        super(MarketFrontSimpleBashSaveHtmlReport, self).on_enqueue()
        setup_container(self)

    def on_execute(self):
        super(MarketFrontSimpleBashSaveHtmlReport, self).on_execute()

        with MetatronEnv(self, nodejs_version=self.Parameters.node_version):
            self._run_build()

    @property
    def _git_url(self):
        return "https://github.yandex-team.ru/{}/{}.git".format(
            self.Parameters.github_owner,
            self.Parameters.github_repo
        )

    @property
    def html_report_path(self):
        return os.path.join(self.APP_SRC_DIR, self.Parameters.report_dir, self.Parameters.report_file_name)

    def _run_build(self):
        self._clone_repo()

        try:
            rich_check_call(
                ["bash", "-c", self.Parameters.execution_command],
                self, "execution_command", cwd=self.APP_SRC_DIR, env=self.check_env
            )

        finally:
            self._get_report()
    
    def _create_html_report_resource(self, report_path, **params):
        if params:
            report_res = MARKET_AUTOTEST_REPORT(self, "Screenshot report", report_path, **params)
        else:
            report_res = MarketCIReport(self, "Report", report_path)

        report_res_data = sdk2.ResourceData(report_res)
        report_res_data.ready()

        return report_res
    
    def _clone_repo(self):
        git = Git(self._git_url)
        if (self.Parameters.commit_hash):
            git.clone(self.APP_SRC_DIR, commit=self.Parameters.commit_hash)
        else:
            git.clone(self.APP_SRC_DIR, self.Parameters.app_branch)
    
    def _get_report(self, **params):
        if self.html_report_path is not None:
            res = self._create_html_report_resource(os.path.dirname(self.html_report_path), **params)
            http_report_url = '{}/{}'.format(get_resource_http_proxy_link(res),
                                                os.path.basename(self.html_report_path))
            self.set_info(
                "Отчёт: <a href=\"{url}\">{url}</a>".format(url=http_report_url),
                do_escape=False
            )

            self.report_url = http_report_url
        else:
            raise ValueError("Report not found by providing path. Set correct report dir and filename")
