# coding=utf-8

import json
import os
import logging
from urlparse import urljoin

from sandbox import sdk2
import sandbox.common.types.misc as ctm
from sandbox.common.errors import TaskError

from sandbox.projects.market.front.helpers.MetatronEnv import MetatronEnv
from sandbox.projects.market.front.helpers.sandbox_helpers import rich_check_call
from sandbox.projects.market.front.helpers.ubuntu import create_ubuntu_selector, setup_container
from sandbox.projects.market.front.helpers.node import create_node_selector
from sandbox.projects.market.front.helpers.github import clone_repo
from sandbox.projects.market.resources import MARKET_FRONT_TICOUNT_REPORT

class MarketFrontTicount(sdk2.Task):
    """
    Таска для запуска ticount
    """

    APP_SRC_DIR = "app_src"
    REPORTS_DIR = "report"
    ACTUAL_TRACES_DIR = "report/traces/actual"
    BASE_TRACES_DIR = "report/traces/base"
    AVG_TRACES_DIR = "report/traces/avg"

    class Parameters(sdk2.Task.Parameters):
        ubuntu_version = create_ubuntu_selector()
        node_version = create_node_selector()

        app_owner = sdk2.parameters.String(
            "Github owner",
            default="market",
            required=True
        )

        app_repo = sdk2.parameters.String(
            "Github repo",
            default="marketfront",
            required=True
        )

        app_branch = sdk2.parameters.String(
            "Github branch",
            default="master",
            required=True
        )

        shooting_config_path = sdk2.parameters.String(
            "Путь до конфига стрельб ticount, от корня репозитория",
            default="configs/ticount/shooting.js",
            required=True
        )

        compare_config_path = sdk2.parameters.String(
            "Путь до конфига сравнения ticount, от корня репозитория",
            default="configs/ticount/compare.js",
            required=True
        )

        report_config_path = sdk2.parameters.String(
            "Путь до конфига генерации отчета ticount, от корня репозитория",
            default="configs/ticount/report.json",
            required=True
        )

        pages_config_path = sdk2.parameters.String(
            "Путь до конфига страниц, от корня репозитория",
            default="configs/ticount/pages.json"
        )

        mobile = sdk2.parameters.Bool(
            "Использовать эмулятор мобильного девайса",
            default=False,
            required=True
        )

        sample_host = sdk2.parameters.String(
            "Эталонный балансер",
            required=True
        )

        test_host = sdk2.parameters.String(
            "Актуальный балансер",
            required=True
        )

        test_page = sdk2.parameters.String(
            "Тестируемая страница, если есть конфиг страниц",
            description="Например, index"
        )

        test_url = sdk2.parameters.String(
            "Тестируемый url, если нет конфига страниц",
            description="Например, /search?text=красный%20куб"
        )

    class Requirements(sdk2.Task.Requirements):
        dns = ctm.DnsType.DNS64

    @property
    def _root_path(self):
        return str(self.path())

    @property
    def _app_src_path(self):
        return os.path.join(self._root_path, self.APP_SRC_DIR)

    @property
    def _shooting_config_path(self):
        return os.path.join(self._app_src_path, self.Parameters.shooting_config_path)

    @property
    def _compare_config_path(self):
        return os.path.join(self._app_src_path, self.Parameters.compare_config_path)

    @property
    def _report_config_path(self):
        return os.path.join(self._app_src_path, self.Parameters.report_config_path)

    @property
    def _reports_path(self):
        return os.path.join(self._root_path, self.REPORTS_DIR)

    @property
    def _base_metrics_log_path(self):
        return os.path.join(self._reports_path, "base_metrics.log")

    @property
    def _actual_metrics_log_path(self):
        return os.path.join(self._reports_path, "actual_metrics.log")

    @property
    def _base_data_path(self):
        return os.path.join(self._reports_path, "base_data.json")

    @property
    def _actual_data_path(self):
        return os.path.join(self._reports_path, "actual_data.json")

    @property
    def _excesses_log_path(self):
        return os.path.join(self._reports_path, "excesses.log")

    @property
    def _txt_report_path(self):
        return os.path.join(self._reports_path, "report.txt")

    @property
    def _html_report_path(self):
        return os.path.join(self._reports_path, "report.html")

    @property
    def _avg_traces_path(self):
        return os.path.join(self._root_path, self.AVG_TRACES_DIR)

    @property
    def _base_traces_path(self):
        return os.path.join(self._root_path, self.BASE_TRACES_DIR)

    @property
    def _actual_traces_path(self):
        return os.path.join(self._root_path, self.ACTUAL_TRACES_DIR)

    def _prepare_dirs(self):
        logging.info("Preparing dirs")

        os.makedirs(self.APP_SRC_DIR)
        os.makedirs(self.REPORTS_DIR)
        os.makedirs(self.BASE_TRACES_DIR)
        os.makedirs(self.ACTUAL_TRACES_DIR)
        os.makedirs(self.AVG_TRACES_DIR)

    def _prepare_repo(self):
        logging.info("Preparing repo")

        clone_repo(
            self.Parameters.app_owner,
            self.Parameters.app_repo,
            self.Parameters.app_branch,
            self._app_src_path
        )

        rich_check_call(
            ["bash", "-c", "npm run bootstrap"],
            self, "bootstrap", cwd=self._app_src_path
        )

    def _prepare_configs(self):
        logging.info("Preparing configs")

        self._pages = None

        if self.Parameters.pages_config_path:
            with open(os.path.join(self._app_src_path, self.Parameters.pages_config_path), "r") as file:
                self._pages = json.load(file)

    def _get_test_url(self):
        if self._pages and self.Parameters.test_page:
            return self._pages[self.Parameters.test_page]
        elif self.Parameters.test_url:
            return self.Parameters.test_url
        else:
            raise TaskError("Test url not set")

    def _shooting_page(self, host, metrics_log_path, data_path, traces_dir):
        logging.info("Shooting {}".format(host))

        rich_check_call(
            [
                "bash", "-c",
                "TICOUNT_DATA_FILE={} "
                "node_modules/.bin/ticount write "
                "--config {} --url {} --metrics-log {} --traces-dir {} --mobile {}".format(
                    data_path,
                    self._shooting_config_path,
                    urljoin(host, self._get_test_url()),
                    metrics_log_path,
                    traces_dir,
                    "true" if self.Parameters.mobile else "false"
                )
            ],
            self, "shooting_page", cwd=self._app_src_path
        )

    def _compare_pages(self):
        logging.info("Comparing pages")

        rich_check_call(
            [
                "bash", "-c",
                "TICOUNT_ACTUAL_DATA_FILE={} TICOUNT_BASE_DATA_FILE={} TICOUNT_REPORT_DIR={} "
                "node_modules/.bin/ticount compare --skip-shooting --config {} "
                "--metrics-log-base {} --metrics-log-actual {} "
                "--excesses-log {} 1> {}".format(
                    self._actual_data_path,
                    self._base_data_path,
                    self._avg_traces_path,
                    self._compare_config_path,
                    self._base_metrics_log_path,
                    self._actual_metrics_log_path,
                    self._excesses_log_path,
                    self._txt_report_path
                )
            ],
            self, "compare_pages", cwd=self._app_src_path
        )

    def _generate_html_report(self):
        logging.info("Generating html report")

        rich_check_call(
            [
                "node_modules/.bin/ticount-report",
                "--url", self._get_test_url(),
                "--output", self._html_report_path,
                "--data", self._base_metrics_log_path, self._actual_metrics_log_path,
                "--config", self._report_config_path
            ],
            self, "generate_html_report", cwd=self._app_src_path
        )

    def _create_reports_resource(self):
        logging.info("Create reports resource")

        resource = MARKET_FRONT_TICOUNT_REPORT(self, "Ticount report", self._reports_path)
        sdk2.ResourceData(resource).ready()
        return resource

    def on_enqueue(self):
        super(MarketFrontTicount, self).on_enqueue()
        setup_container(self)

    def on_execute(self):
        super(MarketFrontTicount, self).on_execute()

        with MetatronEnv(self, nodejs_version=self.Parameters.node_version):
            self._prepare_dirs()
            self._prepare_repo()
            self._prepare_configs()
            self._shooting_page(
                self.Parameters.sample_host,
                self._base_metrics_log_path,
                self._base_data_path,
                self._base_traces_path
            )
            self._shooting_page(
                self.Parameters.test_host,
                self._actual_metrics_log_path,
                self._actual_data_path,
                self._actual_traces_path
            )
            self._compare_pages()
            self._generate_html_report()
            self._create_reports_resource()
