# coding=utf-8

import re
import yaml
import json
import logging
import os
from subprocess32 import CalledProcessError

from sandbox import sdk2
from sandbox.common.utils import get_task_link
from sandbox.projects.sandbox.sandbox_lxc_image import RichTextTaskFailure
import sandbox.common.types.resource as ctr

from sandbox.projects.market.front.helpers.yammy.errors import MergeConflictException, TestFailure
from sandbox.projects.market.front.helpers.yammy.git_helpers import YammyGitHelpers, YammyGitParameters
from sandbox.projects.market.front.helpers.yammy.task_helpers import YammyTaskHelpers, YammyTaskParameters
from sandbox.projects.market.front.helpers.yammy.time_helpers import YammyTimeHelpers
from sandbox.projects.market.front.resources.yammy_resources import MarketFrontYammyBuildArtefact, \
    MarketFrontYammyMetaArtefact
from sandbox.projects.market.front.helpers.ubuntu import setup_container
from sandbox.projects.market.front.helpers.sandbox_helpers import rich_check_call, rich_get_output, rich_check_start, \
    get_resource_http_proxy_link
from sandbox.projects.market.front.helpers.github import GitHubStatus, GitHubYammyStatusDescription
from sandbox.projects.market.front.helpers.yammy.requirements import YammySmallRequirements
from sandbox.projects.market.front.helpers.ci import propagate_github_event_data


class MarketFrontYammyBase(sdk2.Task, YammyGitHelpers, YammyTaskHelpers, YammyTimeHelpers):
    """
    Унифицированная таска для запуска yammy-based сценариев в монорепе
    """

    _build_meta = None
    _build_meta_resource = None
    _yammy_hash = None
    watch_proc = None

    RESOURCES_DIR = os.path.join(os.path.dirname(__file__), "resources")
    WATCH_SCRIPT_PATH = os.path.join(RESOURCES_DIR, "watch.bash")
    USAGE_REPORT_NAME = 'usage-report.html'

    class Requirements(YammySmallRequirements):
        pass

    class Parameters(sdk2.Task.Parameters):
        task_never_fail = sdk2.parameters.Bool("Не фейлить джобу")

        git_params = YammyGitParameters()

        with sdk2.parameters.Group('Yammy') as yammy:
            yammy_base_ref = sdk2.parameters.String("Базовая ревизия", required=False)

            yammy_build_meta = sdk2.parameters.Resource(
                "Мета-информация для сборки",
                resource_type=MarketFrontYammyMetaArtefact
            )

            yammy_prebuilt = sdk2.parameters.Resource(
                "Артефакт сборки",
                resource_type=MarketFrontYammyBuildArtefact
            )

            yammy_build_digest = sdk2.parameters.String("Идентификатор билда", required=False)

        task_params = YammyTaskParameters()

    @property
    def ticket(self):
        return None if self.Parameters.head_branch == "master" \
            else re.search('[A-Z]+-[0-9]+', self.Parameters.head_branch).group(0)

    @property
    def _build_meta_res(self):
        if not self.Parameters.yammy_build_meta:
            return None

        if not self._build_meta_resource:
            with self.timer("meta:download"):
                self._build_meta_resource = sdk2.ResourceData(self.Parameters.yammy_build_meta)

        return self._build_meta_resource

    @property
    def build_meta(self):
        if self._build_meta:
            return self._build_meta

        if not self._build_meta_res:
            return None

        raw_resource = self._build_meta_res.path.read_bytes()
        self._build_meta = json.loads(raw_resource)
        logging.debug('Build meta: {}'.format(self._build_meta))

        return self._build_meta

    @property
    def yammy_hash(self):
        if not self._yammy_hash:
            with self.timer("yammy:hash"):
                self._yammy_hash = rich_get_output(
                    [".bundles/yammy", "hash"],
                    self, "yammy_hash", cwd=self.APP_SRC_DIR
                ).strip()

            logging.info("Yammy hash: {}".format(self._yammy_hash))

        return self._yammy_hash

    def create_build_meta(self):
        logging.info("Preparing build meta")

        with self.timer("yammy:meta"):
            meta = rich_get_output(
                ["yammy", "ci", "meta"],
                self, "meta", cwd=self.APP_SRC_DIR
            )

        self._build_meta = json.loads(meta)
        logging.debug('Build meta: {}'.format(self.build_meta))

    def find_build_layer(self, **params):
        suffix = "hash" if "hash" in params else "commit"

        with self.timer("sb:find:{}:{}".format(params["stage"], suffix)):
            resource = MarketFrontYammyBuildArtefact.find(
                state=ctr.State.READY,
                attrs=params,
            ).order(-sdk2.Resource.id).first()

        logging.debug('Find {}'.format(params))
        logging.debug(resource)

        return resource

    def build_usage_report(self):
        self.watch_proc.kill()

        with self.timer("usage-report"):
            try:
                rich_check_call(
                    ["yammy", "bin", "usage-report"],
                    self, "usage-report", cwd=self.APP_SRC_DIR
                )

                if not self.Context.usage:
                    self.Context.usage = list()

                self.Context.usage.append("{}/{}".format(
                    get_resource_http_proxy_link(self.log_resource),
                    MarketFrontYammyBase.USAGE_REPORT_NAME,
                ))

            except Exception as e:
                pass

    def warmup(self):
        with self.timer("ya:warmup"):
            rich_check_call(["ya", "curl", "https://sandbox.yandex-team.ru/api/v1.0/user/current"], self,
                            "fake-ya-warmup")

    def on_save(self):
        super(MarketFrontYammyBase, self).on_save()
        propagate_github_event_data(self)

    def on_enqueue(self):
        self.timer_iteration()
        self.timer_end("wait:children", True)
        self.timer_start('task')
        self.timer_start('start')

        self.append_tags('yammy')
        super(MarketFrontYammyBase, self).on_enqueue()
        setup_container(self)

        self.update_status(GitHubStatus.PENDING, GitHubYammyStatusDescription.pending)

    def on_prepare(self):
        self.timer_start("prepare")

        with self.memoize_stage.styles_fix(max_runs=1):
            self.set_info(
                "Applying style fixes..."
                "<style>"
                ".common-html-content a.status { color: #fff }"
                "</style>",
                do_escape=False
            )

        super(MarketFrontYammyBase, self).on_prepare()

        with self.timer('setup:paths'):
            self.setup_paths()

        with self.timer('setup:env'):
            self.setup_env()

        (proc, log) = rich_check_start(
            ["bash", MarketFrontYammyBase.WATCH_SCRIPT_PATH],
            self, "usage", cwd=MarketFrontYammyBase.RESOURCES_DIR
        )

        self.watch_proc = proc

        self.warmup()

    def on_finish(self, prev_status, status):
        super(MarketFrontYammyBase, self).on_finish(prev_status, status)
        self.on_exit()

    def on_break(self, prev_status, status):
        super(MarketFrontYammyBase, self).on_break(prev_status, status)
        self.update_status(GitHubStatus.ERROR, GitHubYammyStatusDescription.stopped)
        self.on_exit()

    def on_wait(self, prev_status, status):
        super(MarketFrontYammyBase, self).on_wait(prev_status, status)
        self.on_exit()

    def on_terminate(self):
        super(MarketFrontYammyBase, self).on_terminate()
        self.on_exit()

    def on_failure(self, prev_status):
        super(MarketFrontYammyBase, self).on_failure(prev_status)
        self.on_exit()

    def on_exit(self):
        self.timer_end("finish")
        self.timer_end("task")

    def on_execute(self):
        self.timer_end("prepare")
        self.timer_end('start')
        self.timer_start("execute")

        super(MarketFrontYammyBase, self).on_execute()

        self.timer_start('setup:metatron')
        with self.metatron:
            self.timer_end('setup:metatron')

            try:
                self.setup_env_override()

                logging.debug(yaml.dump(dict(os.environ)))
                self.dump_debug_info()

                if self.Parameters.check_mergable:
                    self.check_mergable()

                self.bootstrap()

                with self.timer("run"):
                    self.run_task()

                self.update_status(GitHubStatus.SUCCESS, GitHubYammyStatusDescription.success)

            except MergeConflictException as e:
                logging.exception(e.message)
                self.set_info(GitHubYammyStatusDescription.conflict)
                self.update_status(GitHubStatus.FAILURE, GitHubYammyStatusDescription.conflict)
                raise

            except TestFailure as e:
                logging.exception(e.message)
                self.update_status(GitHubStatus.FAILURE, GitHubYammyStatusDescription.test_fail)

                if not self.Parameters.task_never_fail:
                    raise

            # Ошибка выполнения вызываемого скрипта - штатная ситация
            except (CalledProcessError, RichTextTaskFailure) as e:
                logging.exception(e.message)
                self.update_status(GitHubStatus.FAILURE, GitHubYammyStatusDescription.failure)

                if not self.Parameters.task_never_fail:
                    raise

            # Ошибка выполнения таски - нештатная ситуация
            except Exception as e:
                logging.exception(e.message)
                self.update_status(GitHubStatus.ERROR, GitHubYammyStatusDescription.error)
                raise

            except sdk2.WaitTask as e:
                logging.debug('Going to sleep and wait for child tasks ready')
                self.update_status(GitHubStatus.PENDING, GitHubYammyStatusDescription.pending)
                raise

            finally:
                self.timer_end("execute")
                self.build_usage_report()
                self.timer_start("finish")

    def setup_env_override(self):
        if self.Parameters.env_override:
            os.environ.update(self.Parameters.env_override)

    def setup_env(self):
        env = os.environ

        if self.Parameters.head_branch:
            env["CURRENT_GIT_BRANCH"] = str(self.Parameters.head_branch)

        env["ROOT_PATH"] = self.ROOT_DIR
        env["APP_SRC"] = self.APP_SRC_DIR
        env["PUBLISH_ROOT"] = self.PUBLISH_DIR
        env["YARN_CACHE_DIR"] = self.CACHE_DIR

        env['CI'] = '1'
        env['YENV'] = 'development'

        env["PATH"] = "{}:{}".format(
            os.path.join(env["APP_SRC"], 'node_modules', '.bin'),
            env["PATH"],
        )

        env["TASK_URL"] = get_task_link(self.id)
        env["TASK_AUTHOR"] = self.author

        env["PROCESS_LOG"] = str(self.log_path("process-usage.log"))
        env["USAGE_REPORT"] = str(self.log_path(MarketFrontYammyBase.USAGE_REPORT_NAME))

        if self.Parameters.head_branch != 'master':
            env["YAMMY_TICKET"] = self.ticket

        if self.Parameters.yammy_build_meta:
            env["BUILD_META"] = str(self._build_meta_res.path)

        if self.server.host != 'sandbox.yandex-team.ru':
            env["SANDBOX_HOST"] = "http://{}".format(self.server.host)

        self.setup_env_override()

    def find_prebuild(self):
        return self.Parameters.yammy_prebuilt

    def install_prebuilt(self, prebuilt):
        with self.timer("layer:{}:download".format(prebuilt.id)):
            res = sdk2.ResourceData(prebuilt)

        with self.timer("layer:{}:unpack".format(prebuilt.id)):
            logging.debug("Extracting prebuilt artefact {}".format(res.path))

            rich_check_call(
                ["tar", "-xf", str(res.path), "-C", self.APP_SRC_DIR],
                self, "prebuilt"
            )

    def run_bootstrap(self):
        logging.debug("No prebuilt artefact. Running clean bootstrap")

        with self.timer("bootstrap"):
            rich_check_call(
                ["yarn", "run", "bootstrap"],
                self, "bootstrap", cwd=self.APP_SRC_DIR
            )

    def bootstrap(self):
        logging.info("Running monorepo bootstrap")

        with self.timer("gh:clone"):
            self.git_helper.clone()

        os.environ["YAMMY_BASE_REF"] = self.base_commit

        rich_check_call(
            ["ln", "-fsT", "./lib", "./packages"],
            self, "fix-packages-symlink", cwd=self.APP_SRC_DIR
        )

        prebuilt = self.find_prebuild()

        if prebuilt:
            logging.info("Bootstrap from resource")
            self.install_prebuilt(prebuilt)
        else:
            logging.info("Bootstrap from repo")
            self.run_bootstrap()

    def run_task(self):
        pass
