# coding=utf-8

import re
import json
import logging

from sandbox import sdk2

from sandbox.projects.sandbox_ci.resources import SANDBOX_CI_ARTIFACT
from sandbox.projects.market.front.resources.yammy_resources import MarketFrontYammyMetaArtefact, \
    MarketFrontYammyBuildArtefact
from sandbox.projects.market.front.MarketFrontYammyBase import MarketFrontYammyBase
from sandbox.projects.market.front.helpers.sandbox_helpers import rich_get_output, rich_check_call, \
    get_resource_http_proxy_link
from sandbox.projects.market.front.helpers.github import GitHubStatus, GitHubYammyStatusDescription, get_pr_info
from sandbox.projects.market.front.helpers.yammy.requirements import YammyBigRequirements


class MarketFrontYammyBuildMeta(MarketFrontYammyBase):
    """
    Унифицированная таска для запуска yammy-based сценариев в монорепе
    """

    prebuilt = None

    class Requirements(YammyBigRequirements):
        pass

    class Parameters(MarketFrontYammyBase.Parameters):
        MarketFrontYammyBase.Parameters.description(default_value="yammy ci meta")

        with MarketFrontYammyBase.Parameters.yammy() as yammy:
            yammy_prebuilt = None
            yammy_build_digest = None
            yammy_build_meta = None

        with sdk2.parameters.Output:
            build_meta = sdk2.parameters.Resource(
                "Мета-информация для сборки",
                resource_type=MarketFrontYammyMetaArtefact
            )

            bootstrap = sdk2.parameters.Resource(
                "Инициализированный репозиторий",
                resource_type=MarketFrontYammyBuildArtefact
            )

            prebuilt = sdk2.parameters.Resource(
                "Начальный слой сборки",
                resource_type=MarketFrontYammyBuildArtefact
            )

            digest = sdk2.parameters.String(
                "Идентифиактор сборки"
            )

    @property
    def changed_packages(self):
        changed_packages = list()

        for (package_name, package) in self.build_meta["packages"].items():
            if package["needPublish"] or package["needTesting"]:
                changed_packages.append(package_name)

        return changed_packages

    def on_enqueue(self):
        super(MarketFrontYammyBuildMeta, self).on_enqueue()
        self.prepend_tags('meta')

    def publish_mq_hits(self):
        logging.info("Publishing MQ hints artefact")

        with self.timer("mq:create"):
            # MergeQueue ищет по типу ресурса, так что придётся дублировать
            res = SANDBOX_CI_ARTIFACT(
                self, 'Merge Queue Info Artifact',
                self.publish_path('digest.txt'),
                head_commit=self.git_helper.commit,
                base_commit=self.build_meta["baseRef"],
                branch=self.Parameters.head_branch,
                head_ref=self.head_ref,
                pr_number=str(self.Parameters.pr_number),
                repo=self.Parameters.github_repo,
                repo_full_name="{}/{}".format(self.Parameters.github_owner, self.Parameters.github_repo),
                changed_packages=",".join(self.changed_packages),
                type="changed-packages"
            )
            data = sdk2.ResourceData(res)

        with self.timer("mq:ready"):
            data.path.write_text(self.build_meta['digest'], 'utf8')
            data.ready()

    def publish_meta(self):
        logging.info("Publishing build meta")

        with self.timer("meta:create"):
            res = MarketFrontYammyMetaArtefact(
                self, 'Build metainfo',
                self.publish_path('build-meta.json'),
                head_commit=self.git_helper.commit,
                base_commit=self.build_meta["baseRef"],
                branch=self.Parameters.head_branch,
                head_ref=self.head_ref,
                pr_number=str(self.Parameters.pr_number),
                repo=self.Parameters.github_repo,
                repo_full_name="{}/{}".format(self.Parameters.github_owner, self.Parameters.github_repo),
                changed_packages=",".join(self.changed_packages),
            )
            data = sdk2.ResourceData(res)

        with self.timer("meta:ready"):
            data.path.write_bytes(json.dumps(self.build_meta))
            data.ready()

        self.Parameters.build_meta = res
        self.Parameters.digest = self.build_meta['digest']

        self.set_info('<b>Build meta:</b> <a class="status status_success" href="{}">{}</a>'.format(
            get_resource_http_proxy_link(res),
            "view"
        ), do_escape=False)

    def update_description(self):
        with self.timer("yammy:describe:html"):
            desc_html = rich_get_output(
                ["yammy", "describe", "html"],
                self, "describe.html", cwd=self.APP_SRC_DIR
            ).strip()

        if desc_html == '':
            return

        self.set_info(desc_html, do_escape=False)

        if not self.git_helper.pr:
            return

        with self.timer("yammy:describe:md"):
            desc_md = rich_get_output(
                ["yammy", "describe"],
                self, "describe.md", cwd=self.APP_SRC_DIR
            ).strip()

        desc_md = '\n<!-- YAMMY-DESCRIPTION-START --><hr/>\n\n{}\n<!-- YAMMY-DESCRIPTION-END -->'.format(desc_md)

        pr_info = get_pr_info(self.git_helper.owner, self.git_helper.repo, self.git_helper.pr)

        chunks = re.split(
            r'\n*<!-- YAMMY-DESCRIPTION-START -->[\s\S]*<!-- YAMMY-DESCRIPTION-END -->|(<section id="info">)',
            pr_info["body"]
        )

        if chunks[1] is None:
            chunks[1] = desc_md
        else:
            chunks.insert(1, desc_md)
        description = "".join(chunks)

        with self.timer("gh:description"):
            self.git_helper.update_pr_body(description)

    def update_plan(self):
        if not self.Parameters.commit_hash or not self.Parameters.github_context:
            return

        logging.info("Setting github statuses for planned tasks")

        for (stage, stage_tests) in self.build_meta["test"].items():
            if stage in ("bootstrap", "build"):
                continue

            for (package, test_list) in stage_tests.items():
                for test_config in test_list:
                    context = self.package_task_context(package, "test:{}".format(stage), test_config)
                    self.update_status(GitHubStatus.PENDING, GitHubYammyStatusDescription.planned, context, url="")

        for (stage, stage_tests) in self.build_meta["skipped"].items():
            if stage in ("bootstrap", "build"):
                continue

            for (package, test_list) in stage_tests.items():
                for test_config in test_list:
                    context = self.package_task_context(package, "test:{}".format(stage), test_config)
                    self.update_status(GitHubStatus.SUCCESS, GitHubYammyStatusDescription.skipped, context, url="")

    def find_prebuild(self):
        if self.git_helper.commit:
            self.prebuilt = self.find_build_layer(
                stage='bootstrap',
                hash=self.yammy_hash
            ) or self.find_build_layer(
                stage='bootstrap',
                commit=self.git_helper.commit
            )

        return self.prebuilt or super(MarketFrontYammyBuildMeta, self).find_prebuild()

    def run_bootstrap(self):
        last_layer = self.find_build_layer(stage='bootstrap')

        if last_layer:
            self.install_prebuilt(last_layer)

            with self.timer("git:reset"):
                rich_check_call(["git", "reset", "--hard"], self, "reset", cwd=self.APP_SRC_DIR)

            with self.timer("yammy:clean"):
                rich_check_call(["yammy", "clean"], self, "clean", cwd=self.APP_SRC_DIR)

        super(MarketFrontYammyBuildMeta, self).run_bootstrap()

    def run_task(self):
        with self.timer("yammy:validate"):
            rich_check_call(["yammy", "validate"], self, "validate", cwd=self.APP_SRC_DIR)

        if self.prebuilt:
            self.Parameters.bootstrap = self.prebuilt
        else:
            self.publish_prebuilt("bootstrap")

        self.create_build_meta()
        self.publish_meta()
        self.publish_mq_hits()

        with self.timer("info:description"):
            self.update_description()

        with self.timer("gh:plan"):
            self.update_plan()

        with self.timer("yammy:changelog"):
            changelog = rich_get_output(
                ["yammy", "changelogs", "sandbox"],
                self, "meta", cwd=self.APP_SRC_DIR
            )

        self.set_info('<h3>Changelog:</h3>{}'.format(changelog), do_escape=False)

        resource = self.find_build_layer(
            stage='build',
            commit=self.git_helper.commit,
            build=self.build_meta['digest']
        ) or self.find_build_layer(
            stage='prebuilt',
            commit=self.git_helper.commit,
            build=self.build_meta['digest']
        )

        if resource:
            self.Parameters.prebuilt = resource
            return

        if len(self.build_meta['prebuilt']) == 0:
            self.Parameters.prebuilt = self.Parameters.bootstrap
            return

        logging.debug('Installing prebuilt artefacts: {}'.format(self.build_meta['prebuilt'].keys()))

        for (artifact_name, artifact_id) in self.build_meta['prebuilt'].items():
            self.install_package_artefact(artifact_name, artifact_id)

        if len(self.build_meta["build"]) > 0:
            self.publish_prebuilt("prebuilt")
        else:
            self.Parameters.prebuilt = self.create_prebuild_resource("build")
