# coding=utf-8

import json
import logging

from sandbox import sdk2

from sandbox.projects.market.front.resources.yammy_resources import MarketFrontYammyPublishSummary
from sandbox.projects.market.front.MarketFrontYammyBase import MarketFrontYammyBase
from sandbox.projects.market.front.MarketFrontYammyPublishNpm import MarketFrontYammyPublishNpm
from sandbox.projects.market.front.MarketFrontYammyPublishResource import MarketFrontYammyPublishResource
from sandbox.projects.market.front.MarketFrontYammyPublishS3 import MarketFrontYammyPublishS3


class MarketFrontYammyPublishAll(MarketFrontYammyBase):
    """
    Таск сборки пакетов
    """

    class Parameters(MarketFrontYammyBase.Parameters):
        MarketFrontYammyBase.Parameters.description(default="Publish all changed packages")

        with MarketFrontYammyBase.Parameters.yammy() as yammy:
            yammy_build_meta = MarketFrontYammyBase.Parameters.yammy_build_meta(required=True)
            yammy_prebuilt = MarketFrontYammyBase.Parameters.yammy_prebuilt(required=True)

            with sdk2.parameters.String('Тип публикуемого ресурса', multiline=True) as yammy_publish_type:
                yammy_publish_type.values['npm'] = yammy_publish_type.Value('npm', default=True)
                yammy_publish_type.values['artefacts'] = yammy_publish_type.Value('artefacts')
                yammy_publish_type.values['templates'] = yammy_publish_type.Value('templates')

        with sdk2.parameters.Output:
            result = sdk2.parameters.Resource("Список опубликованных артефактов",
                                              resource_type=MarketFrontYammyPublishSummary)

    def skip_publish(self, publish_config):
        if self.Parameters.yammy_publish_type == 'npm' and publish_config['name'] == 'npm':
            return False

        if self.Parameters.yammy_publish_type == 'templates' and publish_config['name'] == 'template':
            return False

        if self.Parameters.yammy_publish_type in ('npm', 'templates') or publish_config['name'] in ('npm', 'template'):
            return True

        return False

    def on_enqueue(self):
        super(MarketFrontYammyPublishAll, self).on_enqueue()
        self.prepend_tags('publish', 'publish-all-{}'.format(self.Parameters.yammy_publish_type))

    def bootstrap(self):
        if not self.Context.tasks_ready:
            self.wait_tasks(self.create_tasks().values())

        self.Context.tasks_ready = True

    def create_tasks(self):
        with self.memoize_stage.create_tasks(max_runs=1):
            tasks = dict()

            for (package, publish_list) in self.build_meta["publish"].items():
                for publish_config in publish_list:
                    if self.skip_publish(publish_config):
                        continue

                    tasks["[{}] {}".format(package, publish_config["name"])] \
                        = self.run_package_task(package, 'publish', config=publish_config)

            self.Context.tasks = tasks

        return self.Context.tasks

    def result_text(self, result):
        if isinstance(result, (int, long, float, complex, basestring)):
            return '<span class="status status_success" style="text-transform: none;">{}</span>'.format(
                result)
        elif isinstance(result, sdk2.Resource):
            return '<a class="status status_success" href="//{}/resource/{}/view">resource: {}</a>'.format(
                self.server.host,
                result.id,
                result.id
            )
        else:
            raise Exception("Unknown result type")

    def result_value(self, result):
        if isinstance(result, (int, long, float, complex, basestring)):
            return result
        elif isinstance(result, sdk2.Resource):
            return dict(
                id=result.id,
                type=result.type.name,
                task_id=result.task_id,
                task_type=sdk2.Task[result.task_id].type.name
            )
        else:
            raise Exception("Unknown result type")

    def run_task(self):
        result = dict(
            sandboxResources=list(),
            npmVersions=list(),
            s3Upload=list(),
        )

        cell_style = 'style="padding: 0 10px;"'

        results = [('<table style="border-spacing: 0;">'
                    '<thead style="background: #eeeeee; font-weight: bold;"><tr>'
                    '<td {style}>Пакет</td>'
                    '<td {style}>Артефакт</td>'
                    '<td {style}>Таск</td>'
                    '<td {style}>Версия/Ресурс</td>'
                    '</tr></thead>').format(style=cell_style)]

        publish_results = []

        for (package, publish_list) in self.build_meta["publish"].items():
            for publish_config in publish_list:
                if self.skip_publish(publish_config):
                    continue

                task_id = self.Context.tasks["[{}] {}".format(package, publish_config["name"])]
                task = sdk2.Task[task_id]

                if task.type.name == MarketFrontYammyPublishNpm.name:
                    result['npmVersions'].append(dict(
                        packageName=package,
                        version=task.Parameters.result
                    ))
                elif task.type.name == MarketFrontYammyPublishResource.name:
                    result['sandboxResources'].append(dict(
                        packageName=package,
                        resourceName=publish_config["name"],
                        resourceId=task.Parameters.result.id,
                        resourceType=task.Parameters.result.type.name,
                        taskId=task_id,
                        taskType=task.type.name
                    ))
                elif task.type.name == MarketFrontYammyPublishS3.name:
                    result['s3Upload'].append(dict(
                        packageName=package,
                        path=task.Parameters.result
                    ))

                publish_results.append(dict(
                    package=package,
                    title=publish_config["config"]["title"],
                    task_id=task_id,
                    status=task.status,
                    result=task.Parameters.result,

                    weight='{}:{}'.format(
                        package,
                        publish_config["config"]["title"]
                    )
                ))

        def comparator(a, b):
            if a['weight'] > b['weight']:
                return 1

            if a['weight'] < b['weight']:
                return -1

            return 0

        publish_results.sort(comparator)

        for publish_result in publish_results:
            results.append((
                '<tr>'
                '<td {style}>'
                '<span class="status status_assigned"'
                ' style="font-size: 12px; text-transform: none;vertical-align: text-top;">{package}</span>'
                '</td>'
                '<td {style}><span class="status status_draft">{title}</span></td>'
                '<td {style}><a href="//{host}/task/{id}/view" class="status status_{status_class}">{status}</a></td>'
                '<td {style}>{result}</td>'
                '</tr>'
            ).format(
                package=publish_result['package'],
                title=publish_result["title"],
                host=self.server.host,
                id=publish_result['task_id'],
                status_class=publish_result['status'].lower(),
                status=publish_result['status'],
                style=cell_style,
                result=self.result_text(publish_result['result'])
            ))

        results.append('</table>')

        self.set_info('\n'.join(results), do_escape=False)

        res = MarketFrontYammyPublishSummary(
            self, "Publish artefacts summary",
            self.publish_path('publish.json'),
            commit=self.git_helper.commit
        )

        data = sdk2.ResourceData(res)

        logging.debug('Result: {}'.format(result))

        data.path.write_bytes(json.dumps(result))
        data.ready()

        self.Parameters.result = res
