# coding=utf-8

import logging
import json

from sandbox import sdk2

from sandbox.projects.market.front.helpers.yammy.errors import ResourceMissingExcpecion
from sandbox.projects.market.front.MarketFrontYammyBase import MarketFrontYammyBase
from sandbox.projects.market.front.MarketFrontYammyBuildMeta import MarketFrontYammyBuildMeta
from sandbox.projects.market.front.MarketFrontYammyBuildAll import MarketFrontYammyBuildAll
from sandbox.projects.market.front.MarketFrontYammyPublishAll import MarketFrontYammyPublishAll


class MarketFrontYammySpokBuildApp(MarketFrontYammyBase):
    pr_number = None

    class Parameters(MarketFrontYammyBase.Parameters):
        github_context = None
        github_use_merge_commit = None
        check_mergable = None
        yammy_build_digest = None
        yammy_build_meta = None
        yammy_prebuilt = None

        with sdk2.parameters.Group("Общие параметры") as common_params:
            ticket = sdk2.parameters.String("Тикет", required=True)

        with sdk2.parameters.Group("Параметры сервиса") as service_params:
            service_name = sdk2.parameters.String("Название сервиса", required=True)

        with sdk2.parameters.Output:
            result = sdk2.parameters.Resource("Собранный ресурс приложения")

    @property
    def ticket(self):
        return self.Parameters.ticket

    @property
    def service_npm_name(self):
        return "@yandex-market/{}".format(self.Parameters.service_name)

    @property
    def meta_task(self):
        return sdk2.Task[self.Context.meta_task]

    @property
    def build_task(self):
        return sdk2.Task[self.Context.build_task]

    @property
    def publish_task(self):
        return sdk2.Task[self.Context.publish_task]

    def on_enqueue(self):
        super(MarketFrontYammySpokBuildApp, self).on_enqueue()
        self.prepend_tags('spok', 'spok-publish')

    def bootstrap(self):
        pass

    def notify_stage(self, ready, stage, task_id):
        full_stage = "{}_{}".format("ready" if ready else "start", stage)
        with self.memoize_stage[full_stage](max_runs=1):
            if ready:
                self.set_info(
                    'Finished {stage} task: <a class="status status_success" href="//{host}/task/{id}">{id}</a>'.format(
                        stage=stage,
                        host=self.server.host,
                        id=task_id,
                    ), do_escape=False)
            else:
                self.set_info(
                    'Running {stage} task: <a class="status status_assigned" href="//{host}/task/{id}">{id}</a>'.format(
                        stage=stage,
                        host=self.server.host,
                        id=task_id,
                    ), do_escape=False)

    def run_meta(self):
        logging.info("Meta stage")

        with self.memoize_stage.create_meta_task(max_runs=1):
            task = MarketFrontYammyBuildMeta(
                self,
                **dict(
                    dict(self.Parameters),
                    description='Child task of <a class="status status_pending" href="//{host}/task/{id}">{id}</a>'.format(
                        host=self.server.host,
                        id=self.id,
                    ),
                    env_override=dict(
                        self.Parameters.env_override,
                        YAMMY_RELEASE=self.service_npm_name
                    )
                )
            )
            task.enqueue()
            self.Context.meta_task = task.id
            self.notify_stage(False, "build meta", self.Context.meta_task)

        self.wait_tasks([self.Context.meta_task])
        self.notify_stage(True, "build meta", self.Context.meta_task)

    def run_build(self):
        logging.info("Build stage")

        with self.memoize_stage.create_build_task(max_runs=1):
            meta_task = self.meta_task

            task = MarketFrontYammyBuildAll(
                self,
                **dict(
                    dict(self.Parameters),
                    description='Child task of <a class="status status_pending" href="//{host}/task/{id}">{id}</a>'.format(
                        host=self.server.host,
                        id=self.id,
                    ),
                    yammy_build_digest=meta_task.Parameters.digest,
                    yammy_build_meta=meta_task.Parameters.build_meta,
                    yammy_prebuilt=meta_task.Parameters.prebuilt,
                )
            )
            task.enqueue()
            self.Context.build_task = task.id
            self.notify_stage(False, "build all", self.Context.build_task)

        self.wait_tasks([self.Context.build_task])
        self.notify_stage(True, "build all", self.Context.build_task)

    def run_publish(self):
        logging.info("Publish stage")

        with self.memoize_stage.create_publish_task(max_runs=1):
            meta_task = self.meta_task
            build_task = self.build_task

            task = MarketFrontYammyPublishAll(
                self,
                **dict(
                    dict(self.Parameters),
                    description='Child task of <a class="status status_pending" href="//{host}/task/{id}">{id}</a>'.format(
                        host=self.server.host,
                        id=self.id,
                    ),
                    yammy_publish_type="artefacts",
                    yammy_build_digest=meta_task.Parameters.digest,
                    yammy_build_meta=meta_task.Parameters.build_meta,
                    yammy_prebuilt=build_task.Parameters.build,
                )
            )
            task.enqueue()
            self.Context.publish_task = task.id
            self.notify_stage(False, "publish all", self.Context.publish_task)

        self.wait_tasks([self.Context.publish_task])
        self.notify_stage(True, "publish all", self.Context.publish_task)

    def process_results(self):
        logging.info("Final stage")

        publish_task = self.publish_task
        publish_res = sdk2.ResourceData(publish_task.Parameters.result)
        publish_contents_raw = publish_res.path.read_bytes()
        publish_contents = json.loads(publish_contents_raw)

        logging.debug("Published results: {}".format(publish_contents))

        for result in publish_contents["sandboxResources"]:
            if result["packageName"] == self.service_npm_name:
                self.Parameters.result = sdk2.Resource[result["resourceId"]]
                return

        raise ResourceMissingExcpecion("Resource not found")

    def run_task(self):
        self.run_meta()
        self.run_build()
        self.run_publish()
        self.process_results()
