# coding=utf-8

import logging
import os
import re

from sandbox import sdk2

from sandbox.projects.common.vcs.arc import Arc

from sandbox.projects.market.front.helpers.yammy.errors import ResourceMissingExcpecion, UnknownErrorException, \
    ServiceAlreadyExistsException
from sandbox.projects.market.front.MarketFrontYammyBase import MarketFrontYammyBase
from sandbox.projects.market.front.helpers.sandbox_helpers import rich_check_call


class MarketFrontYammySpokConfigsTemplate(MarketFrontYammyBase):
    vcs_type = "arcadia"
    pr_number = None
    arc = None
    arcadia = None
    configs_root = "market/sre/nanny/front"

    class Parameters(MarketFrontYammyBase.Parameters):
        github_refs = None
        github_owner = None
        github_repo = None
        github_context = None
        head_branch = None
        github_use_merge_commit = None
        commit_hash = None
        check_mergable = None
        yammy_base_ref = None
        yammy_build_meta = None
        yammy_prebuilt = None
        yammy_build_digest = None

        with sdk2.parameters.Group("Общие параметры") as common_params:
            ticket = sdk2.parameters.String("Тикет", required=True)
            force_not_exist = sdk2.parameters.Bool("Падать если сервис уже существует", default_value=False)

        with sdk2.parameters.Group("Параметры шаблона") as template_params:
            template_name = sdk2.parameters.String("Тип шаблона", default_value="empty-template")

        with sdk2.parameters.Group("Параметры сервиса") as service_params:
            service_name = sdk2.parameters.String("Название сервиса", required=True)

        with sdk2.parameters.Output:
            pr_number = sdk2.parameters.Integer("Номер пул-реквеста")

    @property
    def ticket(self):
        return self.Parameters.ticket

    @property
    def branch(self):
        return "users/{}/sandboxtask-{}".format("robot-metatron", self.id)

    @property
    def service_path(self):
        return os.path.join(self.arcadia, self.configs_root, str(self.Parameters.service_name))

    @property
    def templates_path(self):
        return os.path.join(self.arcadia, self.configs_root, "templates")

    @property
    def service_npm_name(self):
        return "@yandex-market/{}".format(self.Parameters.service_name)

    @property
    def service_resource_name(self):
        return "MARKET_FRONT_{}_APP".format(self.Parameters.service_name.replace("-", "_").upper())

    @property
    def service_config_name(self):
        return "MARKET_FRONT_{}_CONFIG".format(self.Parameters.service_name.replace("-", "_").upper())

    @property
    def commit_message(self):
        return "{}: Создание конфигов сервиса {}".format(
            self.ticket,
            self.Parameters.service_name
        )

    @property
    def template_replacements(self):
        return dict(
            SERVICE_NAME=self.Parameters.service_name,
            NPM_PACKAGE_NAME=self.service_npm_name,
            SANDBOX_PACKAGE_NAME=self.service_resource_name,
            SANDBOX_CONFIG_NAME=self.service_config_name,
        )

    def on_enqueue(self):
        super(MarketFrontYammySpokConfigsTemplate, self).on_enqueue()
        self.prepend_tags('spok', 'spok-config')

    def find_template(self):
        template_path = os.path.join(self.templates_path, str(self.Parameters.template_name))

        if not os.path.exists(template_path):
            raise ResourceMissingExcpecion("Template not found")

        logging.debug("Template found:")
        logging.debug(template_path)

        self.set_info(
            'Using template: <a class="status status_success" href="https://a.yandex-team.ru/arc_vcs/{root}/templates/{template}">{root}/templates/{template}</a>'.format(
                root=self.configs_root,
                template=self.Parameters.template_name,
            ), do_escape=False)

        return template_path

    def install_template(self):
        logging.info("Loading template")

        template = self.find_template()

        with self.timer("template:install"):
            rich_check_call([
                "cp", "-vrT",
                os.path.relpath(template, self.arcadia),
                os.path.relpath(self.service_path, self.arcadia),
            ], self, "template-copy", cwd=self.arcadia)

    def apply_template(self):
        logging.info("Applying template")

        with self.timer("template:apply"):
            for root_path, dirs, files in os.walk(self.service_path):
                for file_name in files:
                    file_path = os.path.join(root_path, file_name)

                    with open(file_path, "r") as fd:
                        contents = fd.read()

                    for [key, value] in self.template_replacements.iteritems():
                        contents = contents.replace("{{" + key + "}}", value)

                    with open(file_path, "w") as fd:
                        fd.write(contents)

    def commit_template(self):
        logging.info("Creating commit")

        with self.timer("arcadia:add"):
            logging.debug("Arc: add")
            self.arc.add(self.arcadia, all_changes=True)

        with self.timer("arcadia:commit"):
            logging.debug("Arc: commit")
            self.arc.commit(self.arcadia, self.commit_message, all_changes=True)

        with self.timer("arcadia:push"):
            logging.debug("Arc: push")
            self.arc.push(self.arcadia, upstream=self.branch)

        with self.timer("arcadia:pr"):
            logging.debug("Arc: pr")
            pr = self.arc.pr_create(self.arcadia, message=self.commit_message, publish=True, auto=True)

        match_id = re.search(r'a\.yandex-team\.ru/review/(\d+)', pr)

        if match_id:
            self.pr_number = int(match_id.group(1))
            self.Parameters.pr_number = self.pr_number
            self.set_info(
                'PR created: <a class="status status_success" href="https://a.yandex-team.ru/review/{pr}">{pr}</a>'.format(
                    pr=self.pr_number,
                ), do_escape=False)
        else:
            raise UnknownErrorException("Cannot find pr number")

    def bootstrap(self):
        pass

    def checkout(self):
        logging.info("Checkout repo")
        self.arc.checkout(self.arcadia, branch=self.branch, create_branch=True)

    def run_task(self):
        self.arc = Arc(arc_oauth_token=os.environ["ARC_TOKEN"])

        with self.arc.mount_path(None, "trunk", fetch_all=False) as arcadia:
            self.arcadia = arcadia

            self.checkout()

            if os.path.exists(self.service_path):
                self.set_info(
                    'Service already exists: <a class="status status_success" href="https://a.yandex-team.ru/arc_vcs/{root}/{service}">{root}/{service}</a>'.format(
                        root=self.configs_root,
                        service=self.Parameters.service_name,
                    ), do_escape=False)

                if self.Parameters.force_not_exist:
                    raise ServiceAlreadyExistsException(
                        'Service configs {} already exists'.format(self.Parameters.service_name))

                return

            self.install_template()
            self.apply_template()
            self.commit_template()
