# coding=utf-8

import logging
import os
import json
import time

from sandbox import sdk2
import sandbox.common.types.resource as ctr

from sandbox.projects.market.front.helpers.yammy.errors import ResourceMissingExcpecion
from sandbox.projects.market.front.MarketFrontYammyBase import MarketFrontYammyBase
from sandbox.projects.market.front.helpers.sandbox_helpers import rich_check_call


class MarketFrontYammySpokServiceTemplate(MarketFrontYammyBase):
    pr_number = None

    class Parameters(MarketFrontYammyBase.Parameters):
        github_context = None
        head_branch = None
        github_use_merge_commit = None
        commit_hash = None
        check_mergable = None
        yammy_base_ref = None
        yammy_build_meta = None
        yammy_prebuilt = None
        yammy_build_digest = None

        with sdk2.parameters.Group("Общие параметры") as common_params:
            ticket = sdk2.parameters.String("Тикет", required=True)

        with sdk2.parameters.Group("Параметры шаблона") as template_params:
            template_name = sdk2.parameters.String("Тип шаблона", default_value="MARKET_FRONT_TEMPLATE_EMPTY_TEMPLATE")

        with sdk2.parameters.Group("Параметры сервиса") as service_params:
            service_name = sdk2.parameters.String("Название сервиса", required=True)

        with sdk2.parameters.Output:
            pr_number = sdk2.parameters.Integer("Номер пул-реквеста")

    @property
    def ticket(self):
        return self.Parameters.ticket

    @property
    def service_path(self):
        return os.path.join(self.APP_SRC_DIR, "lib", "apps", str(self.Parameters.service_name))

    @property
    def service_resource_name(self):
        return "MARKET_FRONT_{}_APP".format(self.Parameters.service_name.replace("-", "_").upper())

    @property
    def service_config_name(self):
        return "MARKET_FRONT_{}_CONFIG".format(self.Parameters.service_name.replace("-", "_").upper())

    @property
    def commit_message(self):
        return "{}: Создание сервиса {}".format(
            self.ticket,
            self.Parameters.service_name
        )

    @property
    def template_replacements(self):
        return dict(
            SERVICE_NAME=self.Parameters.service_name,
            NPM_PACKAGE_NAME=self.service_npm_name,
            SANDBOX_PACKAGE_NAME=self.service_resource_name,
            SANDBOX_CONFIG_NAME=self.service_config_name,
        )

    def on_enqueue(self):
        super(MarketFrontYammySpokServiceTemplate, self).on_enqueue()
        self.prepend_tags('spok', 'spok-create')

    def find_template(self):
        with self.timer("sb:find:{}".format(self.Parameters.template_name)):
            template = sdk2.Resource[self.Parameters.template_name] \
                .find(state=ctr.State.READY, head="master") \
                .order(-sdk2.Resource.id) \
                .first()

        if not template:
            raise ResourceMissingExcpecion("Template not found")

        logging.debug("Template found:")
        logging.debug(template)

        self.set_info('Using template: <a class="status status_success" href="//{host}/resource/{id}">{id}</a>'.format(
            id=template.id,
            host=self.server.host,
        ), do_escape=False)

        return template

    def install_template(self):
        logging.info("Loading template")

        template = self.find_template()

        with self.timer("template:download"):
            template_data = sdk2.ResourceData(template)

        with self.timer("template:unpack"):
            rich_check_call(["mkdir", "-p", self.service_path], self, "mkdir")
            rich_check_call(["tar", "-xf", str(template_data.path), "-C", self.service_path], self, "unpack")

    def choice_template(self):
        logging.info("Choosing template options")

        # Тут будет логика выбора опций шаблонизации - в монорепе или не в монорепе.
        # Сейчас реализован только вариант для монорепы
        with self.timer("template:choice"):
            rich_check_call(
                ["rm", "-rf", os.path.join(self.service_path, "package.standalone.json")],
                self, "choice-package-json"
            )

            with open(os.path.join(self.APP_SRC_DIR, "publish.json"), "w") as fd:
                fd.write("{}")

    def apply_template(self):
        logging.info("Applying template")

        with self.timer("template:apply"):
            with open(os.path.join(self.service_path, "package.json"), "r") as fd:
                contents = fd.read()

            for [key, value] in self.template_replacements.iteritems():
                contents = contents.replace("{{" + key + "}}", value)

            with open(os.path.join(self.service_path, "package.json"), "w") as fd:
                fd.write(contents)

    def commit_template(self):
        logging.info("Creating commit")

        with self.timer("template:commit"):
            self.git_helper.git.execute("checkout", "-b", self.ticket, cwd=self.APP_SRC_DIR)
            self.git_helper.git.execute("add", ".", cwd=self.APP_SRC_DIR)
            self.git_helper.git.execute("commit", "-m", self.commit_message, cwd=self.APP_SRC_DIR)
            self.git_helper.git.execute("push", "--set-upstream", "origin", self.ticket, cwd=self.APP_SRC_DIR)

    def create_pr(self):
        logging.info("Creating pull request")

        with self.timer("template:pr"):
            url = self.git_helper.api.get_request_url_by_path(
                "repos/{}/{}/pulls".format(self.git_helper.owner, self.git_helper.repo)
            )
            logging.debug("Github API call: {}".format(url))
            result = self.git_helper.api.http_send_request("post", url, dict(
                head=self.ticket,
                base="master",
                title=self.commit_message
            ))
            logging.debug("Github reply: {}".format(result.text))
            result.raise_for_status()

        data = json.loads(result.text)

        self.pr_number = data["number"]
        self.Parameters.pr_number = self.pr_number

        self.set_info(
            ('Pull request created:'
             ' <a class="status status_success" target="_blank"'
             ' href="https://github.yandex-team.ru/{owner}/{repo}/pull/{pr}"'
             '>{ticket}:{pr}</a>').format(
                owner=self.git_helper.owner,
                repo=self.git_helper.repo,
                ticket=self.ticket,
                pr=self.pr_number,
            ),
            do_escape=False
        )

        return self.pr_number

    def start_merge(self):
        logging.info("Start MergeQueue")

        with self.timer("pr:mq-start"):
            url = self.git_helper.api.get_request_url_by_path(
                "repos/{}/{}/issues/{}/comments".format(self.git_helper.owner, self.git_helper.repo, self.pr_number),
            )
            logging.debug("Github API call: {}".format(url))
            result = self.git_helper.api.http_send_request("post", url, dict(body="/merge"))
            logging.debug("Github reply: {}".format(result.text))
            result.raise_for_status()

    def start_review(self):
        logging.info("Start Review")

        with self.timer("pr:review-start"):
            time.sleep(30)
            url = self.git_helper.api.get_request_url_by_path(
                "repos/{}/{}/issues/{}/comments".format(self.git_helper.owner, self.git_helper.repo, self.pr_number),
            )
            logging.debug("Github API call: {}".format(url))
            result = self.git_helper.api.http_send_request("post", url, dict(body="/start"))
            logging.debug("Github reply: {}".format(result.text))
            result.raise_for_status()

    def bootstrap(self):
        logging.info("Checkout repo")
        self.git_helper.branch = "master"

        with self.timer("gh:clone"):
            self.git_helper.clone()

    def run_task(self):
        self.install_template()
        self.choice_template()
        self.apply_template()
        self.commit_template()
        self.create_pr()
        self.start_merge()
        self.start_review()
