# coding=utf-8

import json
import uuid
import logging

from sandbox import sdk2
from sandbox.common.types.task import Status
import sandbox.common.types.resource as ctr

from sandbox.projects.market.front.helpers.yammy.errors import TestFailure, ResourceMissingExcpecion
from sandbox.projects.market.front.MarketFrontYammyBase import MarketFrontYammyBase
from sandbox.projects.market.front.helpers.sandbox_helpers import get_resource_http_proxy_link
from sandbox.projects.market.front.helpers.yammy.test_params import make_test_params
from sandbox.projects.market.front.resources.yammy_resources import MarketFrontYammyTestsFailedArtefact


class MarketFrontYammyTestAll(MarketFrontYammyBase):
    """
    Таск сборки пакетов
    """

    _actual_test_list = None

    class Parameters(MarketFrontYammyBase.Parameters):
        MarketFrontYammyBase.Parameters.description(default="Test all changed packages")

        with MarketFrontYammyBase.Parameters.yammy() as yammy:
            yammy_build_meta = MarketFrontYammyBase.Parameters.yammy_build_meta(required=True)
            yammy_prebuilt = MarketFrontYammyBase.Parameters.yammy_prebuilt(required=True)

            yammy_stage = sdk2.parameters.String("Стадия тестирования", required=True)
            yammy_package = sdk2.parameters.String("Пакет для тестирования", required=False)

        test_params_group = make_test_params()

        with sdk2.parameters.Group("Перезапуск тестов") as test_restart_params:
            test_run_id = sdk2.parameters.String("Идентификатор запуска", default=uuid.uuid4())
            test_rerun_failed = sdk2.parameters.Bool("Перезапустить упавшие тесты")

        with sdk2.parameters.Output:
            run_id = sdk2.parameters.String("Созданный идентификатор запуска")
            success = sdk2.parameters.Bool("Успешность прогона тестов")
            failed_tests = sdk2.parameters.Resource(
                "Отчёт об упавших тестах",
                resource_type=MarketFrontYammyTestsFailedArtefact
            )

    @property
    def test_filters(self):
        if not self.Parameters.test_rerun_failed:
            return None

        with self.timer("failed-tests:find"):
            res = MarketFrontYammyTestsFailedArtefact.find(
                state=ctr.State.READY,
                attrs=dict(
                    run_id=self.Parameters.test_run_id,
                    test_stage=self.Parameters.yammy_stage
                )
            ).order(-sdk2.Resource.id).first()

        if not res:
            logging.error("Failed tests resource not found")
            raise ResourceMissingExcpecion()

        with self.memoize_stage.notify_failed_tests(max_runs=1):
            self.set_info("<a class='status status_failure' href='{}'>Ресурс упавших тестов: {}</a>".format(
                get_resource_http_proxy_link(res),
                res.id
            ), do_escape=False)

        with self.timer("failed-tests:load"):
            data = sdk2.ResourceData(res)
            raw = data.path.read_bytes()
            result = json.loads(raw)

        logging.info("Failed tests config: {}".format(result))

        return result

    @property
    def actual_test_list(self):
        if self._actual_test_list:
            return self._actual_test_list

        result = []
        filters = self.test_filters

        if self.Parameters.yammy_stage in self.build_meta["test"]:
            for (package, test_list) in self.build_meta["test"][self.Parameters.yammy_stage].items():
                if self.Parameters.yammy_package and package != self.Parameters.yammy_package:
                    continue

                if filters and (package not in filters):
                    continue

                for test_config in test_list:
                    if filters and (test_config["name"] not in filters[package]):
                        continue

                    result.append((package, test_config))

        self._actual_test_list = result

        return result

    def on_enqueue(self):
        super(MarketFrontYammyTestAll, self).on_enqueue()
        self.prepend_tags('test', 'test-all', 'test-stage-{}'.format(self.Parameters.yammy_stage))

    def bootstrap(self, prebuilt=None):
        if not self.Context.tasks_ready:
            self.wait_tasks(self.create_tasks().values(), accept_errors=True)

        self.Context.tasks_ready = True

    def create_tasks(self):
        with self.memoize_stage.create_tasks(max_runs=1):
            tasks = dict()

            for (package, test_config) in self.actual_test_list:
                tasks["[{}] {}".format(package, test_config["command"])] = self.run_package_task(
                    package, 'test:{}'.format(self.Parameters.yammy_stage),
                    task_type="MARKET_FRONT_YAMMY_TEST",
                    config=test_config,
                )

            self.Context.tasks = tasks

        return self.Context.tasks

    def run_task(self):
        cell_style = 'style="padding: 0 10px;"'

        results = [('<table style="border-spacing: 0;">'
                    '<thead style="background: #eeeeee; font-weight: bold;"><tr>'
                    '<td {style}>Пакет</td><td {style}>Проверка</td><td {style}>Статус</td><td {style}>Отчёт</td>'
                    '</tr></thead>'
                    '<tboby>').format(style=cell_style)]

        failed = False
        failed_tests = dict()
        test_results = []

        for (package, test_config) in self.actual_test_list:
            task_id = self.Context.tasks["[{}] {}".format(package, test_config["command"])]
            task = sdk2.Task[task_id]

            if task.status not in Status.Group.SUCCEED:
                failed = True

                if package not in failed_tests:
                    failed_tests[package] = dict()

                failed_tests[package][test_config["name"]] = dict(
                    title=test_config["config"]["title"],
                    taskId=task_id,
                    packageName=package,
                    testName=test_config["name"]
                )

            test_results.append(dict(
                package=package,
                title=test_config["config"]["title"],
                task_id=task_id,
                status=task.status,
                report=task.Parameters.result,

                weight='{}:{}:{}:{}'.format(
                    2 if task.status in Status.Group.SUCCEED else
                    0 if task.status in Status.Group.BREAK else 1,
                    task.status,
                    package,
                    test_config["config"]["title"]
                )
            ))

        def comparator(a, b):
            if a['weight'] > b['weight']:
                return 1

            if a['weight'] < b['weight']:
                return -1

            return 0

        test_results.sort(comparator)

        for test_result in test_results:
            result = (
                '<tr><td {style}>'
                '<span class="status status_assigned"'
                ' style="font-size: 12px; text-transform: none;vertical-align: text-top;">{package}</span>'
                '</td>'
                '<td {style}><span class="status status_draft">{title}</span></td>'
                '<td {style}><a href="//{host}/task/{id}/view" class="status status_{status_class}">{status}</a></td>'
            ).format(
                package=test_result['package'],
                title=test_result["title"],
                host=self.server.host,
                id=test_result['task_id'],
                status_class=test_result['status'].lower(),
                status=test_result['status'],
                style=cell_style,
            )

            if test_result['report']:
                result += (
                    '<td {style}><a class="status" href="{report}">Report</a>'
                    ' <a class="status" href="//{host}/resource/{id}/view">Resource: {id}</a></td>'
                ).format(
                    report="{}/{}".format(
                        get_resource_http_proxy_link(test_result['report']),
                        test_result['report'].report_file,
                    ),
                    host=self.server.host,
                    id=test_result['report'].id,
                    style=cell_style,
                )
            else:
                result += '<td {style}><span class="status status_failure">No report</span></td>'.format(
                    style=cell_style
                )

            result += '</tr>'

            results.append(result)

        results.append('</tbody></table>')

        self.set_info("".join(results), do_escape=False)

        self.Parameters.run_id = self.Parameters.test_run_id
        self.Parameters.success = not failed

        if failed:
            with self.timer("failed-tests:create"):
                res = MarketFrontYammyTestsFailedArtefact(
                    self,
                    "Артефакт упавших тестов. Стадия {}, пакет {}".format(
                        self.Parameters.yammy_stage,
                        self.Parameters.yammy_package or "*"
                    ),
                    self.publish_path('failed_tests.json'),
                    commit=self.git_helper.commit,
                    ref=self.head_ref,
                    test_stage=self.Parameters.yammy_stage,
                    run_id=self.Parameters.test_run_id,
                )

                data = sdk2.ResourceData(res)

            with self.timer("failed-tests:ready"):
                data.path.write_bytes(json.dumps(failed_tests))
                data.ready()

            self.Parameters.failed_tests = res

            raise TestFailure(self, "Some tests are not ok")
