# coding=utf-8

import logging
import os
import os.path

import sandbox.common.types.misc as ctm
from sandbox.common.types import task as ctt

from sandbox import sdk2
from sandbox.common.urls import get_task_link
from sandbox.projects.market.front.helpers.MetatronEnv import MetatronEnv
from sandbox.projects.market.front.helpers.github import clone_repo
from sandbox.projects.market.front.helpers.sandbox_helpers import rich_check_call, get_resource_http_proxy_link
from sandbox.projects.market.front.resources import MarketCIReport
from sandbox.projects.market.front.helpers.node import create_node_selector
from sandbox.projects.market.front.helpers.ubuntu import create_ubuntu_selector, setup_container
from sandbox.projects.market.front.helpers.startrack import ReleaseTicket, cut_text, link_text, colored_text, hidden_id
from sandbox.sandboxsdk.environments import PipEnvironment


class MarketSeoTestsIntegration(sdk2.Task):
    """
    Таска для запуска интеграционных SEO-тестов
    """

    APP_REPO_DIR = "app_src"
    APP_SRC_DIR = APP_REPO_DIR
    HTML_REPORT_DIR = "html_reports"
    HTML_REPORT_FILE = "index.html"

    GITHUB_TOKEN_VAULT_KEY = "robot-metatron-github-token"

    class Parameters(sdk2.Task.Parameters):
        app_owner = sdk2.parameters.String(
            "Github owner",
            default_value="market",
        )

        app_repo = sdk2.parameters.String(
            "Github repo",
            required=True
        )

        app_branch = sdk2.parameters.String(
            "Тестируемая ветка",
            required=True
        )

        app_src_dir = sdk2.parameters.String(
            "Кастомный путь корня приложения внутри репозитория"
        )

        base_url = sdk2.parameters.String(
            "Url инстанса",
            required=True
        )

        testpalm_run_issue = ReleaseTicket(
            'Релизный тикет',
            description='Пример: "MARKETVERSTKA-12345"'
        )

        check_env = sdk2.parameters.Dict("Переменные окружения для запуска проверки")

        ubuntu_version = create_ubuntu_selector()
        node_version = create_node_selector()

        platform = sdk2.parameters.String(
            "Платформа приложения",
            required=True
        )

    class Requirements(sdk2.Task.Requirements):
        dns = ctm.DnsType.DNS64
        environments = [
            PipEnvironment('yandex_tracker_client', version="1.3", custom_parameters=["--upgrade-strategy only-if-needed"]),
            PipEnvironment('startrek_client', version="2.3.0", custom_parameters=["--upgrade-strategy only-if-needed"])
        ]

    def on_enqueue(self):
        super(MarketSeoTestsIntegration, self).on_enqueue()
        setup_container(self)

    @property
    def check_env(self):
        env = os.environ.copy()

        env.update({
            "BASE_URL": self.Parameters.base_url,
        })

        if self.Parameters.check_env:
            env.update(self.Parameters.check_env)

        return env

    @property
    def html_report_path(self):
        return os.path.join(self.APP_SRC_DIR, self.HTML_REPORT_DIR)

    def on_execute(self):
        self.report_url = ''
        if self.Parameters.app_src_dir:
            self.APP_SRC_DIR = os.path.join(self.APP_REPO_DIR, self.Parameters.app_src_dir)

        with MetatronEnv(self, nodejs_version=self.Parameters.node_version):
            self._clone_repo()
            self._build()

            try:
                rich_check_call(
                    ["npm", "run", "integration:seo:" + self.Parameters.platform],
                    task=self, alias="npm_integration", cwd=self.APP_SRC_DIR, env=self.check_env
                )
            except Exception as e:
                self._exception = e
                raise
            finally:
                res = self._create_html_report_resource(self.html_report_path)
                self.report_url = get_resource_http_proxy_link(res)
                self.set_info(
                    "Отчёт: <a href=\"{url}/index.html\">{url}/index.html</a>".format(url=self.report_url),
                    do_escape=False
                )
                rich_check_call(
                    ["npm", "run", "ci:jest:seo:allure:" + self.Parameters.platform],
                    task=self, alias="npm_integration_allure", cwd=self.APP_SRC_DIR, env=self.check_env
                )

    def on_success(self, prev_status):
        self._send_report(ctt.Status.SUCCESS)

    def on_failure(self, prev_status):
        self._send_report(ctt.Status.FAILURE)

    def on_break(self, prev_status, status):
        self._send_report(ctt.Status.FAILURE)

    def _build(self):
        rich_check_call(
            ["make", "bootstrap"],
            task=self, alias="bootstrap", cwd=self.APP_SRC_DIR, env=self.check_env
        )

    def _clone_repo(self):
        clone_repo(
            self.Parameters.app_owner,
            self.Parameters.app_repo,
            self.Parameters.app_branch,
            self.APP_REPO_DIR
        )

    def _create_html_report_resource(self, report_path):
        report_res = MarketCIReport(self, "CI Report", report_path)

        report_res_data = sdk2.ResourceData(report_res)
        report_res_data.ready()

        return report_res

    def _send_report(self, status):
        actual_issue = self.Parameters.testpalm_run_issue

        if not actual_issue:
            logging.debug("Release ticket not specified")
            return

        if status == ctt.Status.SUCCESS:
            reported_status = colored_text('green', 'PASSED')
        else:
            reported_status = colored_text('red', 'FAILED')

        task_link = link_text(get_task_link(self.id), 'Таска')

        if self.report_url:
            additional_info = link_text(self.report_url + '/index.html', 'Отчет здесь')
        else:
            additional_info = cut_text('Инфо из таски', self.info)

        report = 'Интеграционные SEO-тесты ({platform}): {status}\n{task_link}\n{info}\n{hidden_id}'.format(
            platform=self.Parameters.platform,
            status=reported_status,
            task_link=task_link,
            info=additional_info,
            hidden_id=hidden_id('seo-'+ self.Parameters.platform + '-integration-report')
        )

        oauth_token = sdk2.Vault.data("robot-metatron-st-token")

        from startrek_client import Startrek

        st = Startrek(useragent="robot-metatron", token=oauth_token)

        issue = st.issues[actual_issue]
        issue.comments.create(text=report)

        logging.debug("Report was sent to {}".format(actual_issue))
