# -*- coding: utf-8 -*-

import logging
import os
import shutil

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp

import ssh
import gnupg
import node

VAULT_OWNER = "MARKET"

SSH_VAULT_KEY = "robot-metatron-ssh"

GIT_USERNAME = "Robot Metatron"
GIT_EMAIL = "robot-metatron@yandex-team.ru"

GPG_PUBRING_VAULT_KEY = "robot-metatron-gpg-public"
GPG_SECRING_VAULT_KEY = "robot-metatron-gpg-private"
GPG_KEY_ID = "BA27E445"

S3_KEYID_VAULT_KEY = "robot-metatron-s3-keyid"
S3_SECRETKEY_VAULT_KEY = "robot-metatron-s3-secretkey"

ST_OAUTH_TOKEN_VAULT_KEY = "robot-metatron-st-token"

TESTPALM_API_TOKEN_VAULT_KEY = "robot-metatron-testpalm-token"

TESTPALM_OAUTH_API_TOKEN_VAULT_KEY = "robot-metatron-testpalm-api-token"

BUNKER_API_TOKEN_VAULT_KEY = "robot-metatron-bunker-token"

ARCANUM_API_TOKEN_VAULT_KEY = "robot-metatron-arcanum-token"

ARC_TOKEN_VAULT_KEY = "robot-metatron-arc-oauth"

GITHUB_TOKEN_VAULT_KEY = "robot-metatron-github-token"

TUS_OAUTH_TOKEN_KEY = 'robot-metatron-tus'

YA_TOOL_TOKEN_KEY = "robot-metatron-ya-token"

YQL_TOKEN_VAULT_KEY = "robot-metatron-yql-token"

NPMRC_VALUT_KEY = "robot-metatron-npm-auth"

TSUM_TOKEN_VAULT_KEY = "robot-metatron-tsum-token"

ABT_TOKEN_VAULT_KEY = "robot-metatron-abt-token"

NODE_EXTRA_CA_CERTS = "/usr/share/yandex-internal-root-ca/YandexInternalRootCA.crt"


class MetatronEnv(object):
    def __init__(
        self,
        task,
        ssh_env=True,
        git_config=True,
        gpg_keyring=True,
        s3_env=True,
        nodejs_version=node.NODE_DEFAULT,
        st_env=True,
        testpalm_env=True,
        bunker_env=True,
        arcanum_env=True,
        arc_env=True,
        github_env=True,
        tus_env=True,
        tsum_env=True,
        ya_env=True,
        yql_env=True,
        npm_env=True,
        abt_env=True,
    ):
        logging.error("ENV __INIT__")
        super(MetatronEnv, self).__init__()

        self.task = task
        self.ssh_env = ssh_env
        self.git_config = git_config
        self.s3_env = s3_env
        self.gpg_keyring = gpg_keyring
        self.nodejs_version = nodejs_version
        self.st_env = st_env
        self.testpalm_env = testpalm_env
        self.bunker_env = bunker_env
        self.arcanum_env = arcanum_env
        self.arc_env = arc_env
        self.github_env = github_env
        self.tus_env = tus_env
        self.tsum_env = tsum_env
        self.ya_env = ya_env
        self.yql_env = yql_env
        self.npm_env = npm_env
        self.abt_env = abt_env

        self.__env_backup = {}
        self.__ssh_key = None
        self.__gpg_key = None

    @staticmethod
    def nodejs_only(task, node_version):
        return MetatronEnv(
            task,
            nodejs_version=node_version, npm_env=True,
            ssh_env=False, git_config=False, gpg_keyring=False, s3_env=False, st_env=False, testpalm_env=False,
            bunker_env=False, arcanum_env=False, arc_env=False, github_env=False, tus_env=False, tsum_env=False, ya_env=False,
            yql_env=False,
        )

    def __enter__(self):
        self.__env_backup = os.environ.copy()

        os.environ["NODE_EXTRA_CA_CERTS"] = NODE_EXTRA_CA_CERTS

        if self.ssh_env:
            self.__ssh_key = ssh.Key(self.task, VAULT_OWNER, SSH_VAULT_KEY)
            self.__ssh_key.__enter__()

        if self.git_config:
            if os.path.isfile(os.path.expanduser("~/.gitconfig")):
                shutil.copyfile(
                    os.path.expanduser("~/.gitconfig"),
                    os.path.expanduser("~/.gitconfig.bak"),
                )

            sp.check_call(["git", "config", "--global", "user.name", GIT_USERNAME])
            sp.check_call(["git", "config", "--global", "user.email", GIT_EMAIL])

        if self.gpg_keyring:
            os.environ["GPGKEY"] = GPG_KEY_ID
            self.__gpg_key = gnupg.GpgKey(GPG_SECRING_VAULT_KEY, GPG_PUBRING_VAULT_KEY)
            self.__gpg_key.__enter__()

        if self.s3_env:
            os.environ["AWS_ACCESS_KEY_ID"] = sdk2.Vault.data(S3_KEYID_VAULT_KEY)
            os.environ["AWS_SECRET_ACCESS_KEY"] = sdk2.Vault.data(S3_SECRETKEY_VAULT_KEY)

        if self.nodejs_version:
            node_path = node.get_node_path(self.nodejs_version)
            os.environ["PATH"] = "{}:{}".format(node_path, os.environ["PATH"])

        if self.st_env:
            os.environ["ST_OAUTH_TOKEN"] = sdk2.Vault.data(ST_OAUTH_TOKEN_VAULT_KEY)

        if self.testpalm_env:
            os.environ["TESTPALM_API_TOKEN"] = sdk2.Vault.data(TESTPALM_API_TOKEN_VAULT_KEY)
            os.environ["TESTPALM_OAUTH_API_TOKEN"] = sdk2.Vault.data(TESTPALM_OAUTH_API_TOKEN_VAULT_KEY)

        if self.bunker_env:
            os.environ["BUNKER_API_TOKEN"] = sdk2.Vault.data(BUNKER_API_TOKEN_VAULT_KEY)

        if self.arcanum_env:
            os.environ["ARCANUM_API_TOKEN"] = sdk2.Vault.data(ARCANUM_API_TOKEN_VAULT_KEY)

        if self.arc_env:
            os.environ["ARC_TOKEN"] = sdk2.Vault.data(ARC_TOKEN_VAULT_KEY)

        if self.github_env:
            os.environ["GITHUB_API_TOKEN"] = sdk2.Vault.data(GITHUB_TOKEN_VAULT_KEY)

        if self.tus_env:
            os.environ["TUS_OAUTH_TOKEN"] = sdk2.Vault.data(TUS_OAUTH_TOKEN_KEY)

        if self.tsum_env:
            # Внимание! Этот токен подходит для работы с https://tsum.yandex-team.ru/api,
            # и не подходит для https://tsum-api.market.yandex.net:4203. Для работы со вторым использовать
            # любой валидный OAuth токен (например ST_OAUTH_TOKEN)
            os.environ["TSUM_OAUTH_TOKEN"] = sdk2.Vault.data(TSUM_TOKEN_VAULT_KEY)

        if self.ya_env:
            os.environ["YA_TOKEN"] = sdk2.Vault.data(YA_TOOL_TOKEN_KEY)

        if self.yql_env:
            os.environ["YQL_TOKEN"] = sdk2.Vault.data(YQL_TOKEN_VAULT_KEY)

        if self.abt_env:
            os.environ["ABT_API_TOKEN"] = sdk2.Vault.data(ABT_TOKEN_VAULT_KEY)

        if self.npm_env:
            if os.path.isfile(os.path.expanduser("~/.npmrc")):
                shutil.copyfile(
                    os.path.expanduser("~/.npmrc"),
                    os.path.expanduser("~/.npmrc.bak"),
                )

            with open(os.path.expanduser("~/.npmrc"), "w") as f:
                f.write(sdk2.Vault.data(NPMRC_VALUT_KEY))

            sp.check_call(["npm", "config", "set", "script-shell", 'bash'])

    def __exit__(self, exc_type, exc_val, exc_tb):
        if self.ssh_env:
            self.__ssh_key.__exit__(exc_type, exc_val, exc_tb)

        if self.git_config:
            if os.path.isfile(os.path.expanduser("~/.gitconfig.bak")):
                shutil.move(
                    os.path.expanduser("~/.gitconfig.bak"),
                    os.path.expanduser("~/.gitconfig"),
                )
            elif os.path.isfile(os.path.expanduser("~/.gitconfig")):
                os.remove(os.path.expanduser("~/.gitconfig"))

        if self.gpg_keyring:
            self.__gpg_key = gnupg.GpgKey(GPG_SECRING_VAULT_KEY, GPG_PUBRING_VAULT_KEY)
            self.__gpg_key.__exit__(exc_type, exc_val, exc_tb)

        if self.npm_env:
            if os.path.isfile(os.path.expanduser("~/.npmrc.bak")):
                shutil.move(
                    os.path.expanduser("~/.npmrc.bak"),
                    os.path.expanduser("~/.npmrc"),
                )
            elif os.path.isfile(os.path.expanduser("~/.npmrc")):
                os.remove(os.path.expanduser("~/.npmrc"))

        os.environ.clear()
        os.environ.update(self.__env_backup)
