# coding=utf-8

import json

from sandbox import common, sdk2
from sandbox.sdk2 import Vault
from sandbox.sdk2.vcs.git import Git
from sandbox.projects.sandbox_ci.utils.github import GitHubApi
from sandbox.projects.market.front.helpers.sandbox_helpers import rich_get_output
from sandbox.projects.sandbox_ci.utils.request import send_request, make_request, RETRY_METHODS

GITHUB_TOKEN_VAULT_KEY = "robot-metatron-github-token"


class GitHubStatus(common.utils.Enum):
    SUCCESS = "success"
    PENDING = "pending"
    ERROR = "error"
    FAILURE = "failure"


class GitHubStatusDescription(common.utils.Enum):
    pending = "Проверка в очереди →"
    success = "Перейти к отчету →"
    error = "Неудача! Смотри отчёт →"
    failure = "Свалились без отчёта, смотри логи →"
    stopped = "Задача прервана"
    conflict = "PR с конфликтом. Задача не выполнялась"


class GitHubYammyStatusDescription(common.utils.Enum):
    planned = "Задача запланирована"
    skipped = "Задача пропущена"
    pending = "Задача в очереди →"
    success = "Перейти к отчету →"
    error = "Задача упала →"
    failure = "Случилась ошибка →"
    test_fail = "Тесты красные →"
    stopped = "Задача прервана"
    conflict = "PR с конфликтом. Задача не выполнялась"


class GitHubCloneStrategy(common.utils.Enum):
    branch = "Branch",
    pr = "Pull request"


class FriendlyGitHubApi(GitHubApi):
    def get_request_url_by_path(self, path):
        return '{}/{}'.format(self._url, path)

    def http_send_request(self, method, url, data, **kwargs):
        headers = {}

        if self._token:
            headers.update({
                'Authorization': 'token {}'.format(self._token),
                'Content-Type': 'application/json'
            })

        return send_request(method, url, headers=headers, data=json.dumps(data), max_retries=6, **kwargs)

    def http_send_request_post(self, path, data, **kwargs):
        url = self.get_request_url_by_path(path)
        return self.http_send_request('post', url, data, **kwargs)

    @staticmethod
    def post_request_with_retries(method, url, headers, max_retries, retry_methods=RETRY_METHODS):
        factor = 0.8
        methods = frozenset(list(retry_methods) + ['POST'])

        return make_request(method, url, headers, max_retries, backoff_factor=factor, retry_methods=methods)


class GitHubHelper:
    __merge_bases = dict()

    def __init__(self, owner, repo, path,
                 strategy=GitHubCloneStrategy.branch, branch=None, commit=None, pr=None, refs=None):
        token_proxy_host = 'github-token-proxy.si.yandex-team.ru'
        token = _get_token()

        self.owner = owner
        self.repo = repo
        self.path = str(path)
        self.strategy = strategy
        self.branch = branch
        self.pr = pr
        self.commit = commit
        self.refs = refs
        self.api = FriendlyGitHubApi(token=token) if token else FriendlyGitHubApi(host=token_proxy_host)
        self.git = Git(self.url, filter_branches=strategy == GitHubCloneStrategy.branch)

    @property
    def full_repo_name(self):
        return "{}/{}".format(self.owner, self.repo)

    @property
    def url(self):
        return "git@github.yandex-team.ru:{}/{}.git".format(self.owner, self.repo)

    def clone(self):
        if self.strategy == GitHubCloneStrategy.pr:
            self.git.clone(self.path, "pull/{}/merge".format(self.pr))

            if self.branch:
                self.git.execute("checkout", "-B", self.branch, cwd=self.path)
        else:
            self.git.clone(self.path, self.branch, self.commit)

        if self.branch:
            self.git.execute("branch", "-C", self.branch, "origin/{}".format(self.branch), cwd=self.path)

        refs = None
        if self.refs:
            refs = list(self.refs)
            if self.branch and self.branch in refs:
                refs.remove(self.branch)

        if refs and len(refs) > 0:
            self.git.update_cache_repo(*self.refs)

            cmd = ["fetch", "--progress", "origin"]
            cmd.extend("+{0}:{0}".format(ref) for ref in self.refs)
            self.git.execute(*cmd, cwd=self.path)

        self.commit = rich_get_output(
            ["git", "rev-parse", "HEAD"], sdk2.task.Task.current,
            "commit_hash", cwd=self.path
        ).strip()

    def merge_base(self, branch="master"):
        if branch in self.__merge_bases:
            return self.__merge_bases[branch]

        base = rich_get_output(
            ["git", "merge-base", self.commit, branch], sdk2.task.Task.current,
            "merge-base.{}".format(branch.replace('/', '-')), cwd=self.path
        ).strip()

        self.__merge_bases[branch] = base
        return base

    def pr_info(self):
        return self.api.get_pr_info(self.owner, self.repo, self.pr)

    def update_pr_body(self, body):
        url = self.api.get_request_url_by_path('repos/{owner}/{repo}/pulls/{pull_number}'.format(
            owner=self.owner,
            repo=self.repo,
            pull_number=self.pr,
        ))

        self.api.http_send_request('patch', url, {"body": body})


def _get_token():
    token = None

    try:
        # В on_enqueue нет доступка к секретам, поэтому будем делать запросы через специальный сервер
        # @see: SANDBOX-3882, FEI-10158
        token = Vault.data(GITHUB_TOKEN_VAULT_KEY)
    except common.errors.VaultNotAllowed as err:
        # TODO: определять как-то получше.
        if "task session token" in str(err):
            return token
        raise

    return token


def _get_api():
    host = 'github-token-proxy.si.yandex-team.ru'
    token = _get_token()

    return GitHubApi(token=token) if token else GitHubApi(host=host)


def build_repo_url(owner, name):
    """
    Собирает URL к репозиторию.
    """
    return 'git://github.yandex-team.ru/{}/{}.git'.format(owner, name)


def change_status(owner, repo, context, sha, url, state, description):
    """
    Меняет статус хука в PR.
    """
    api = _get_api()

    return api.create_status(owner, repo, context, sha, state, url, description)


def get_pr_info(owner, repo, number):
    """
    Возвращает информацию о PR.
    """
    api = _get_api()

    return api.get_pr_info(owner, repo, number)


def clone_repo(owner, name, ref, target_dir):
    """
    Клонирует репозиторий.
    """
    url = build_repo_url(owner, name)
    git = Git(url)
    git.clone(str(target_dir), ref)


def clone_repo_for_diff(owner, name, master_ref, working_ref, target_dir):
    """
    Клонирует репозиторий и чекаутит на целевую ветку.
    """
    formated_target_dir = str(target_dir)
    url = build_repo_url(owner, name)
    git = Git(url)
    git.clone(formated_target_dir, master_ref)
    git.execute("fetch", "origin", working_ref, cwd=formated_target_dir)
    git.checkout(formated_target_dir, working_ref)


def clone_repo_for_merge_commit(owner, name, commit_sha, target_dir):
    """
    Клонирует репозиторий и чекаутит merge коммит.
    """
    url = build_repo_url(owner, name)
    git = Git(url)
    git.clone(str(target_dir), "master")
    git.update_cache_repo("refs/pull/*")
    git.execute("checkout", "-f", commit_sha, cwd=str(target_dir))


def get_branch(owner, repo, branch):
    """
    Возвращает информацию о ветке
    @see https://developer.github.com/v3/repos/branches/#get-branch
    """
    api = _get_api()

    return api.get_branch(owner, repo, branch)


def search_issues(q, sort):
    """
    Возвращает найденные ишью/пулл-реквесты
    @see https://developer.github.com/v3/search/#search-issues-and-pull-requests
    """
    api = _get_api()

    return api.request(
        'search/issues',
        # Если передавать Dict, то параметры энкодятся, и api гитхаба их не переваривает, поэтому собираю строку
        params='q={}&sort={}'.format(q, sort)
    )


def find_opened_release_branch(owner, repo, branch_starts_with):
    """
    Возвращает ветку текущего релиза или None
    @see https://developer.github.com/v3/search/#search-issues-and-pull-requests
    """
    pull_requests = search_issues(
        q='repo:{owner}/{repo}+is:pr+is:open+head:{branch_starts_with}'.format(
            owner=owner,
            repo=repo,
            branch_starts_with=branch_starts_with
        ),
        sort='created',
    )

    if pull_requests['total_count'] == 0:
        return None

    api = _get_api()
    release_pr = api.get_pr_info(owner, repo, pull_requests['items'][0]['number'])

    return release_pr['head']['ref']
