# coding=utf-8
import logging
import json
import os
import hashlib

import sandbox.sdk2.path as spath
from sandbox import sdk2

logger = logging.getLogger(__name__)
level = 'error'
project = 'market_autotests_hermione'
language = 'nodejs'

TEST_TYPE_RELEASE = 'release'
TEST_TYPE_FEATURE = 'feature'
TEST_TYPE_MASTER = 'master'

# DEPRECATED, используй market/frontarc/helpers/metrics

def create_metrics_params():
    class MetricsParameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group('Metrics') as metrics_block:
            upload_metrics = sdk2.parameters.Bool(
                'Сохранять статистику и ошибки в ресурсы',
                default=True,
            )
            send_metrics = sdk2.parameters.Bool(
                'Выгружать статистику и ошибки в logbroker',
                default=False,
            )
            logbroker_endpoint = sdk2.parameters.String(
                'Хост логброкера для записи статистики',
                default="logbroker.yandex.net",
            )
            logbroker_port = sdk2.parameters.Integer(
                'Порт логброкера для записи статистики',
                default=2135,
            )
            logbroker_topic = sdk2.parameters.String(
                'Топик логброкера для записи статистики',
                default="market-front-infra/stable/autotest-stats",
            )
            logbroker_errors_topic = sdk2.parameters.String(
                'Топик логброкера для записи ошибок',
                default="market-front-infra/market-infra-error-booster",
            )

    return MetricsParameters


def select_metrics_parameters(taskParameters):
    return {
        'upload_metrics': taskParameters.upload_metrics,
        'send_metrics': taskParameters.send_metrics,
        'logbroker_endpoint': taskParameters.logbroker_endpoint,
        'logbroker_port': taskParameters.logbroker_port,
        'logbroker_topic': taskParameters.logbroker_topic,
    }


def collect_cases_stats(resource_path, ticket, repo, branch, platform, task_id, is_retry):
    allure_cases_dir_path = os.path.join(resource_path, 'data/test-cases')

    cases_stats = []
    for test_case in os.listdir(allure_cases_dir_path):
        test_case_data_raw = spath.Path(allure_cases_dir_path, test_case).read_bytes()
        test_case_data = json.loads(test_case_data_raw)

        test_name = test_case_data.get('name')
        md5_hash = hashlib.md5()
        md5_hash.update(test_name)
        test_id = md5_hash.hexdigest()[:7]

        item = {
            'ticket': ticket,
            'repo': repo,
            'branch': branch,
            'platform': platform,
            'sandboxTaskId': task_id,
            'testId': test_id,
            'testName': test_name,
            'isRetry': 1 if test_case_data.get('retry') or is_retry else 0,
            'status': test_case_data.get('status'),
            'timestamp': int(test_case_data.get('time', {}).get('start') / 1000),
            'duration': test_case_data.get('time', {}).get('duration'),
            'tags': [],
        }

        if 'kadavr' in test_case_data.get('parameterValues'):
            item.get('tags').append('kadavr')

        cases_stats.append(item)

    return cases_stats


def collect_cases_errors(resource_path, ticket, repo, branch, platform, task_id):
    allure_cases_dir_path = os.path.join(resource_path, 'data/test-cases')
    allure_attachments_dir_path = os.path.join(resource_path, 'data/attachments')

    cases_errors = []
    for test_case in os.listdir(allure_cases_dir_path):
        test_case_data_raw = spath.Path(allure_cases_dir_path, test_case).read_bytes()
        test_case_data = json.loads(test_case_data_raw)

        status = test_case_data.get('status')

        if status != 'broken' and status != 'failed':
            continue

        kadavr_session_id_file_name = None
        kadavr_session_id = None

        for step in test_case_data.get('testStage', {}).get('steps', []):
            for attach in step.get('attachments', []):
                if attach.get('name') == 'kadavr_session_id':
                    kadavr_session_id_file_name = attach.get('source')
                    break
            if kadavr_session_id_file_name is not None:
                break

        if kadavr_session_id_file_name is not None:
            for attach_file_name in os.listdir(allure_attachments_dir_path):
                if kadavr_session_id_file_name == attach_file_name:
                    attach_file_raw = spath.Path(allure_attachments_dir_path, attach_file_name).read_bytes()
                    kadavr_session_id = attach_file_raw
                    break

        item = {
            'stack': test_case_data.get('statusTrace'),
            'level': level,
            'file': test_case_data.get('source'),
            'message': test_case_data.get('statusMessage'),
            'project': project,
            'service': platform,
            'block': test_case_data.get('name'),
            'language': language,
            'timestamp': int(test_case_data.get('time', {}).get('stop')),
            'additional': {
                'ticket': ticket,
                'repo': repo,
                'branch': branch,
                'sandboxTaskId': str(task_id),
                'testType': get_test_type(branch)
            }
        }

        if kadavr_session_id is not None:
            item.get('additional')['kadavrSessionId'] = kadavr_session_id

        for label in test_case_data.get('labels', []):
            item.get('additional')[label.get('name')] = label.get('value')

        cases_errors.append(item)

    return cases_errors


def get_test_type(branch):
    test_type = TEST_TYPE_FEATURE

    if branch == 'master':
        test_type = TEST_TYPE_MASTER

    if 'release/' in branch:
        test_type = TEST_TYPE_RELEASE

    return test_type
