# -*- coding: utf-8 -*-

import hashlib
import json
import logging
import tarfile
import time
from pprint import pformat
from typing import List

import requests

from sandbox.projects.sandbox_ci.utils.request import send_request

CHUNK_SIZE_BYTES = 1048576  # 1 MByte
RETRY_COUNT = 5
RETRY_PAUSE_SECONDS = 10


def send_report_to_tsum(archive, tsum_api_hosts, sandbox_resource_type, sandbox_ticket_num, sandbox_resource_id=None):
    if not tsum_api_hosts or not len(tsum_api_hosts):
        raise RuntimeError("Parameters tsum_api_hosts must be sets")

    headers = {'Content-type': 'application/json'}
    report_json = json.dumps(_build_md5_report(archive, sandbox_resource_type, sandbox_ticket_num, sandbox_resource_id))

    for tsum_api_host in tsum_api_hosts:
        for i in range(RETRY_COUNT):
            try:
                logging.debug("Send report to TSUM host %s", tsum_api_host)

                report_url = tsum_api_host + '/agent/addResourceInfo'
                response = requests.post(url=report_url, data=report_json, headers=headers)

                response.raise_for_status()
                break
            except StandardError as e:
                logging.info("Error %s", str(e.message))

                if i == RETRY_COUNT - 1:
                    raise RuntimeError("Can't send report to tsum")
                else:
                    time.sleep(i * RETRY_PAUSE_SECONDS)


def _build_md5_report(archive_path, sandbox_resource_type, sandbox_task_id, sandbox_resource_id):
    logging.debug("Build report for %s", archive_path)

    checksums = []
    tar = tarfile.open(archive_path)
    members = filter(lambda m: m.isfile(), tar.getmembers())

    for sub_file in members:
        sub_file = tar.extractfile(sub_file)

        if sub_file:
            checksums.append({
                "path": sub_file.name,
                "md5": _get_md5(sub_file)
            })
            sub_file.close()

    tar.close()

    with open(archive_path) as tar_archive:
        md5_report = {
            "resource": {
                "resourceType": sandbox_resource_type,
                "resourceId": sandbox_resource_id,
                "taskId": sandbox_task_id
            },
            "md5": _get_md5(tar_archive),
            "checksums": checksums
        }

    logging.debug("Report body %s", pformat(md5_report))

    return md5_report


def _get_md5(check_file):
    hash_md5 = hashlib.md5()

    while True:
        file_bytes = check_file.read(CHUNK_SIZE_BYTES)
        if file_bytes:
            hash_md5.update(file_bytes)
        else:
            break

    return hash_md5.hexdigest()


TSUM_TELEGRAM_MESSAGE_API_URL = "https://tsum-api.market.yandex.net:4203/v1/bot/message?onlyCommandProcessing=false"

CHATID_FIELD = "chat_id"
TEXT_FIELD = "text"
PARSEMODE_FIELD = "parse_mode"
DISABLEWEBPAGEPREVIEW_FIELD = "disable_web_page_preview"
DISABLENOTIFICATION_FIELD = "disable_notification"
REPLYTOMESSAGEID_FIELD = "reply_to_message_id"
REPLYMARKUP_FIELD = "reply_markup"
TELEGRAM_MAX_MESSAGE_LEN = 4095

# См. org.telegram.telegrambots.meta.api.methods.send.SendMessage
# https://a.yandex-team.ru/arc/trunk/arcadia/market/infra/tsum/tsum-api/src/main/java/ru/yandex/market/tsum/api/telegram/TsumBotMessagingController.java
KNOWN_TSUM_TG_ARGS = [
    CHATID_FIELD,
    TEXT_FIELD,
    PARSEMODE_FIELD,
    DISABLEWEBPAGEPREVIEW_FIELD,
    DISABLENOTIFICATION_FIELD,
    REPLYTOMESSAGEID_FIELD,
    REPLYMARKUP_FIELD,
]


def send_telegram_message_via_tsum_bot(
    token,  # type: str
    chat_id,  # type: str
    text,  # type: str
    **kwargs
):
    """
    Однократно пытается отправить сообщение в канал в телеграм от имени ЦУМ бота
    Сообщение должно быть короче 4096 символов (ограничение телеги)
    :param token: любой валидный OAuth токен. Проверки на наличие грантов для цум-апи нет.
    :param chat_id: ID чата в телеграм. Можно получить у ЦУМ-бота командой /getchatid
    :param text: Текст сообщения
    :param kwargs: Любые другие известные аргументы API ЦУМа. см. константу KNOWN_TSUM_TG_ARGS
    :return: Ответ tsum_api (requests.models.Response)
    """
    if token is None or len(token) == 0:
        raise Exception("Не указан oauth_token для обращения к ЦУМ api")
    if len(text) > TELEGRAM_MAX_MESSAGE_LEN:
        logging.error("Сообщение слишком длинное! {} > {}, ЦУМ не отправит сообщение.".format(len(text), TELEGRAM_MAX_MESSAGE_LEN))
        # Не падаем, но в логи ошибку роняем. А то ответ ЦУМа понять сложно.

    headers = {
        'Authorization': 'OAuth {}'.format(token),
        'Content-Type': 'application/json'
    }

    # Оставляем из kwargs только те, про которые знаем что их поддерживает tsum api
    body = {known_arg: kwargs[known_arg] for known_arg in KNOWN_TSUM_TG_ARGS if known_arg in kwargs.keys()}
    body[CHATID_FIELD] = chat_id
    body[TEXT_FIELD] = text
    logging.debug("Шлём сообщение с текстом:\n{}".format(text))

    return send_request(
        method="post",
        url=TSUM_TELEGRAM_MESSAGE_API_URL,
        headers=headers,
        data=json.dumps(body),
    )  # type: requests.models.Response


def send_long_message_list(
    token,  # type: str
    chat_id,  # type: str
    messages_list,  # type: List
    msg_splitter="\n",  # type: str
    concat_messages=True,  # type: bool
    **kwargs
):
    """
    Пытается отправить список сообщений
    :param token: любой валидный OAuth токен. Проверки на наличие грантов для цум-апи нет.
    :param chat_id: ID чата в телеграм. Можно получить у ЦУМ-бота командой /getchatid
    :param messages_list: Список сообщений для отправки. Сообщения будут склеиваиться между собой через msg_splitter
    :param msg_splitter: Строка, которой будут джойниться между собой сообщения
    :param concat_messages: Если False, то каждое сообщение из списка будет отправлено отдельно, без склейки.
    :param kwargs: Любые другие известные аргументы API ЦУМа. см. константу KNOWN_TSUM_TG_ARGS
    :return: Ничего
    """
    text = msg_splitter.join(messages_list)

    if concat_messages and len(text) <= TELEGRAM_MAX_MESSAGE_LEN:
        logging.info("Отправляем сообщение в чат через ЦУМ")
        response = send_telegram_message_via_tsum_bot(
            token,
            chat_id,
            text,
            **kwargs
        )
        logging.info("Ответ ЦУМа: {}, {}".format(response.status_code, response.text))
        return

    logging.info("Сообщение слишком длинное, нужно отправлять по частям.")
    text_chunks = []
    chunk_messages = []
    chunk_len = 0
    for message in messages_list:
        len_msg = len(message) + len(msg_splitter)
        if len_msg > TELEGRAM_MAX_MESSAGE_LEN:
            for chunk in chunkstring(message, TELEGRAM_MAX_MESSAGE_LEN):
                text_chunks.append(chunk)
            continue

        if not concat_messages:
            text_chunks.append(message)
            continue

        if chunk_len + len_msg > TELEGRAM_MAX_MESSAGE_LEN:
            # Добавляем готовый чанк в список сообщений на отправку и начинаем готовить новый
            text_chunks.append(msg_splitter.join(chunk_messages))
            chunk_messages = []
            chunk_len = 0

        chunk_messages.append(message)
        chunk_len += len_msg

    if chunk_len > 0:
        text_chunks.append(msg_splitter.join(chunk_messages))

    for text_chunk in text_chunks:
        logging.info("Отправляем часть сообщения в чат через ЦУМ")
        response = send_telegram_message_via_tsum_bot(
            token,
            chat_id,
            text_chunk,
            **kwargs
        )
        logging.info("Ответ ЦУМа: {}, {}".format(response.status_code, response.text))


# https://stackoverflow.com/questions/18854620/whats-the-best-way-to-split-a-string-into-fixed-length-chunks-and-work-with-the
def chunkstring(string, length):
    """
    This function returns a generator, using a generator comprehension. The generator returns the string sliced,
    from 0 + a multiple of the length of the chunks, to the length of the chunks + a multiple of the length of the chunks.
    """
    return (string[0 + i:length + i] for i in range(0, len(string), length))


def escape_telegram_markdown(
    message  # type: str
):
    return message\
        .replace("_", "\\_")\
        .replace("*", "\\*")\
        .replace("[", "\\[")\
        .replace("`", "\\`")

# #############################################################################################


ID_TEMPLATE_WITH_PULL = "{project}_{git_owner}/{git_repo}_{git_branch}_{pipeline_id}_pull{pull_request_id}"
ID_TEMPLATE = "{project}_{git_owner}/{git_repo}_{git_branch}_{pipeline_id}"
URL_TEMPLATE = "https://tsum.yandex-team.ru/api/projects/{project}/per_commit/state/"


def restart_percommit(auth_token, project, git_owner, git_repo, git_branch, pipeline_id, pull_request_id=None):
    """
    front-market-unified_market/marketfront_MARKETFRONT-68926-visible-checkout-cart_per-commit_pull1581844 - пример ID покоммитного пайплайна в цуме
    :param auth_token: Токен TSUM_TOKEN_VAULT_KEY
    :param project: Проект в ЦУМе. Пример: front-market-unified
    :param git_owner: Владелец репозитория в гитхабе. Пример: market
    :param git_repo: Репозиторий в гитахбе. Пример: marketfront
    :param git_branch: Ветка в репозитории. Пример: MARKETFRONT-68926-visible-checkout-cart
    :param pipeline_id: Вид покоммитного пайплайна для запуска. Например: per-commit, из https://tsum.yandex-team.ru/pipe/projects/front-market-unified/pipelines
    :param pull_request_id: ID (не номер!) пулл-реквеста.
    :return: Ответ tsum_api (requests.models.Response)
    """
    if auth_token is None or len(auth_token) == 0:
        raise Exception("Не указан oauth_token для обращения к ЦУМ api")

    headers = {
        # Attention! Тут запрос на tsum.yandex-team.ru/api , поэтому тут не OAuth!
        'Authorization': auth_token,
        'Content-Type': 'application/json'
    }

    if pull_request_id is None:
        internal_id = ID_TEMPLATE.format(
            project=project,
            git_owner=git_owner,
            git_repo=git_repo,
            git_branch=git_branch,
            pipeline_id=pipeline_id,
        )
    else:
        internal_id = ID_TEMPLATE_WITH_PULL.format(
            project=project,
            git_owner=git_owner,
            git_repo=git_repo,
            git_branch=git_branch,
            pipeline_id=pipeline_id,
            pull_request_id=pull_request_id,
        )

    body = {
        'id': internal_id,
    }

    logging.info("Пытаемся перезапустить пайплайн {}".format(internal_id))

    return send_request(
        method="post",
        url=URL_TEMPLATE.format(project=project),
        headers=headers,
        data=json.dumps(body),
    )  # type: requests.models.Response
