# coding=utf-8

import logging

import time

from sandbox import sdk2, common

from sandbox.projects.market.front.helpers.yammy.task_helpers import YammyTaskHelpers
from sandbox.projects.market.front.helpers.yammy.errors import MergeConflictException
from sandbox.projects.market.front.helpers.github import change_status, get_pr_info, GitHubHelper, GitHubCloneStrategy


# значения проставляются в on_save c помощью propagate_github_event_data
class YammyGitParameters(sdk2.Task.Parameters):
    with sdk2.parameters.Group('Github') as github:
        github_owner = sdk2.parameters.String("Github owner", default_value="market")
        github_repo = sdk2.parameters.String("Github repo", default_value="monomarket")
        github_context = sdk2.parameters.String("Контекст в github")
        head_branch = sdk2.parameters.String("Тестируемая ветка")
        pr_number = sdk2.parameters.Integer("Номер пулл-реквеста")
        github_use_merge_commit = sdk2.parameters.Bool("Использовать merge-коммит", default=False)
        commit_hash = sdk2.parameters.String("Хэш коммита, которому проставляется статус")
        check_mergable = sdk2.parameters.Bool("Проверять на конфликты мёржа", default=False)

        github_refs = sdk2.parameters.List(
            "Дополнительные ref'ы которые нужно получить",
            value_type=sdk2.parameters.String,
            default=["master"]
        )


class YammyGitHelpers(object):
    # Статус мержа из хука. Если None — значит, на момент отправки хука ещё неизвестен.
    pr_is_mergeable = None

    __git_helper = None
    __base_commit = None

    @staticmethod
    def github_context(context):
        return "[YAMMY] {}".format(context)

    @property
    def git_helper(self):
        if not self.__git_helper:
            use_merge_commit = self.Parameters.pr_number > 0 and self.Parameters.github_use_merge_commit

            self.__git_helper = GitHubHelper(
                self.Parameters.github_owner, self.Parameters.github_repo, self.APP_SRC_DIR,
                strategy=GitHubCloneStrategy.pr if use_merge_commit else GitHubCloneStrategy.branch,
                branch=self.Parameters.head_branch,
                pr=self.Parameters.pr_number,
                commit=self.Parameters.commit_hash,
                refs=self.Parameters.github_refs,
            )

        return self.__git_helper

    @property
    def head_commit(self):
        if self.Parameters.commit_hash:
            return self.Parameters.commit_hash
        else:
            return self.git_helper.commit

    @property
    def base_commit(self):
        if self.__base_commit:
            return self.__base_commit

        if self.Parameters.yammy_base_ref:
            return self.Parameters.yammy_base_ref

        if self.Parameters.yammy_build_meta:
            return self.build_meta['baseRef']

        self.__base_commit = self.git_helper.merge_base()
        return self.__base_commit

    @property
    def head_ref(self):
        if self.Parameters.pr_number:
            head_ref = "pull/{}".format(self.Parameters.pr_number)
            if self.Parameters.github_use_merge_commit:
                head_ref = "{}/merge".format(head_ref)
        elif self.Parameters.head_branch:
            head_ref = self.Parameters.head_branch
        else:
            head_ref = self.git_helper.commit

        return head_ref

    @property
    def is_pr_mergeable(self):
        pr_info = get_pr_info(
            str(self.Parameters.github_owner),
            str(self.Parameters.github_repo),
            int(self.Parameters.pr_number)
        )

        return pr_info["mergeable"]

    def package_task_context(self, package, task_name, config):
        if not self.Parameters.github_context:
            return None

        return "[{}] [{}] {}".format(
            YammyTaskHelpers.abbrev_package(package),
            task_name,
            config["config"]["title"]
        )

    def update_status(self, state, description, context=None, url=None):
        if not context:
            context = self.Parameters.github_context

        if not self.Parameters.commit_hash or not context:
            return

        with self.timer("gh:status:{}:{}".format(state, description)):
            if url is None:
                url = common.utils.get_task_link(self.id)

            context = YammyGitHelpers.github_context(context)

            logging.debug("Setting github status on {}. context='{}', state='{}', description='{}', link='{}'".format(
                self.Parameters.commit_hash, context, state, description, url
            ))

            change_status(
                owner=self.Parameters.github_owner,
                repo=self.Parameters.github_repo,
                sha=self.Parameters.commit_hash,
                context=context,
                state=state,
                description=description,
                url=url,
            )

    def check_mergable(self):
        if not self.Parameters.pr_number:
            return

        with self.timer('gh:mergable'):
            if self.pr_is_mergeable is None:
                for i in range(5, 120, 5):
                    logging.debug(
                        "Github doesn't know, if PR is mergeable yet. Sleeping {} seconds and retrying".format(i)
                    )
                    time.sleep(i)
                    mergeable = self.is_pr_mergeable
                    if mergeable is not None:
                        self.pr_is_mergeable = mergeable
                        break

        if not self.pr_is_mergeable:
            raise MergeConflictException("Can't run check on a confilcting branch")
