# coding=utf-8

import re
import os
import json
import logging
from collections import OrderedDict

from sandbox import sdk2
from sandbox.common.types.task import Status
from sandbox.common.utils import chain
from sandbox.common.errors import TaskError
import sandbox.common.types.resource as ctr

from sandbox.projects.market.front.resources.yammy_resources import MarketFrontYammyBuildArtefact, \
    MarketFrontYammyDebugInfo, MarketFrontYammyMetaArtefact, PREBUILT_TTLS
from sandbox.projects.market.front.helpers.MetatronEnv import MetatronEnv
from sandbox.projects.market.front.helpers.node import create_node_selector
from sandbox.projects.market.front.helpers.ubuntu import create_ubuntu_selector
from sandbox.projects.market.front.helpers.sandbox_helpers import rich_check_call, get_resource_http_proxy_link
from sandbox.projects.market.front.helpers.yammy.json_encoder import SafeJsonEncoder

FINAL_STATUSES = set(chain(Status.Group.FINISH, Status.Group.BREAK))


class YammyTaskParameters(sdk2.Task.Parameters):
    kill_timeout = 1200

    with sdk2.parameters.Group('Окружение') as envs:
        ubuntu_version = create_ubuntu_selector()
        node_version = create_node_selector()
        env_override = sdk2.parameters.Dict("Переменные окружения")


class YammyTaskHelpers(object):
    ROOT_DIR = None
    APP_SRC_DIR = None
    PUBLISH_DIR = None
    CACHE_DIR = None

    vcs_type = "git"

    _metatron = None

    @sdk2.header()
    def header(self):
        links = OrderedDict()

        if self.ticket:
            links["Ticket"] = "https://st.yandex-team.ru/{ticket}".format(
                ticket=self.ticket
            )

        if self.Parameters.pr_number:
            if self.vcs_type == "arcadia":
                links["Pull_Request"] = "https://a.yandex-team.ru/review/{pr}".format(pr=self.Parameters.pr_number)
            else:
                links["Pull_Request"] = "https://github.yandex-team.ru/{owner}/{repo}/pull/{pr}".format(
                    owner=self.Parameters.github_owner,
                    repo=self.Parameters.github_repo,
                    pr=self.Parameters.pr_number
                )

                if self.Parameters.head_branch and self.Parameters.head_branch != "master":
                    links["TSUM"] = \
                        "https://tsum.yandex-team.ru/pipe/projects/market-frontech/multitestings/environments/{tsum}".format(
                            tsum=re.sub(r'_|\/', '-', self.Parameters.head_branch.lower())
                        )

        debug_info = MarketFrontYammyDebugInfo.find(task=self, state=ctr.State.READY).first()
        if debug_info:
            links["env.json"] = "{}/{}".format(get_resource_http_proxy_link(debug_info), 'env.json')

        if self.Parameters.yammy_build_meta:
            links["meta.json"] = get_resource_http_proxy_link(self.Parameters.yammy_build_meta)

        meta = MarketFrontYammyMetaArtefact.find(task=self, state=ctr.State.READY).first()
        if meta:
            links["meta.json"] = get_resource_http_proxy_link(meta)

        if self.Context.usage:
            i = 0
            for report in self.Context.usage:
                links['usage_{}'.format(i)] = report
                i += 1

        return ' '.join([
            '<a class="status status_assigned" href="{}">{}</a>'.format(href, title)
            for (title, href) in links.items()
        ])

    @property
    def metatron(self):
        if not self._metatron:
            self._metatron = MetatronEnv(self, nodejs_version=self.Parameters.node_version)

        return self._metatron

    def wait_tasks(self, tasks, accept_errors=False, title="children"):
        tasks_ready = True

        for task_id in tasks:
            task = sdk2.Task[task_id]

            if task.status not in FINAL_STATUSES:
                tasks_ready = False
                break

            if task.status not in Status.Group.SUCCEED and not accept_errors:
                raise TaskError('Subtask {} failed'.format(task_id))

        if not tasks_ready:
            logging.info('Some builds not ready. Waiting...')
            self.timer_start("wait:{}".format(title), True)
            raise sdk2.WaitTask(tasks, FINAL_STATUSES, wait_all=True)

    @staticmethod
    def abbrev_package(package):
        chunks = package.split('/')
        if len(chunks) == 1:
            return package

        return '@' + ''.join([x[0] for x in chunks[0][1:].split('-')]).upper() + '/' + chunks[1]

    @property
    def current_tags(self):
        tags = []
        for tag in self.Parameters.tags:
            tags.append(str(tag))
        return tags

    def dump_debug_info(self):
        with self.memoize_stage.debug_info(max_runs=1), self.timer('dump:debug'):
            logging.info("Dumping debug info")

            with open(self.publish_path("debug", "param.json"), "w") as fd:
                json.dump(dict(self.Parameters), fd, cls=SafeJsonEncoder)

            env = dict(os.environ)
            for key in env.keys():
                if key.endswith("_TOKEN") or key.endswith("_ACCESS_KEY"):
                    del env[key]

            with open(self.publish_path("debug", "env.json"), "w") as fd:
                json.dump(env, fd, cls=SafeJsonEncoder)

            res = MarketFrontYammyDebugInfo(
                self, "Отладочная информация",
                self.publish_path("debug")
            )
            data = sdk2.ResourceData(res)
            data.ready()

    def append_tags(self, *tags):
        current_tags = self.current_tags

        for tag in tags:
            tag_formated = tag.upper()
            if tag_formated not in current_tags:
                self.Parameters.tags.append(tag_formated)

    def prepend_tags(self, *tags):
        tags = list(tags)
        tags.reverse()
        current_tags = self.current_tags

        for tag in tags:
            tag_formated = tag.upper()
            if tag_formated not in current_tags:
                self.Parameters.tags.insert(0, tag_formated)

    def setup_paths(self):
        root = str(self.path().absolute())

        if not os.path.exists('publish'):
            os.mkdir('publish')

        self.PUBLISH_DIR = os.path.join(root, 'publish')
        self.CACHE_DIR = os.path.join(root, 'yarn-cache')

        if self.ramdrive and not self.ramdrive.path:
            logging.debug('Ramdrive path not found')

        if self.ramdrive and self.ramdrive.path:
            root = str(self.ramdrive.path.absolute())
            logging.debug('Using ram drive at {}'.format(root))
        else:
            logging.debug('Using disk drive at {}'.format(root))

        self.ROOT_DIR = root
        self.APP_SRC_DIR = os.path.join(root, 'sources')

    def publish_path(self, *chunks):
        path = self.PUBLISH_DIR

        for chunk in chunks[0:-1]:
            path = os.path.join(path, chunk)
            if not os.path.exists(path):
                os.mkdir(path)

        return os.path.join(self.PUBLISH_DIR, *chunks)

    def install_package_artefact(self, artefact_name, artefact_id):
        logging.info('Installing package build artefact {}:{}'.format(artefact_name, artefact_id))

        with self.timer("{}:{}:download".format(artefact_name, artefact_id)):
            res = sdk2.Resource[artefact_id]
            data = sdk2.ResourceData(res)

        with self.timer("{}:{}:unpack".format(artefact_name, artefact_id)):
            rich_check_call(
                ["tar", "-xf", str(data.path), "-C", self.APP_SRC_DIR],
                self, "install.{}".format(artefact_name.replace('#', '--').replace(":", "-").replace('/', '__'))
            )

    def create_prebuild_resource(self, stage):
        if stage not in ("bootstrap", "prebuilt", "build"):
            raise TypeError("Stage must be one of 'bootstrap', 'prebuilt', 'build', but got '{}'".format(stage))

        logging.info("Publishing {} artefact".format(stage))

        with self.timer("layer:{}:create".format(stage)):
            res = MarketFrontYammyBuildArtefact(
                self, "Build artefact: {}".format(stage),
                self.publish_path(stage, 'build.tar.gz'),
                commit=self.git_helper.commit,
                ref=self.head_ref,
                stage=stage,
                build=self.build_meta['digest'] if stage != 'bootstrap' else None,
                hash=self.yammy_hash,
                ttl=PREBUILT_TTLS[stage]
            )

            data = sdk2.ResourceData(res)

        with self.timer("layer:{}:pack".format(stage)):
            rich_check_call(
                ["yammy", "ci", "pack-layer", str(data.path)],
                self, "{}.pack".format(stage), cwd=self.APP_SRC_DIR
            )

        with self.timer("layer:{}:ready".format(stage)):
            data.ready()

        return res

    def publish_prebuilt(self, stage):
        with self.timer("layer:{}".format(stage)):
            res = self.create_prebuild_resource(stage)

        # Использовать self.Parameters[stage] нельзя:
        # TypeError: 'Parameters' object does not support item assignment
        if stage == "bootstrap":
            self.Parameters.bootstrap = res
        elif stage == "prebuilt":
            self.Parameters.prebuilt = res
        else:
            self.Parameters.build = res

    def run_package_task(self, package, title, task_type=None, config={}, custom_params={}):
        logging.info("Creating task for {}, with config: {}".format(title, config))

        if "task" in config["config"]:
            task_type = config["config"]["task"]

        env_override = self.Parameters.env_override.copy()
        if "env" in config["config"]:
            env_override.update(config["config"]["env"])

        if "params" in config["config"]:
            custom_params = dict(custom_params, **config["config"]["params"])

        if not task_type:
            raise TypeError("Task type must be defined")

        parent_params = dict(self.Parameters)
        del parent_params["task_never_fail"]
        if "yammy_package" in parent_params:
            del parent_params["yammy_package"]

        task = sdk2.Task[task_type](
            self,

            yammy_package=package,
            yammy_command=config["command"],
            yammy_command_name=config["name"],
            yammy_command_title=config["config"]["title"],

            **dict(
                parent_params,

                description='<span class="status">{}</span>'
                            ' <span class="status status_assigned"'
                            ' style="font-size: 12px; text-transform: none;vertical-align: text-top;">{}</span>'
                            ' <span class="status status_draft">{}</span>'
                            '\n<b>{}</b>'
                    .format(title,
                            package,
                            config["config"]["title"],
                            config["command"]),

                github_context=self.package_task_context(package, title, config),

                env_override=env_override,

                **custom_params
            )
        )

        with self.timer("child:{}:enqueue".format(task.id)):
            task.enqueue()

        return task.id
