import math
import time

from sandbox import sdk2


class Timer(object):
    task = None
    name = None

    def __init__(self, task, name):
        self.task = task
        self.name = name

    def __enter__(self):
        self.task.timer_start(self.name)

    def __exit__(self, exc_type, exc_val, exc_tb):
        self.task.timer_end(self.name)


class YammyTimeHelpers(object):
    @staticmethod
    def format_time(ts):
        round_time = math.trunc(ts)
        ms = str(int(round(ts - round_time, 1) * 10))
        seconds = int(round_time % 60)
        minutes = int(math.floor((round_time - seconds) / 60))

        if minutes:
            return "{}:{}.{}".format(str(minutes).zfill(2), str(seconds).zfill(2), ms)
        else:
            return "{}.{}".format(seconds, ms)

    @staticmethod
    def make_timers(now, source, prefix=None, task_id=None):
        return [dict(
            name=name if not prefix else "{}:{}".format(prefix, name),
            start=info["start"],
            end=info["end"] if "end" in info else now,
            duration=(info["end"] if "end" in info else now) - info["start"],
            task_id=task_id,
        ) for (name, info) in source.items()]

    @staticmethod
    def build_report(timers):
        timers.sort(lambda a, b: 1 if a["start"] > b["start"] else -1)

        run = 0
        execute = 0
        for t in timers:
            if ":run" in t["name"]:
                run += t["duration"]
            if ":execute" in t["name"]:
                execute += t["duration"]

        start = min(timers, key=lambda t: t["start"])["start"]
        end = max(timers, key=lambda t: t["end"])["end"]
        duration = end - start

        styles = (
            '.summary {margin-bottom: 30px; display: flex;}\n'
            '.summary-col { margin: 0 20px; }\n'
            '.timers {overflow: hidden; line-height: 1.5em;}\n'
            '.timers-line {display: flex; align-items: center;}\n'
            '.timers-line-pad {flex: 0 0 150px;}\n'
            '.timers-line-content {flex: 0 1 100%;}\n'
            '.timers-bar {display: block; background: #e9f3ff; color: #1774e9; text-align: center; position: relative; min-width: 1px;}\n'
            'a.timers-bar {background: #fbecca; color: #b68411;}\n'
            '.timers-bar-inner {width: fit-content; position: relative; left: 50%;}\n'
            '.timers-bar-title {margin-left: -100%; text-transform: uppercase; white-space: nowrap;}\n'
        )

        if execute:
            summary = (
                '<div class="summary-col">'
                '<b>Total:</b> {total}<br />'
                '<b>Total execute:</b> {execute}<br />'
                '<b>Total run:</b> {run}<br />'
                '</div>'

                '<div class="summary-col">'
                '<b>Ratio run/total:</b> {ratio_run_total}%<br />'
                '<b>Ratio execute/total:</b> {ratio_execute_total}%<br />'
                '<b>Ratio run/execute:</b> {ratio_run_execute}%<br />'
                '</div>'

                '<div class="summary-col">'
                '<b>Tax total - run:</b> {tax_run_total}<br />'
                '<b>Tax total - execute:</b> {tax_execute_total}<br />'
                '<b>Tax execute - run:</b> {tax_run_execute}<br />'
                '</div>'
            ).format(
                total=YammyTimeHelpers.format_time(duration),
                run=YammyTimeHelpers.format_time(run),
                execute=YammyTimeHelpers.format_time(execute),

                ratio_execute_total=round(execute * 100 / duration, 1),
                ratio_run_total=round(run * 100 / duration, 1),
                ratio_run_execute=round(run * 100 / execute, 1),

                tax_execute_total=YammyTimeHelpers.format_time(duration - execute),
                tax_run_total=YammyTimeHelpers.format_time(duration - run),
                tax_run_execute=YammyTimeHelpers.format_time(execute - run),
            )
        else:
            summary = ''

        table = [(
            '<div class="timers-line">'
            '<div class="timers-line-pad"></div>'
            '<div class="timers-line-content">'
            '<{bar_start} class="timers-bar" style="left: {start}%; width: {duration}%;">'
            '<div class="timers-bar-inner"><div class="timers-bar-title"><b>{name}</b> - {timer}</div></div>'
            '</{bar_end}>'
            '</div>'
            '<div class="timers-line-pad"></div>'
            '</div>'
        ).format(
            bar_start='a href="/task/{}"'.format(t["task_id"]) if t["task_id"] else 'div',
            bar_end='a' if t["task_id"] else 'div',
            name=t["name"],
            start=round((t["start"] - start) / duration * 100, 1),
            duration=round(t["duration"] / duration * 100, 1),
            timer=YammyTimeHelpers.format_time(t["duration"])
        ) for t in timers]

        return (
            '<style>{styles}</style>'
            '<div class="summary">{summary}</div>'
            '<div class="timers">{table}</div>'
        ).format(
            styles=styles,
            summary=summary,
            table=''.join(table),
        )

    @sdk2.report(title="Timers", label="timers")
    def report_timers(self):
        if not self.Context.timers:
            return "Timers is not ready"

        now = time.time()

        try:
            timers = YammyTimeHelpers.make_timers(now, self.Context.timers)

            if "tasks" in self.Context:
                for task_id in self.Context.tasks.values():
                    task = sdk2.Task[task_id]
                    if "timers" not in task.Context:
                        continue

                    task_log = dict()
                    for (name, value) in task.Context.timers.items():
                        if ":task" in name or ":execute" in name or ":run" in name or "wait:" in name:
                            task_log[name] = value

                    timers = timers + YammyTimeHelpers.make_timers(now, task_log, 'child:{}'.format(task_id), task_id)

            return YammyTimeHelpers.build_report(timers)
        except Exception as e:
            return e.message

    @property
    def timers(self):
        if not self.Context.timers:
            self.Context.timers = dict()

        return self.Context.timers

    @property
    def iteration(self):
        if not self.Context.iteration:
            self.Context.iteration = 0

        return self.Context.iteration

    def timer_iteration(self):
        self.Context.iteration = self.iteration + 1

    def timer(self, name):
        return Timer(self, name)

    def timer_start(self, name, thru=False):
        if not thru:
            name = "{}:{}".format(self.iteration, name)

        self.timers[name] = dict(start=time.time())

    def timer_end(self, name, thru=False):
        if not thru:
            name = "{}:{}".format(self.iteration, name)

        if name not in self.timers:
            return

        self.timers[name]['end'] = time.time()
