# -*- coding: utf-8 -*-

import json
import logging
import os
import uuid

from sandbox import sdk2
from sandbox.common.errors import ResourceNotFound
from sandbox.common.types import misc as ctm
from sandbox.common.types import resource as ctr
from sandbox.common.types import task as ctt
from sandbox.common.utils import get_task_link
from sandbox.projects.market.frontarc.MarketAutotestsGeminiUpdateArc import MarketAutotestsGeminiUpdateArc
from sandbox.projects.market.frontarc.helpers import sandbox_helpers
from sandbox.projects.market.frontarc.helpers.MetatronEnvArc import MetatronEnvArc
from sandbox.projects.market.frontarc.helpers.sandbox_helpers import rich_check_call, \
    unpack_resource, report_data, format_header, \
    prepare_browsers_semaphores, prepare_ginny_user_config
from sandbox.projects.market.frontarc.helpers.node import create_node_selector
from sandbox.projects.market.frontarc.helpers.ubuntu import create_ubuntu_selector, setup_container
from sandbox.projects.market.frontarc.helpers.startrack import ReleaseTicket, build_test_statistic
from sandbox.projects.market.frontarc.helpers.ArcDefaultParameters import ArcDefaultParameters
from sandbox.projects.market.resources import MARKET_SCREENSHOTS_PACK, MARKET_AUTOTEST_REPORT
from sandbox.projects.sandbox_ci.managers import ConfigManager
from sandbox.projects.sandbox_ci.utils import env
from sandbox.sandboxsdk import errors
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.projects.common import task_env

KILL_TIMEOUT = 3 * 60 * 60  # 3 h
SUBTASK_TIMEOUT = 3 * 60 * 60  # 3 h
DISK_SPACE = 3 * 1024  # 3 Gb


class MarketAutotestsGeminiArc(sdk2.Task):
    """
    Gemini-тесты сервисов Маркета
    """
    TOOL = 'gemini'
    HTML_REPORT_DIR = 'html_reports'
    HTML_REPORT_FILE = 'index.html'
    RESULTS_FOLDER = 'results_temp_folder'

    class Context(sdk2.Context):
        report_resource_id = None
        update_subtask_id = None

    class Parameters(sdk2.Task.Parameters):
        ubuntu_version = create_ubuntu_selector()
        node_version = create_node_selector()

        app_resource_id = sdk2.parameters.Integer(
            'Id ресурса с приложением',
            required=True
        )

        arc_params = ArcDefaultParameters()

        app_src_dir = sdk2.parameters.String("Кастомный путь корня приложения внутри репозитория")

        with sdk2.parameters.Group('Project build') as project_build_block:
            project_build_context = sdk2.parameters.String(
                'Профиль конфигурации в Genisys',
                default=''
            )

        with sdk2.parameters.Group('Тестовое окружение') as test_target:
            base_url = sdk2.parameters.String(
                'Тестируемый хост',
                description='Адрес машины, на который будут ходить тесты'
            )
            update_screen_pack = sdk2.parameters.Bool(
                'Обновить эталоны',
                description='Отметьте, если нужно обновить ресурс эталонов перед прогоном тестов',
                default=False
            )
            screen_pack_url = sdk2.parameters.String(
                'Хост стенда эталонов',
                description='При обновлении или поиске ресурса с эталонами скриншотов'
                            ' можно изменить адрес хоста, которому он соответствует'
            )
            screen_pack = sdk2.parameters.Resource(
                'Ресурс с эталонами',
                resource_type=MARKET_SCREENSHOTS_PACK,
            )

        with sdk2.parameters.Group('Startrack') as testpalm_block:
            st_issue = ReleaseTicket(
                'Релизный тикет',
                description='Пример: "MARKETVERSTKA-12345"'
            )

        with sdk2.parameters.Group('Семафор: Gemini') as semaphore_block:
            use_browser_semaphore = sdk2.parameters.Bool(
                'Использовать семафор для браузеров gemini',
                default=False,
                description='Использовать семафор для браузеров'
            )

        with sdk2.parameters.Group('Environment') as environ_block:
            environ = sdk2.parameters.Dict('Environment variables')

    class Requirements(task_env.BuildRequirements):
        dns = ctm.DnsType.DNS64
        disk_space = DISK_SPACE
        environments = [
            PipEnvironment('yandex_tracker_client', version="1.3", custom_parameters=["--upgrade-strategy only-if-needed"]),
            PipEnvironment('startrek_client', version="2.3.0", custom_parameters=["--upgrade-strategy only-if-needed"])
        ]

    @property
    def base_url(self):
        if self.Parameters.base_url:
            return self.Parameters.base_url
        else:
            return self._get_conf_param(['environ', 'gemini_root_url'])

    @property
    def need_update_pack(self):
        """
        Приоритет:
        1. Не обновляем, если ресурс уже передали в параметрах
        2. Обновляем, если так задано в параметрах джобы
        3. Обновляем, если так задано в конфигах genisys
        Не обновляем во всех остальных случаях
        :return:
        """
        if self.Parameters.screen_pack:
            return False

        if self.Parameters.update_screen_pack:
            return True

        return self._get_conf_param(['tests', 'gemini-update', 'enabled'], False)

    @property
    def screen_pack_url(self):
        if self.Parameters.screen_pack_url:
            return self.Parameters.screen_pack_url
        else:
            return self._get_conf_param(['tests', 'gemini-update', 'screen_pack_url'], self.base_url)

    # Genisys-config manager
    # requires `project_conf` property to be specified
    @property
    def config_manager(self):
        return ConfigManager(self)

    @property
    def project_conf(self):
        whole_config = self.config_manager.get_properties(section='sandbox-ci-market')

        return self.config_manager.get_project_conf(whole_config, {
            'project_name': self.project_name,
            'build_context': self.Parameters.project_build_context or None,
        })

    @property
    def project_name(self):
        parts = self.Parameters.root_path.split('/')
        return parts[-1]

    @property
    def report_description(self):
        return self.project_name

    @property
    def github_context(self):
        return self.format_github_context(self.report_description)

    @classmethod
    def format_github_context(cls, description):
        return u'{}: {}'.format(cls.TOOL.capitalize(), description)

    @staticmethod
    def find_existing_screen_pack(host):
        return sdk2.Resource["MARKET_SCREENSHOTS_PACK"].find(
            host=host,
            state=ctr.State.READY,
            attrs={"tool": "gemini"},
        ).first()

    @sdk2.header()
    def header(self):
        resource_id = self.Context.report_resource_id

        if resource_id:
            resource = self.server.resource[resource_id].read()
            data = report_data(resource)
            report = {'<h3 id="checks-reports">Report</h3>': [format_header(**data)]}

            return report

    def _call_test(self):
        rich_check_call(
            ["make", "gemini"],
            task=self, alias="test", cwd=self.app_src_path,
        )

    def _prepare_env(self):
        self.root_dir = os.path.join(str(self.path()), self.RESULTS_FOLDER)
        self.app_src_path = os.path.join(self.root_dir, self.project_name)

        if self.Parameters.app_src_dir:
            self.app_src_path = os.path.join(self.app_src_path, self.Parameters.app_src_dir)

        # export order for correct priority
        env.export(self._get_conf_environ())
        env.export(self.Parameters.environ)

        env.export({'gemini_root_url': self.base_url})

    def _prepare_screen_pack(self):
        screen_pack = self.Parameters.screen_pack
        screen_pack_url = self.screen_pack_url
        update_subtask_id = self.Context.update_subtask_id

        # get pack from parameters
        if screen_pack:
            logging.debug('Using custom screenshot pack with ID: {}'.format(str(screen_pack.id)))
        # use pack, generated by subtask
        elif update_subtask_id:
            logging.info('Getting update subtask with id: {}', str(update_subtask_id))
            update_subtask = self.find(id=update_subtask_id).first()
            screen_pack = sdk2.Resource.find(
                resource_type=MARKET_SCREENSHOTS_PACK,
                task=update_subtask
            ).first()
        # get last with specified url
        else:
            screen_pack = MarketAutotestsGeminiArc.find_existing_screen_pack(screen_pack_url)
            logging.info('Screen pack resource id: {}', str(screen_pack.id))

        return screen_pack

    def _prepare_browser_semaphore(self):
        prepare_browsers_semaphores(self)

    def _prepare_ginny_user_config(self):
        prepare_ginny_user_config(self, 'gemini')

    def _test(self):
        status = ctt.Status.SUCCESS

        try:
            logging.debug('Running {} tests on host: {}'.format(self.TOOL, self.base_url))
            self._call_test()
        except errors.SandboxSubprocessError:
            status = ctt.Status.FAILURE
            raise
        finally:
            self._make_reports(status)

    def _create_update_subtask_if_have_not_already(self, screen_pack_url):
        """
        Создаст подзадачу если эту функцию ещё не вызывали.
        Если вызывали - ничего не сделает.
        :param screen_pack_url:
        :return:
        """
        with self.memoize_stage.create_update_subtask(max_runs=1):
            _id = int(str(uuid.uuid4().int)[:4] + str(uuid.uuid4().int)[:4])
            params = {
                'id': _id,

                'task_description': self.Parameters.task_description,
                'head_branch': self.Parameters.head_branch,
                'pr_number': self.Parameters.pr_number,
                'commit_hash': self.Parameters.commit_hash,
                'root_path': self.Parameters.root_path,
                'app_src_dir': self.Parameters.app_src_dir,

                'node_version': self.Parameters.node_version,
                'kill_timeout': max(self.Parameters.kill_timeout, KILL_TIMEOUT),
                'app_resource_id': self.Parameters.app_resource_id,
                'description': self.Parameters.description,
                'project_build_context': self.Parameters.project_build_context,
                'screen_pack_url': screen_pack_url,
                'priority': {
                    'class': self.Parameters.priority.cls,
                    'subclass': self.Parameters.priority.scls,
                }
            }

            logging.info("Start SandboxCiMarketGeminiUpdate subtask with id: {id} and params: {parameters}".format(
                id=params['id'],
                parameters=params
            ))

            subtask = MarketAutotestsGeminiUpdateArc(self, **params)
            subtask.enqueue()

            self.Context.update_subtask_id = subtask.id

            raise sdk2.WaitTask(
                subtask,
                ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
                timeout=SUBTASK_TIMEOUT
            )

    def _resolve_screen_pack_url(self, default):
        if self.Parameters.screen_pack_url:
            return self.Parameters.screen_pack_url
        else:
            return self._get_conf_param(['tests', 'gemini-update', 'screen_pack_url'], default)

    def _create_report(self, **params):
        task_root_dir = str(self.path())
        target_report_dir = os.path.join(task_root_dir, self.HTML_REPORT_DIR)
        html_report_path = os.path.join(self.app_src_path, self.HTML_REPORT_DIR)

        sdk2.helpers.subprocess.check_call(['mv', html_report_path, task_root_dir])

        resource = MARKET_AUTOTEST_REPORT(self, "Gemini report", target_report_dir, **params)
        sdk2.ResourceData(resource).ready()

        self.Context.report_resource_id = resource.id

        return resource

    def _make_reports(self, status):
        logging.info('Generating Gemini report')

        resource = self._create_report(
            resource_path=self.HTML_REPORT_DIR,
            type='{}-report'.format(self.TOOL),
            status=status,
            root_path=self.HTML_REPORT_FILE
        )

        report_url = sandbox_helpers.get_resource_http_proxy_link(resource)

        self._send_startrek_report(report_url)

    def _send_startrek_report(self, report_url):
        logging.info('Export statistic')
        stat_reporter_path = '{}/stat-reporter.json'.format(self.app_src_path)

        if not self.Parameters.st_issue:
            logging.debug('Release ticket not specified'.format(stat_reporter_path))
            return

        if not os.path.isfile(stat_reporter_path):
            logging.debug('Report file {} fot found'.format(stat_reporter_path))
            return

        oauth_token = sdk2.Vault.data('robot-metatron-st-token')

        from startrek_client import Startrek

        st = Startrek(useragent='robot-metatron', token=oauth_token)
        # todo rename this shit
        issue = st.issues[self.Parameters.st_issue]
        http_report_url = '{}/{}'.format(report_url, self.HTML_REPORT_FILE)

        with open(stat_reporter_path, 'r') as f:
            html_report = build_test_statistic(
                tool=self.TOOL,
                pack=self.Parameters.project_build_context,
                data=json.loads(f.read()),
                report_url=http_report_url,
                task_url=get_task_link(self.id)
            )

            issue.comments.create(text=html_report)

            logging.debug('Statistic has been sent to {}'.format(self.Parameters.st_issue))

    def _get_conf_param(self, path, default=None):
        return self.config_manager.get_deep_value(path, default)

    def _get_conf_environ(self):
        return self.project_conf.get('environ', {})

    def _get_conf_browsers(self):
        return self.project_conf.get('gemini_browsers', {})

    def _get_browsers(self):
        return self._get_conf_browsers()

    def on_enqueue(self):
        setup_container(self)
        self._prepare_browser_semaphore()

    def _prepare_task_resources(self):
        if self.need_update_pack:
            self._create_update_subtask_if_have_not_already(self.screen_pack_url)

        # Сюда джоба попадёт на первом запуске если self.need_update_pack == False
        # И на перезапуске после выполнения подзадачи если self.need_update_pack == True
        screen_pack = self._prepare_screen_pack()

        if screen_pack:
            logging.debug('Using screenshot pack with ID: {}'.format(str(screen_pack.id)))
        else:
            raise ResourceNotFound('Screenshot pack not found. '
                                   'Please set the correct resource "screenshots-pack" or '
                                   'rebuild the screenshots pack for your host '
                                   '(by using SandboxCiMarketGeminiUpdate task)')

        _id = self.Parameters.app_resource_id
        app_resource = sdk2.Resource[_id]

        unpack_resource(self, app_resource, self.root_dir)
        unpack_resource(self, screen_pack, self.app_src_path)

    def on_execute(self):
        # NB: MetatronEnvArc clears env on exit
        # Без мемоизации, т.к. если в _prepare_task_resources создастся подзадача - нам надо чтобы он выполнился ещё раз
        # на повторном вызове on_execute после выполнения подзадачи
        with MetatronEnvArc(self, nodejs_version=self.Parameters.node_version):
            logging.debug('Prepare expected screens pack')
            self._prepare_env()
            self._prepare_task_resources()

        # NB: MetatronEnvArc clears env on exit
        with MetatronEnvArc(self, nodejs_version=self.Parameters.node_version), self.memoize_stage.execute(max_runs=1):
            logging.debug('Start task execute')

            self._prepare_env()
            self._prepare_ginny_user_config()
            self._test()
