# -*- coding: utf-8 -*-

import logging
import os

from sandbox import sdk2
from sandbox.common.types import misc as ctm
from sandbox.common.types import task as ctt
from sandbox.projects.market.frontarc.helpers.MetatronEnvArc import MetatronEnvArc
from sandbox.projects.market.frontarc.helpers.sandbox_helpers import rich_check_call, unpack_resource, format_header, report_data, prepare_ginny_user_config
from sandbox.projects.market.frontarc.helpers.node import create_node_selector
from sandbox.projects.market.frontarc.helpers.ubuntu import create_ubuntu_selector, setup_container
from sandbox.projects.market.frontarc.helpers.ArcDefaultParameters import ArcDefaultParameters
from sandbox.projects.market.resources import MARKET_SCREENSHOTS_PACK, MARKET_AUTOTEST_REPORT
from sandbox.projects.sandbox_ci.managers import ConfigManager
from sandbox.projects.sandbox_ci.utils import env
from sandbox.sandboxsdk import errors
from sandbox.projects.common import task_env

DISK_SPACE = 3 * 1024  # 3 Gb


class MarketAutotestsGeminiUpdateArc(sdk2.Task):
    """
    Обновление эталонов для Gemini-тестов сервисов Маркета
    """
    TOOL = 'gemini'
    GEMINI_ROOT_PATH = 'index.html'
    HTML_REPORT_DIR = 'html_reports'
    RESULTS_FOLDER = 'results_temp_folder'
    PACK_ARCHIVE_NAME = 'screenshots-pack.tar.gz'
    pack_path = None

    class Context(sdk2.Context):
        report_resource_id = None

    class Parameters(sdk2.Task.Parameters):
        ubuntu_version = create_ubuntu_selector()
        node_version = create_node_selector()

        app_resource_id = sdk2.parameters.Integer(
            'Id ресурса с приложением',
            required=True
        )

        arc_params = ArcDefaultParameters()

        app_src_dir = sdk2.parameters.String("Кастомный путь корня приложения внутри репозитория")

        with sdk2.parameters.Group('Project build') as project_build_block:
            project_build_context = sdk2.parameters.String(
                'Профиль конфигурации в Genisys',
                default=''
            )

        with sdk2.parameters.Group('Тестовое окружение') as test_target:
            screen_pack_url = sdk2.parameters.String(
                'Хост стенда эталонов',
                description='При обновлении или поиске ресурса с эталонами скриншотов'
                            ' можно изменить адрес хоста, которому он соответствует'
            )

    class Requirements(task_env.TinyRequirements):
        dns = ctm.DnsType.DNS64
        disk_space = DISK_SPACE
        cores = 2
        ram = 1024 * 12

    def on_enqueue(self):
        super(MarketAutotestsGeminiUpdateArc, self).on_enqueue()
        setup_container(self)

    # Genisys-config manager
    # requires `project_conf` property to be specified
    @property
    def config_manager(self):
        return ConfigManager(self)

    @property
    def project_conf(self):
        whole_config = self.config_manager.get_properties(section='sandbox-ci-market')

        return self.config_manager.get_project_conf(whole_config, {
            'project_name': self.project_name,
            'build_context': self.Parameters.project_build_context or None,
        })

    @property
    def screen_pack_url(self):
        if self.Parameters.screen_pack_url:
            return str(self.Parameters.screen_pack_url)
        else:
            return str(
                self._get_conf_param(['tests', 'gemini-update', 'screen_pack_url'])
                or
                self._get_conf_param(['environ', 'gemini_root_url'])
            )

    @property
    def project_name(self):
        parts = self.Parameters.root_path.split('/')
        return parts[-1]

    @property
    def report_description(self):
        return self.project_name

    @classmethod
    def format_github_context(cls, description):
        return u'{}: {}'.format(cls.TOOL.capitalize(), description)

    @sdk2.header()
    def header(self):
        resource_id = self.Context.report_resource_id

        if resource_id:
            resource = self.server.resource[resource_id].read()
            data = report_data(resource)
            report = {'<h3 id="checks-reports">Report</h3>': [format_header(**data)]}

            return report

    def _call_gemini_update(self):
        rich_check_call(
            ["make", "gemini_update"],
            task=self, alias="gemini_update", cwd=self.app_src_path,
        )

    def _call_gemini_pack(self):
        rich_check_call(
            ["make", "gemini_pack"],
            task=self, alias="gemini_pack", cwd=self.app_src_path,
        )
        self.pack_path = os.path.join(self.app_src_path, self.PACK_ARCHIVE_NAME)

    def _prepare_dirs(self):
        self.root_dir = os.path.join(str(self.path()), self.RESULTS_FOLDER)
        self.app_src_path = os.path.join(self.root_dir, self.project_name)

        if self.Parameters.app_src_dir:
            self.app_src_path = os.path.join(self.app_src_path, self.Parameters.app_src_dir)

    def _prepare_env(self):
        env.export(self._get_conf_environ())
        env.export({
            'gemini_root_url': self.screen_pack_url
        })

    def _prepare_ginny_user_config(self):
        prepare_ginny_user_config(self, 'gemini')

    def _prepare_app_resource(self):
        _id = self.Parameters.app_resource_id
        self.app_resource = sdk2.Resource[_id]

        logging.info('Unpacking application resource {}'.format(_id))
        unpack_resource(self, self.app_resource, self.root_dir)

    def _update(self):
        try:
            logging.info('Running {} update screenshot pack'.format(self.TOOL))
            self._call_gemini_update()
            return ctt.Status.SUCCESS
        except errors.SandboxSubprocessError:
            return ctt.Status.FAILURE

    def _create_pack(self, **kwargs):
        logging.info('Saving {} screenshot pack'.format(self.TOOL))

        if not os.path.exists(self.pack_path):
            logging.debug('Not adding resource {}: no such path'.format(self.pack_path))
            return

        attributes = dict({
            'status': ctt.Status.SUCCESS,
            'ttl': 365,
            'tool': self.TOOL,
            'type': 'screenshots-pack',
            'host': self.screen_pack_url
        },
            **kwargs
        )

        resource = MARKET_SCREENSHOTS_PACK(self, "Screenshots pack", self.pack_path, **attributes)

        sdk2.ResourceData(resource).ready()

    def _create_report(self, status):
        logging.info('Generating Gemini report')

        params = {
            'type': '{}-report'.format(self.TOOL),
            'status': status,
            'root_path': self.GEMINI_ROOT_PATH
        }

        resource = MARKET_AUTOTEST_REPORT(self, "Gemini report", self.target_report_dir, **params)
        sdk2.ResourceData(resource).ready()

        self.Context.report_resource_id = resource.id

        return resource

    def _get_conf_param(self, path):
        return self.config_manager.get_deep_value(path, None)

    def _get_conf_environ(self):
        return self.project_conf.get('environ', {})

    def _get_conf_browsers(self):
        return self.project_conf.get('gemini_browsers', {})

    def _get_browsers(self):
        return self._get_conf_browsers()

    def _move_results(self):
        task_root_dir = str(self.path())
        self.target_report_dir = os.path.join(task_root_dir, self.HTML_REPORT_DIR)
        html_report_path = os.path.join(self.app_src_path, self.HTML_REPORT_DIR)

        sdk2.helpers.subprocess.check_call(['mv', html_report_path, task_root_dir])

        target_pack_path_dir = str(self.path())
        sdk2.helpers.subprocess.check_call(['mv', self.pack_path, target_pack_path_dir])
        self.pack_path = os.path.join(target_pack_path_dir, self.PACK_ARCHIVE_NAME)

    def on_execute(self):
        # all environmental variables must be set inside MetatronEnvArc
        with MetatronEnvArc(self, nodejs_version=self.Parameters.node_version):
            logging.debug('Start task execute')
            self._prepare_dirs()
            self._prepare_app_resource()
            self._prepare_env()
            status = self._update()
            # Упаковка в тарник должна быть вызвана до _move_results
            self._call_gemini_pack()

            self._move_results()

            self._create_report(status)
            self._create_pack()
